/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['baja!', 'jquery', 'Promise', 'mobile/fieldeditors/MobileFieldEditor', 'mobile/fieldeditors/mobile/DurationSelectRelTimeEditor', 'mobile/fieldeditors/mobile/RelTimeEditor'], function (baja, $, Promise, MobileFieldEditor, DurationSelectRelTimeEditor, RelTimeEditor) {

  'use strict';

  var OverrideRelTimeEditor = function OverrideRelTimeEditor() {
    MobileFieldEditor.apply(this, arguments);
  };
  OverrideRelTimeEditor.prototype = Object.create(MobileFieldEditor.prototype);
  OverrideRelTimeEditor.prototype.constructor = OverrideRelTimeEditor;

  /**
   * Builds a composite editor for a RelTime - one editor for the duration
   * select dropdown, and one editor for the actual datebox.
   *
   * Arms a handler on the duration select dropdown. Whenever the user changes
   * the selection, the selected RelTime value will be loaded into the
   * datebox. The datebox will be set to read-enabled if Custom is selected,
   * otherwise read-only.
   * 
   * @param {JQuery} dom
   */
  OverrideRelTimeEditor.prototype.doInitialize = function (dom) {
    var that = this,
        override = that.value(),
        durationEd = new DurationSelectRelTimeEditor(baja.RelTime.make(0), override, 'duration', { parent: that }),
        relTimeEd = new RelTimeEditor(baja.RelTime.make(0), override, 'duration', { parent: that }),
        maxOverride = that.facets.get('maxOverrideDuration');

    function buildEditors(maxOverride) {
      //propagate maxOverride facet down as max facet of the relTime editor
      //and duration dropdown
      if (maxOverride !== null) {
        relTimeEd.facets = baja.Facets.make(relTimeEd.facets, {
          max: maxOverride,
          min: baja.RelTime.make(1000)
        });
        relTimeEd.maxOverride = maxOverride; //mark flag so we don't overwrite it in overrideRelTimeLoad
        // max override is set, so don't allow permanent etc.
        durationEd.facets = baja.Facets.make(durationEd.facets, {
          max: maxOverride,
          min: baja.RelTime.make(1000)
        });
      }

      return durationEd.initialize(dom).then(function () {
        return relTimeEd.initialize(dom).then(function () {
          return maxOverride && relTimeEd.load(maxOverride);
        });
      }).then(function () {
        that.$durationSelectEditor = durationEd;
        that.$relTimeEditor = relTimeEd;

        dom.on('change', 'select', function () {
          var val = $(this).val();
          if (val === '-1') {
            relTimeEd.setEnabled(true);
            relTimeEd.load(baja.RelTime.make(0));
          } else {
            relTimeEd.setEnabled(false);
            relTimeEd.load(baja.RelTime.make(Number(val)));
          }
          that.setModified(true);
        });
      });
    }

    if (maxOverride !== null) {
      return buildEditors(maxOverride);
    } else {
      return baja.Ord.make('station:|slot:/').get({ lease: 100 }).then(function (station) {
        var facets = station.get('sysInfo');
        return buildEditors(facets.get('maxOverrideDuration'));
      });
    }
  };

  /**
   * Sets the values of both the duration select dropdown and the datebox
   * to the given RelTime. If the given RelTime does not match an existing
   * duration in the dropdown, then the dropdown will be set to Custom
   * and the datebox set to read-enabled.
   * @param {baja.RelTime} relTime
   */
  OverrideRelTimeEditor.prototype.doLoad = function (relTime) {
    var that = this,
        durationEd = that.$durationSelectEditor,
        relTimeEd = that.$relTimeEditor,
        maxOverride = relTimeEd.maxOverride;

    delete relTimeEd.maxOverride;

    return durationEd.load(maxOverride || relTime).then(function () {
      return durationEd.read();
    }).then(function (selectedRelTime) {
      return Promise.join(relTimeEd.load(maxOverride || relTime), relTimeEd.setEnabled(!that.isEnabled() || selectedRelTime === -1));
    });
  };

  /**
   * Just retrieves the save data from the RelTime datebox.
   */
  OverrideRelTimeEditor.prototype.doRead = function () {
    return this.$relTimeEditor.read();
  };

  return OverrideRelTimeEditor;
});
