/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * @private
 * @module mobile/fieldeditors/mobile/FacetsEditor
 */
define(['baja!', 'jquery', 'Promise', 'mobile/fieldeditors/MobileFieldEditor', 'mobile/util/mobile/dialogs'], function (baja, $, Promise, MobileFieldEditor, dialogs) {

  'use strict';

  /**
   * @private
   * @class
   * @alias module:mobile/fieldeditors/mobile/FacetsEditor
   * @extends module:mobile/fieldeditors/MobileFieldEditor
   */

  var FacetsEditor = function FacetsEditor() {
    MobileFieldEditor.apply(this, arguments);
    this.currentEditor = null;
    this.currentKey = null;
  };
  FacetsEditor.prototype = Object.create(MobileFieldEditor.prototype);
  FacetsEditor.prototype.constructor = FacetsEditor;

  /**
   * For editing a `baja.Facets` object, creates a select dropdown containing
   * available facets on that object and a div to contain the sub-editor for
   * an individual facet.
   *
   * Arms a change listener on the select dropdown so that whenever a new
   * facet is selected, a sub-field-editor will be loaded for that facet.
   * Also undelegates the usual user value change detection - selecting a new
   * facet from the dropdown should not in itself trigger
   * `modified.fieldeditor`.
   * 
   * @param {JQuery} dom
   */
  FacetsEditor.prototype.doInitialize = function (dom) {
    var that = this,
        facetSelect = $('<select data-role="selectmenu" data-theme="a"><option value="-1">loading</option></select>'),
        subEditorDiv = $('<div class="subEditor"/>');

    facetSelect.appendTo(dom);
    subEditorDiv.appendTo(dom);

    dom.undelegate();

    facetSelect.on('change', function () {
      var key = $(this).val();
      saveFacetEditor(that, that.currentEditor).then(function () {
        loadFacetEditor(that, subEditorDiv, key);
        that.$lastKey = key;
      }, function (err) {
        dialogs.error(err);
        facetSelect.val(that.$lastKey);
      });
    });
  };

  /**
   * Populates the select dropdown with an entry for each facet on the edited
   * `baja.Facets` object. Also, initializes `this.facetValues` with the
   * initial values of each facet. The sub-field editors will set the values
   * on `this.facetValues`; this object will be assembled into a new
   * `baja.Facets` object in `read()`.
   */
  FacetsEditor.prototype.doLoad = function (value) {
    var that = this,
        facetSelect = that.jq().find('select'),
        facetValues = that.facetValues = {},
        facets = value,
        firstKey = null;

    facetSelect.empty();
    baja.iterate(facets.getKeys(), function (key) {
      var option = $('<option/>').attr('value', key).text(key);
      facetSelect.append(option);
      facetValues[key] = facets.get(key);

      firstKey = firstKey || key;
    });

    facetSelect.val(firstKey).trigger('change');
    that.$lastKey = firstKey;
  };

  /**
   * Assembles the values in `this.facetValues` into a new instance of
   * `baja.Facets`. This will likely be a combination of the original values
   * from the edited `baja.Facets` instance (those which were not edited by
   * the user) and new, user-edited values. If the current facet editor is
   * dirty, will save that editor first before retrieving the value.
   */
  FacetsEditor.prototype.doRead = function () {
    var that = this;

    return saveFacetEditor(that, that.currentEditor).then(function () {
      var newKeys = [],
          newValues = [];
      baja.iterate(that.facetValues, function (value, key) {
        newKeys.push(key);
        newValues.push(value);
      });
      return baja.Facets.make(newKeys, newValues);
    });
  };

  /**
   * Called when first loading the facets editor or when selecting a
   * different facet via the dropdown; loads a field editor so the user
   * can edit that facet.
   *
   * @param {module:mobile/fieldeditors/mobile/FacetsEditor} ed
   * @param {JQuery} subEditorDiv the div in which to load the sub-editor
   * @param {String} key the key for the edited facet
   */
  function loadFacetEditor(ed, subEditorDiv, key) {
    var dom = ed.jq(),
        editedValue = ed.facetValues[key];

    if (editedValue === undefined) {
      return;
    }

    ed.makeChildFor({
      value: editedValue,
      element: subEditorDiv.empty()
    }).then(function (editor) {
      delete ed.currentEditor;
      delete ed.currentKey;
      dom.trigger('create');
      dom.trigger('updatelayout');
      ed.currentEditor = editor;
      ed.currentKey = key;
    });
  }

  /**
   * Retrieves the value from the current facet editor and saves that value
   * into `this.facetValues`. Called when saving the overall `BFacets`
   * editor, or when selecting a different facet from the dropdown.
   *
   * @param {module:mobile/fieldeditors/mobile/FacetsEditor} facetsEditor the editor
   * for the enclosing `BFacets` object
   * @param {module:mobile/fieldeditors/BaseFieldEditor} subFacetEditor the editor
   * for the currently editor sub-facet
   * @returns {Promise} promise to be resolved after current facet is applied
   */
  function saveFacetEditor(facetsEditor, subFacetEditor) {
    if (subFacetEditor && subFacetEditor.isModified()) {
      return subFacetEditor.read().then(function (readValue) {
        facetsEditor.facetValues[facetsEditor.currentKey] = readValue;
      });
    }
    return Promise.resolve();
  }

  return FacetsEditor;
});
