/**
 * @copyright 2021 Tridium, Inc. All Rights Reserved.
 */

define(['baja!', 'Promise', 'underscore', 'nmodule/webEditors/rc/baja/PasswordStrength', 'nmodule/webEditors/rc/servlets/password'], function (baja, Promise, _, PasswordStrength, password) {
  'use strict';

  /**
   * API Status: **Private**
   * @exports nmodule/platCrypto/rc/rpc/certManagement
   */
  var exports = {};
  var once = _.once;
  var byteUtil = require('bajaScript/baja/obj/byteUtil');

  /**
   * @private
   * @returns {string} type spec of the RPC class
   */
  exports.$getRpcTypeSpec = function () {
    return "platCrypto:CertificateManagementRpc";
  };
  var doRpc = function doRpc(methodName, args) {
    return baja.rpc({
      typeSpec: exports.$getRpcTypeSpec(),
      method: methodName,
      args: args
    })["catch"](function (err) {
      baja.error(err);
      throw err;
    });
  };
  var getDefaultRsaKeySize = once(function () {
    return doRpc('getDefaultRsaKeySize', []);
  });
  var getSupportedRsaKeySizes = once(function () {
    return doRpc('getSupportedRsaKeySizes', []);
  });
  var getDefaultECKeySpec = once(function () {
    return doRpc('getDefaultECKeySpec', []);
  });
  var getSupportedECKeySpecs = once(function () {
    return doRpc('getSupportedECKeySpecs', []);
  });
  var getSupportedKeyAlgorithms = once(function () {
    return doRpc('getSupportedKeyAlgorithms', []);
  });

  /**
   * @param {String} cryptoStoreId
   * @returns {Promise.<object>}
   */
  exports.getCertStoreData = function (cryptoStoreId) {
    return doRpc('getCertStoreData', [cryptoStoreId]);
  };

  /**
   * @param {String} cryptoStoreId
   * @param {Array.<String>} ids
   * @param {String} [password='']
   * @returns {Promise}
   */
  exports.deleteCertStoreData = function (cryptoStoreId, ids) {
    var password = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : '';
    if (cryptoStoreId === 'userExemptionStore') {
      var batch = new baja.comm.Batch();
      var deleteProms = ids.map(function (host) {
        return doRpc('deleteExemption', [host]);
      });
      return batch.commit(Promise.all(deleteProms));
    } else {
      var _batch = new baja.comm.Batch();
      var _deleteProms = ids.map(function (alias) {
        return doRpc('deleteCertificate', [cryptoStoreId, alias, password]);
      });
      return _batch.commit(Promise.all(_deleteProms));
    }
  };

  /**
   * Delete all certificates from the user key store
   * @returns {Promise}
   */
  exports.resetUserKeyStoreData = function () {
    return doRpc('resetKeyStore');
  };

  /**
   * Generate a new certificate inside the User Key Store.
   *
   * @returns {Promise}
   */
  exports.generateCertificate = function (_ref) {
    var alias = _ref.alias,
      subjectDn = _ref.subjectDn,
      certUsage = _ref.certUsage,
      keySize = _ref.keySize,
      keyType = _ref.keyType,
      spec = _ref.spec,
      notBefore = _ref.notBefore,
      notAfter = _ref.notAfter,
      password = _ref.password,
      extensions = _ref.extensions,
      keyStorePassword = _ref.keyStorePassword;
    var submitJson = {
      builder: {
        alias: alias,
        subjectDn: subjectDn,
        certUsage: certUsage,
        keyStorePassword: keyStorePassword,
        notBefore: notBefore.getMillis(),
        notAfter: notAfter.getMillis()
      },
      keyGenerator: {
        generatorType: keyType,
        parameters: {
          keySize: keySize,
          spec: spec
        }
      }
    };
    if (baja.hasType(password, 'baja:Password')) {
      // Doing this should be ok because cert manager RPCs accepts only secure connections
      submitJson['password'] = password.encodeToString();
    }

    // Filter out the extensions that have no value/data
    if (extensions) {
      var validExtensions = extensions.filter(function (_ref2) {
        var value = _ref2.value;
        return !!value;
      });
      if (validExtensions.length > 0) {
        submitJson.builder['extensions'] = extensions.filter(function (_ref3) {
          var value = _ref3.value;
          return !!value;
        });
      }
    }

    // We might want to log warning/info and throw a generic "RPC failed" error maybe?
    return doRpc("generateCertificate", [JSON.stringify(submitJson)]);
  };

  /**
   * @param {Number} requestId the id for the asynchronous RPC request
   * @returns {Promise.<Number>} the status for this request.
   */
  exports.checkRequestStatus = function (requestId) {
    return doRpc("checkRequestStatus", [requestId]);
  };

  /**
   * @param {Array.<String>} hosts the hosts that need to be approved
   * @param {boolean} [approveChanged=false] set to true if the user is ready to approved changes to previously approved hosts
   * @returns {Promise}
   */
  exports.approveExemptions = function (hosts) {
    var approveChanged = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;
    var batch = new baja.comm.Batch();
    var approveProms = hosts.map(function (host) {
      return doRpc('setExemption', [host, true, approveChanged]);
    });
    return batch.commit(Promise.all(approveProms));
  };

  /**
   * @param {Array.<String>} hosts the hosts that need to be unapproved
   * @returns {Promise}
   */
  exports.unapproveExemptions = function (hosts) {
    var batch = new baja.comm.Batch();
    var unapproveProms = hosts.map(function (host) {
      return doRpc('setExemption', [host, false, false]);
    });
    return batch.commit(Promise.all(unapproveProms));
  };

  /**
   * @param {string} pem
   * @param {string} storeId
   * @returns {Promise.<module:nmodule/platCrypto/rc/rpc/certManagement~ParsedPem>}
   */
  function parsePemFile(pem, storeId) {
    return doRpc('parsePemFile', [pem, storeId]);
  }

  /**
   * Decode and validate a PEM file.
   * @see module:nmodule/platCrypto/rc/wb/cert/commands/CertImportCommand
   * @private
   * @param {String} fileName the name of the file that was loaded in, used to set the alias
   * @param {baja.Simple} x509cert the `baja:X509Certificate` containing the certificate(s) being imported
   * @param {string} keyStore the name of the key store the certificates are being imported to
   * @returns {Promise.<{ storeName: string, parsedData: module:nmodule/platCrypto/rc/rpc/certManagement~ParsedPem }>}
   */
  exports.decodePemFile = function (fileName, x509cert, keyStore) {
    var defaultAlias = fileName ? fileName.split('.')[0] : "";
    return parsePemFile(x509cert.encodeToString(), keyStore).then(function (parsedData) {
      parsedData.certificates.forEach(function (cert) {
        var certificate = cert.certificate;
        var hasMatchCertificateInUserKeyStore = !!(parsedData.suggestedKeyStoreAlias && keyStore === "userKeyStore");
        var alias = cert.suggestedAlias || parsedData.suggestedKeyStoreAlias || defaultAlias || certificate.subject;
        cert.alias = alias.toLowerCase();
        cert.selected = true;
        cert.promptForAlias = !hasMatchCertificateInUserKeyStore;
        cert.hasMatchCertificateInUserKeyStore = hasMatchCertificateInUserKeyStore;
      });
      return {
        storeName: keyStore,
        parsedData: parsedData
      };
    });
  };

  /**
   * @param {String} storeType the name of the certificate store
   * @param {module:nmodule/platCrypto/rc/rpc/certManagement~CertificateImportRequest} importRequest the
   * updated certificate data that was retrieved from the parsePemFile RPC
   * @returns {Promise}
   */
  exports.importCertStoreData = function (storeType) {
    var importRequest = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    var useGlobalPassword = importRequest.useGlobalPassword,
      _importRequest$encryp = importRequest.encryptPassword,
      encryptPassword = _importRequest$encryp === void 0 ? "" : _importRequest$encryp,
      _importRequest$keySto = importRequest.keyStorePassword,
      keyStorePassword = _importRequest$keySto === void 0 ? "" : _importRequest$keySto,
      certificates = importRequest.certificates;
    var _ref4 = importRequest.privateKey || {},
      privateKey = _ref4.privateKey,
      pemPrivateKeyPasswordNeeded = _ref4.pemPrivateKeyPasswordNeeded,
      _ref4$password = _ref4.password,
      password = _ref4$password === void 0 ? "" : _ref4$password;
    var params = {
      storeType: storeType,
      passwordToDecryptKey: password,
      keyStorePassword: keyStorePassword,
      privateKey: {
        privateKey: privateKey,
        pemPrivateKeyPasswordNeeded: pemPrivateKeyPasswordNeeded
      },
      //remove certificates that were not selected
      certificates: certificates.filter(function (certificate) {
        return certificate.selected;
      })
    };
    if (useGlobalPassword) {
      params.useGlobalPassword = true;
    } else {
      params.passwordToEncryptKey = encryptPassword;
    }
    return doRpc('importCertificate', [params]);
  };

  /**
   * Call the RPC that exports a certificate from a key store
   * @param {Object} exportParams an object that holds the parameters for the exportCertificate RPC call
   * @returns {Promise.<String>} A String containing the encoded private key/public certificate as specified by the options. The String will be empty if no export is possible and no exception is thrown.
   */
  exports.exportCertificate = function (exportParams) {
    var alias = exportParams.alias,
      exportFormat = exportParams.exportFormat,
      exportCert = exportParams.exportCert,
      exportPrivateKey = exportParams.exportPrivateKey,
      encryptPrivateKey = exportParams.encryptPrivateKey,
      encryptPassword = exportParams.encryptPassword,
      decryptPassword = exportParams.decryptPassword,
      decryptPasswordReqd = exportParams.decryptPasswordReqd,
      reusePassword = exportParams.reusePassword,
      storeType = exportParams.storeType;
    if (decryptPasswordReqd && reusePassword) {
      encryptPassword = decryptPassword;
    }
    if (!decryptPasswordReqd) {
      decryptPassword = baja.$('baja:Password');
    }
    return doRpc('exportCertificate', [storeType, alias, exportCert, exportPrivateKey, encryptPrivateKey, decryptPassword.valueOf(), encryptPassword.valueOf(), exportFormat]).then(function (contents) {
      if (exportFormat === 'PKCS7') {
        contents = byteUtil.base64StringToByteArray(contents);
      }
      return contents;
    });
  };

  /**
   * Returns the default RSA key size
   *
   * @returns {Promise.<Number>} resolves default RSA key size
   */
  exports.getDefaultRsaKeySize = function () {
    return getDefaultRsaKeySize();
  };

  /**
   * Returns the Supported RSA Key sizes
   *
   * @returns {Promise.<Array>} resolves supported RSA key sizes
   */
  exports.getSupportedRsaKeySizes = function () {
    return getSupportedRsaKeySizes();
  };

  /**
   * Returns the default EC KeySpec
   *
   * @returns {Promise.<String>}
   */
  exports.getDefaultECKeySpec = function () {
    return getDefaultECKeySpec();
  };

  /**
   * Returns the Supported EC KeySpecs
   *
   * @returns {Promise.<Array>}
   */
  exports.getSupportedECKeySpecs = function () {
    return getSupportedECKeySpecs();
  };

  /**
   * Returns the Supported KeyAlgorithms
   *
   * @returns {Promise.<Array>}
   */
  exports.getSupportedKeyAlgorithms = function () {
    return getSupportedKeyAlgorithms();
  };

  /**
   * Initiates a request to generate a CSR file for the supplied alias
   *
   * @param {String} alias
   * @param {String} password
   * @returns {Promise.<String>} resolves a string encoding of the generated CSR
   */
  exports.generateCSR = function (alias, password) {
    return doRpc('generateCSR', [alias, password]);
  };

  /**
   * Checks if the certificate (with supplied alias) requires a password.
   * This check will be used, say, in "generateCSR" to check if a password
   * challenge is required.
   *
   * @param {String} alias
   * @returns {Promise.<boolean>} resolves true if a password is required
   */
  exports.isPasswordRequired = function (alias) {
    return doRpc('isPasswordRequired', [alias]);
  };

  /**
   * Calls the RPC that checks to see if the supplied password is valid for the supplied alias
   *
   * @param {String} alias
   * @param {String} password
   * @returns {Promise.<boolean>} returns true if the password is valid for the supplied alias, and false if it is not
   */
  exports.isKeyStorePasswordValid = function (alias, password) {
    return doRpc('isKeyStorePasswordValid', [alias, password]);
  };

  /**
   * Calls the RPC that checks to see if the supplied password is valid for the supplied parsed PEM data
   *
   * @param {module:nmodule/platCrypto/rc/rpc/certManagement~ParsedPem} parsedData the updated certificate data that was returned from the parsePemFile RPC
   * @param {String} password
   * @returns {Promise.<boolean>} returns true if the password is valid for the supplied PEM, and false if it is not
   */
  exports.isParsedPemPasswordValid = function (parsedData, password) {
    return doRpc('isParsedPemPasswordValid', [parsedData, password]);
  };

  /**
   * Gets the password strength setting from the station
   * @returns {Promise.<baja.Complex>} a 'baja:PasswordStrength'
   */
  exports.passwordStrength = function () {
    return password.getDefaultPasswordStrength().then(function (returnVal) {
      return baja.bson.decodeAsync(JSON.parse(returnVal)).then(function (passwordStrength) {
        if (passwordStrength instanceof PasswordStrength) {
          return passwordStrength;
        }
        return baja.$('baja:PasswordStrength');
      });
    });
  };

  /**
   * Gets the valid supportted export formats for exporting a certificate
   * @returns {Promise.<object>}
   */
  exports.getSupportedExportFormats = function () {
    return doRpc('getSupportedExportFormats');
  };
  return exports;
});
