/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Jeremy Narron
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/fe/BacnetUnsignedEditor
 */
define(['baja!', 'bajaux/Properties', 'nmodule/bacnet/rc/baja/datatypes/BacnetUnsigned', 'nmodule/webEditors/rc/fe/CompositeEditor', 'nmodule/webEditors/rc/fe/config/CompositeBuilder'], function (baja, Properties, BacnetUnsigned, CompositeEditor, CompositeBuilder) {
  'use strict';

  /**
   * Editor for handling `bacnet:BacnetUnsigned` values.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/CompositeEditor
   * @alias module:nmodule/bacnet/rc/fe/BacnetUnsignedEditor
   */
  var BacnetUnsignedEditor = function BacnetUnsignedEditor() {
    CompositeEditor.apply(this, arguments);
  };
  BacnetUnsignedEditor.prototype = Object.create(CompositeEditor.prototype);
  BacnetUnsignedEditor.prototype.constructor = BacnetUnsignedEditor;

  /**
   * Configures the composite builder for the loaded `bacnet:BacnetUnsigned`
   * instance.
   *
   * @returns {module:nmodule/webEditors/rc/fe/config/CompositeBuilder}
   */
  BacnetUnsignedEditor.prototype.makeBuilder = function () {
    var that = this,
      builder = new CompositeBuilder();
    builder.getDomFor = function (key) {
      return that.jq().children();
    };
    builder.getKeys = function () {
      return ['value'];
    };
    builder.getValueFor = function (key) {
      var bacnetUnsigned = this.getDataSource(),
        _long = bacnetUnsigned.valueOf(),
        range = that.properties().getValue('range');
      return range ? baja.DynamicEnum.make({
        ordinal: _long,
        range: range
      }) : baja.Long.make(_long);
    };
    builder.getConfigFor = function (key) {
      var parentProps = that.properties();
      if (parentProps.getValue('range')) {
        return {
          properties: parentProps,
          formFactor: 'mini'
        };
      }
      var radix = (parentProps.getValue('radix') || 10).valueOf(),
        min = (parentProps.getValue('min') || BacnetUnsigned.MIN_VALUE).valueOf(),
        max = (parentProps.getValue('max') || BacnetUnsigned.MAX_VALUE).valueOf(),
        unit = (parentProps.getValue('units') || '').toString(),
        label = '';
      if (unit) {
        label += unit;
      }
      if (radix !== 10) {
        if (label.length > 0) {
          label += ' ';
        }
        label += 'Base ' + radix;
      }
      if (min !== BacnetUnsigned.MIN_VALUE || max !== BacnetUnsigned.MAX_VALUE) {
        if (label.length > 0) {
          label += ' ';
        }
        label += '[' + min.toString(radix) + ' - ' + max.toString(radix) + ']';
      }
      return {
        properties: Properties.extend({
          postlabel: label
        }, parentProps),
        formFactor: 'mini'
      };
    };
    return builder;
  };

  /**
   * Create the DOM structure that contains the editor.
   *
   * @param {JQuery} dom
   * @returns {Promise}
   */
  BacnetUnsignedEditor.prototype.doInitialize = function (dom) {
    dom.html('<div/>');
    return CompositeEditor.prototype.doInitialize.apply(this, arguments);
  };

  /**
   * Read a new `bacnet:BacnetUnsigned` instance from the editor in the dom.
   *
   * @returns {Promise.<module:nmodule/bacnet/rc/baja/datatypes/BacnetUnsigned>}
   */
  BacnetUnsignedEditor.prototype.doRead = function () {
    return this.getChildEditors()[0].read().then(function (value) {
      var _long2 = value;
      if (baja.hasType(value, 'baja:Enum')) {
        _long2 = value.getOrdinal();
      }
      return baja.$('bacnet:BacnetUnsigned', _long2);
    });
  };
  return BacnetUnsignedEditor;
});
