/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/baja/enums/BacnetObjectType
 */
define(['baja!', 'underscore', 'nmodule/bacnet/rc/util/BacnetConst'], function (baja, _, BacnetConst) {
  'use strict';

  var FrozenEnum = baja.FrozenEnum;
  var objectIdFacetsArray = [],
    supportedObjectIdOrdinals = _.range(0, 25).concat([40, 46]),
    INVALID_OBJECT_TYPE = 'INVALID',
    MAX_RESERVED = 127,
    MAX_ID = 1023;
  var BacnetObjectType = function BacnetObjectType() {
    FrozenEnum.apply(this, arguments);
  };
  BacnetObjectType.prototype = Object.create(FrozenEnum.prototype);
  BacnetObjectType.prototype.constructor = BacnetObjectType;
  BacnetObjectType.make = function () {
    var d = BacnetObjectType.DEFAULT;
    return d.make.apply(d, arguments);
  };
  BacnetObjectType.DEFAULT = new BacnetObjectType();
  BacnetObjectType.getObjectIdFacets = function (ordinal) {
    if (!baja.hasType(ordinal, 'baja:Number')) {
      throw new Error('ordinal required');
    }
    if (!_.contains(supportedObjectIdOrdinals, ordinal)) {
      return null;
    }
    var f = objectIdFacetsArray[ordinal];
    if (!f) {
      var tag = BacnetObjectType.DEFAULT.getRange().getTag(ordinal);
      f = baja.Facets.make({
        range: baja.EnumRange.make({
          ordinals: [ordinal],
          tags: [tag]
        })
      });
      objectIdFacetsArray[ordinal] = f;
    }
    return f;
  };

  /**
   * Is the given id a proprietary extension?
   *
   * @static
   * @param {number} id - the object type number
   * @returns {boolean} true if this is a proprietary extension.
   */
  BacnetObjectType.isProprietary = function (id) {
    return id > MAX_RESERVED && id <= MAX_ID;
  };

  /**
   * Is the given id an ASHRAE extension?
   *
   * @static
   * @param {number} id - the object typenumber
   * @returns {boolean} true if the id is an ASHRAE extension.
   */
  BacnetObjectType.isAshrae = function (id) {
    var range = BacnetObjectType.DEFAULT.getRange(),
      ordinals = range.getOrdinals();
    return id > ordinals[ordinals.length - 1] && id <= MAX_RESERVED;
  };

  /**
   * Create a string tag for the given id. Note that this might not be one of the ordials
   * for the defined frozen range.
   *
   * @param {Number} id the integer enumeration value.
   *
   * @returns {String} the tag for the ordinal, if it is known, or construct one using standard
   * prefixes.
   */
  BacnetObjectType.tagForId = function (id) {
    if (BacnetObjectType.DEFAULT.getRange().isOrdinal(id)) {
      return BacnetObjectType.DEFAULT.make(id).getTag();
    } else if (BacnetObjectType.isAshrae(id)) {
      return BacnetConst.ASHRAE_PREFIX + id;
    } else if (BacnetObjectType.isProprietary(id)) {
      return BacnetConst.PROPRIETARY_PREFIX + id;
    } else {
      return INVALID_OBJECT_TYPE + ':' + id;
    }
  };

  /**
   * Get the id for the given tag. Note that this might not be one of the tags for the defined
   * frozen range.
   *
   * @param {String} tag - the tag string
   *
   * @returns {Number} return the ordinal for the tag, if it is known, or generate one if the tag
   * uses standard prefixes.
   */
  BacnetObjectType.idForTag = function (tag) {
    if (BacnetObjectType.DEFAULT.getRange().isTag(tag)) {
      return BacnetObjectType.DEFAULT.make(tag).getOrdinal();
    } else if (tag.indexOf(BacnetConst.ASHRAE_PREFIX) === 0) {
      return parseInt(tag.substring(BacnetConst.ASHRAE_PREFIX.length), 10);
    } else if (tag.indexOf(BacnetConst.PROPRIETARY_PREFIX) === 0) {
      return parseInt(tag.substring(BacnetConst.PROPRIETARY_PREFIX.length), 10);
    }
    return -1;
  };
  return BacnetObjectType;
});
