/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Jeremy Narron
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/baja/datatypes/BacnetUnsigned
 */
define(['baja!'], function (baja) {
  'use strict';

  var Simple = baja.Simple;

  /**
   * BajaScript representation of a `bacnet:BacnetUnsigned` value.
   *
   * @class
   * @extends {baja.Simple}
   * @alias {module:nmodule/bacnet/rc/baja/datatypes/BacnetUnsigned}
   * @param {Number} long
   */
  var BacnetUnsigned = function BacnetUnsigned(_long) {
    if (!baja.hasType(_long, 'baja:Number') || isNaN(_long.valueOf())) {
      throw new Error('Number required');
    }
    if (_long > BacnetUnsigned.MAX_VALUE) {
      throw new Error('Value must be <= ' + BacnetUnsigned.MAX_VALUE);
    }
    Simple.apply(this, arguments);
    this.$long = Math.floor(_long);
  };
  BacnetUnsigned.prototype = Object.create(Simple.prototype);
  BacnetUnsigned.prototype.constructor = BacnetUnsigned;

  /**
   * Factory method to create a new `BacnetUnsigned` instance from a long.
   *
   * @static
   * @param {Number} long
   * @returns {module:nmodule/bacnet/rc/baja/datatypes/BacnetUnsigned}
   */
  BacnetUnsigned.make = function (_long2) {
    switch (Math.floor(_long2)) {
      case 0:
        return BacnetUnsigned.DEFAULT;
      case 1:
        return BacnetUnsigned.ONE;
      case 2:
        return BacnetUnsigned.TWO;
      default:
        return new BacnetUnsigned(_long2);
    }
  };

  /**
   * Instance method to create a new `BacnetUnsigned` instance from a long.
   *
   * @param {Number} long
   * @returns {module:nmodule/bacnet/rc/baja/datatypes/BacnetUnsigned}
   */
  BacnetUnsigned.prototype.make = function (_long3) {
    return BacnetUnsigned.make(_long3);
  };

  /**
   * Encode this instance to a string using the value of the backing long.
   *
   * @returns {String} String encoding of long
   */
  BacnetUnsigned.prototype.encodeToString = function () {
    return String(this.$long);
  };

  /**
   * Decode a string to a new `BacnetUnsigned` instance.
   *
   * @param {String} str
   * @returns {module:nmodule/bacnet/rc/baja/datatypes/BacnetUnsigned}
   */
  BacnetUnsigned.prototype.decodeFromString = function (str) {
    return new BacnetUnsigned(parseInt(str));
  };

  /**
   * Return the value of the long backing this instance.
   *
   * @returns {Number}
   */
  BacnetUnsigned.prototype.valueOf = function () {
    return this.$long;
  };

  /**
   * Return the string representation of this instance.
   *
   * @returns {String}
   */
  BacnetUnsigned.prototype.toString = function (cx) {
    var range = cx && cx.range;
    if (range) {
      return baja.SlotPath.unescape(range.getTag(this.$long));
    }
    return String(this.valueOf());
  };

  /**
   * `DEFAULT` is initialized to 0.
   *
   * @type {module:nmodule/bacnet/rc/baja/datatypes/BacnetUnsigned}
   */
  BacnetUnsigned.DEFAULT = new BacnetUnsigned(0);
  BacnetUnsigned.ONE = new BacnetUnsigned(1);
  BacnetUnsigned.TWO = new BacnetUnsigned(2);
  BacnetUnsigned.MIN_VALUE = 0;
  BacnetUnsigned.MAX_VALUE = 4294967295;
  return BacnetUnsigned;
});
