/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Vikram Nagulan
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/baja/datatypes/BacnetDate
 */
define(['baja!', 'underscore', 'lex!bacnet'], function (baja, _, lexs) {
  'use strict';

  var objUtil = require('bajaScript/baja/obj/objUtil');
  var Simple = baja.Simple,
    UNSPECIFIED = -1,
    EVEN_MONTHS = 14,
    ODD_MONTHS = 13,
    ODD_DAY_OF_MONTH = 33,
    EVEN_DAY_OF_MONTH = 34,
    LAST_DAY_OF_MONTH = 32,
    bacLex = lexs[0],
    DEFAULT_ENCODING = "****-**-**-***",
    cacheDecode = objUtil.cacheDecode,
    cacheEncode = objUtil.cacheEncode;
  var WEEKDAYS = ["Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun"];

  /**
   * BajaScript representation of a `bacnet:BacnetDate` value.
   *
   * @class
   * @extends {baja.Simple}
   * @alias {module:nmodule/bacnet/rc/baja/datatypes/BacnetDate}
   */
  var BacnetDate = function BacnetDate() {
    Simple.apply(this, arguments);
  };
  BacnetDate.prototype = Object.create(Simple.prototype);
  BacnetDate.prototype.constructor = BacnetDate;
  BacnetDate.make = function (year, month, dayOfMonth, dayOfWeek) {
    year = parseInt(year, 10);
    month = parseInt(month, 10);
    dayOfMonth = parseInt(dayOfMonth, 10);
    dayOfWeek = parseInt(dayOfWeek, 10);
    if (!inRange(year, 1900, 2154)) {
      throw new Error(bacLex.get("BacnetDate.invalid.year", year));
    }
    if (!inRange(month, 1, EVEN_MONTHS)) {
      throw new Error(bacLex.get("BacnetDate.invalid.month", month));
    }
    if (!inRange(dayOfMonth, 1, EVEN_DAY_OF_MONTH)) {
      throw new Error(bacLex.get("BacnetDate.invalid.dom", dayOfMonth));
    }
    if (!inRange(dayOfWeek, 1, 7)) {
      throw new Error(bacLex.get("BacnetDate.invalid.dow"));
    }
    var bd = new BacnetDate();
    bd.$year = year;
    bd.$month = month;
    bd.$dayOfMonth = dayOfMonth;
    bd.$dayOfWeek = dayOfWeek;
    return bd;
  };

  /**
   * @see .make
   * @returns {module:nmodule/bacnet/rc/baja/datatypes/BacnetDate}
   */
  BacnetDate.prototype.make = function () {
    return BacnetDate.make.apply(BacnetDate, arguments);
  };
  BacnetDate.prototype.getYear = function () {
    if (this.$year === UNSPECIFIED) {
      return UNSPECIFIED;
    }
    return this.$year;
  };
  BacnetDate.prototype.getMonth = function () {
    return this.$month;
  };
  BacnetDate.prototype.getDayOfMonth = function () {
    return this.$dayOfMonth;
  };
  BacnetDate.prototype.getDayOfWeek = function () {
    return this.$dayOfWeek;
  };
  BacnetDate.prototype.equals = function (obj) {
    if (baja.hasType(obj, 'bacnet:BacnetDate')) {
      return this.getYear() === obj.getYear() && this.getMonth() === obj.getMonth() && this.getDayOfMonth() === obj.getDayOfMonth() && this.getDayOfWeek() === obj.getDayOfWeek();
    }
    return false;
  };

  /**
   * Convenience method to make a BacnetDate from a Javascript date object
   * @param {Date} jsDate - Javascript date object
   * @returns {BacnetDate}
   */
  BacnetDate.makeFromJsDate = function (jsDate) {
    if (jsDate) {
      var y = jsDate.getFullYear();
      var m = jsDate.getMonth() + 1;
      var dOm = jsDate.getDate();
      var dOw = jsDate.getDay() || 7; //Bacnet week days is 1 thru 7 (Mon thru Sun)

      return BacnetDate.make(y, m, dOm, dOw);
    }
    return BacnetDate.DEFAULT;
  };

  /**
   * Get a valid js date or return today
   * @returns {Date}
   */
  BacnetDate.prototype.getJsDate = function () {
    if (this.$isAnyUnspecified() || this.$isSpecialDay() || this.$isSpecialMonth()) {
      return new Date();
    }
    return new Date(this.getYear(), this.getMonth() - 1, this.getDayOfMonth());
  };

  /**
   * Decode an `BacnetDate` from a string.
   *
   * @function
   * @param {String} str
   * @returns {module:nmodule/bacnet/rc/baja/datatypes/BacnetDate}
   */
  BacnetDate.prototype.decodeFromString = cacheDecode(function (str) {
    var REG_UNSPECIFIED = /\*/;
    if (!str || str === DEFAULT_ENCODING) {
      return BacnetDate.DEFAULT;
    }
    var bdAr = str.split("-");
    if (bdAr.length !== 4) {
      throw new Error("Incorrect Encoding: " + str);
    }

    //Year part
    var y = bdAr[0];
    if (y.match(REG_UNSPECIFIED)) {
      y = UNSPECIFIED;
    } else {
      y = parseInt(bdAr[0], 10);
    }
    //Month part
    var m = bdAr[1].toUpperCase();
    if (m.match(REG_UNSPECIFIED)) {
      m = UNSPECIFIED;
    } else if (m === "EV") {
      m = EVEN_MONTHS;
    } else if (m === "OD") {
      m = ODD_MONTHS;
    } else {
      m = parseInt(bdAr[1], 10);
    }
    //Day of month part
    var dOm = bdAr[2].toUpperCase();
    if (dOm.match(REG_UNSPECIFIED)) {
      dOm = UNSPECIFIED;
    } else if (dOm === "LD") {
      dOm = LAST_DAY_OF_MONTH;
    } else if (dOm === "OD") {
      dOm = ODD_DAY_OF_MONTH;
    } else if (dOm === "ED") {
      dOm = EVEN_DAY_OF_MONTH;
    } else {
      dOm = parseInt(bdAr[2], 10);
    }
    //day of week part
    var dOw = bdAr[3];
    if (dOw.match(REG_UNSPECIFIED)) {
      dOw = UNSPECIFIED;
    } else {
      dOw = BacnetDate.weekday(bdAr[3]);
    }
    return BacnetDate.make(y, m, dOm, dOw);
  });

  /**
   * Encode this `BacnetDate` to a string.
   *
   * @function
   * @returns {String}
   */
  BacnetDate.prototype.encodeToString = cacheEncode(function () {
    var encStr = "";
    var year = this.getYear(),
      month = this.getMonth(),
      dOfM = this.getDayOfMonth(),
      dOfW = this.getDayOfWeek();
    //Year part
    encStr += year === UNSPECIFIED ? "****-" : year + "-";

    //Month part
    if (month === UNSPECIFIED) {
      encStr += "**-";
    } else if (month === EVEN_MONTHS) {
      encStr += "EV-";
    } else if (month === ODD_MONTHS) {
      encStr += "OD-";
    } else {
      encStr += month < 10 ? "0" + month + "-" : month + "-";
    }

    //DayofMonth part
    if (dOfM === UNSPECIFIED) {
      encStr += "**-";
    } else if (dOfM === LAST_DAY_OF_MONTH) {
      encStr += "LD-";
    } else if (dOfM === ODD_DAY_OF_MONTH) {
      encStr += "OD-";
    } else if (dOfM === EVEN_DAY_OF_MONTH) {
      encStr += "ED-";
    } else {
      encStr += dOfM < 0 ? "0" + dOfM + "-" : dOfM + "-";
    }

    //DayofWeek part
    if (dOfW === UNSPECIFIED) {
      encStr += "***";
    } else {
      encStr += WEEKDAYS[dOfW - 1];
    }
    return encStr;
  });
  BacnetDate.DEFAULT = BacnetDate.make(UNSPECIFIED, UNSPECIFIED, UNSPECIFIED, UNSPECIFIED);

  /**
   * Return the weekday given a short name of the day
   * @param {string} dayShort - The day short version Mon, Tue...
   * @return {Number} The weekday in number ranges from 1 (Mon) to 7 (Sun)
   */
  BacnetDate.weekday = function (dayShort) {
    return _.indexOf(WEEKDAYS, dayShort) + 1;
  };

  /****************** Private functions *********************/

  /**
   * Is any field unspecified?
   * @return true if any field is unspecified.
   */
  BacnetDate.prototype.$isAnyUnspecified = function () {
    return this.$year === UNSPECIFIED || this.$month === UNSPECIFIED || this.$dayOfMonth === UNSPECIFIED || this.$dayOfWeek === UNSPECIFIED;
  };

  /**
   * Is month odd or even (13 or 14)
   * @return {boolean} true if month is odd or even.
   */
  BacnetDate.prototype.$isSpecialMonth = function () {
    var mon = this.$month;
    return mon === ODD_MONTHS || mon === EVEN_MONTHS;
  };

  /**
   * Is day of the month is last, odd or even (32, 33 or 34)
   * @return {boolean} true if day of month  is the last, odd or even.
   */
  BacnetDate.prototype.$isSpecialDay = function () {
    var dom = this.$dayOfMonth;
    return dom === ODD_DAY_OF_MONTH || dom === EVEN_DAY_OF_MONTH || dom === LAST_DAY_OF_MONTH;
  };

  /**********Helper Methods***********************/

  /**
   * Helper - Validates if the input value is in range
   * -1 is considered 'in' this range
   * @param {number} value - The input value
   * @param {number} min - The min value allowed
   * @param {number} max - The max value allowed
   * @returns {boolean} True if the value is in range
   */
  function inRange(value, min, max) {
    return !isNaN(value) && (value >= min && value <= max || value === UNSPECIFIED);
  }
  return BacnetDate;
});
