/**
 * @copyright 2015, Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/bql/rc/fe/filter/FacetsFilterEditor
 */
define(['baja!', 'baja!bql:FacetsFilter,bql:IBqlFilter', 'lex!workbench', 'jquery', 'Promise', 'bajaux/events', 'bajaux/Widget', 'bajaux/commands/Command', 'bajaux/util/CommandButton', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/bql/rc/fe/filter/FacetsFilterConfigEditor', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/feDialogs', 'bajaux/mixin/batchSaveMixin', 'nmodule/webEditors/rc/fe/baja/util/DepthSubscriber', 'nmodule/bql/rc/fe/filter/saveUtil', 'css!nmodule/bql/rc/bql'], function (baja, types, lexs, $, Promise, events, Widget, Command, CommandButton, BaseEditor, FacetsFilterConfigEditor, fe, feDialogs, batchSaveMixin, DepthSubscriber, saveUtil) {
  "use strict";

  var ENABLE_EVENT = events.ENABLE_EVENT,
    DISABLE_EVENT = events.DISABLE_EVENT,
    MODIFY_EVENT = events.MODIFY_EVENT,
    INITIALIZE_EVENT = events.INITIALIZE_EVENT,
    LOAD_EVENT = events.LOAD_EVENT,
    READONLY_EVENT = events.READONLY_EVENT,
    WRITABLE_EVENT = events.WRITABLE_EVENT,
    DESTROY_EVENT = events.DESTROY_EVENT,
    FILTER_ENTRIES_CLASS = 'filterEntries',
    FILTER_ENTRIES_SELECTOR = '.' + FILTER_ENTRIES_CLASS,
    EDIT_CMD_CLASS = "editFacets",
    EDIT_CMD_SELECTOR = "." + EDIT_CMD_CLASS,
    save = saveUtil.save;

  /**
   * FacetsFilterEditor is a field editor for a bql:FacetsFilter instance
   * @class
   * @alias module:nmodule/bql/rc/fe/filter/FacetsFilterEditor
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   */
  var FacetsFilterEditor = function FacetsFilterEditor() {
    BaseEditor.apply(this, arguments);
    batchSaveMixin(this);
  };
  FacetsFilterEditor.prototype = Object.create(BaseEditor.prototype);
  FacetsFilterEditor.prototype.constructor = FacetsFilterEditor;
  function initFilterEntriesEditor(editor) {
    var dom = editor.jq(),
      filterEntriesDom = $("<div/>").addClass(FILTER_ENTRIES_CLASS).appendTo(dom);
    return fe.buildFor({
      dom: filterEntriesDom,
      value: "",
      readonly: true,
      enabled: false,
      formFactor: 'mini'
    });
  }
  function initEditFacetsCommand(editor) {
    var dom = editor.jq(),
      buttonDom = $('<button type="button"></button>').addClass("ux-btn-tb").addClass(EDIT_CMD_CLASS).appendTo(dom);
    return fe.buildFor({
      dom: buttonDom,
      value: new Command({
        icon: "module://icons/x16/edit.png",
        func: function func() {
          return editor.read().then(function (filter) {
            return feDialogs.showFor({
              title: lexs[0].get("fieldSheet.facets.editFacets", "Edit Facets"),
              value: filter,
              type: FacetsFilterConfigEditor
            }).then(function (filter) {
              if (filter) {
                var entries = filter.getFilterEntries().newCopy(true),
                  keys = entries.getSlots().toArray().join(", "),
                  filterEntriesEd = editor.$getFilterEntriesEditor();
                editor.setModified(true);
                editor.$filterEntries = entries;
                return filterEntriesEd.load(keys || "No facets").then(function () {
                  filterEntriesEd.jq().find('input').attr('size', filterEntriesEd.value().length);
                });
              }
            });
          });
        }
      }),
      type: CommandButton,
      formFactor: 'mini'
    });
  }

  /**
   * Returns a reference to the StringEditor for listing filter entry facet keys
   * @returns {StringEditor}
   * @private
   * @see module:nmodule/webEditors/rc/fe/baja/StringEditor
   */
  FacetsFilterEditor.prototype.$getFilterEntriesEditor = function () {
    return Widget["in"](this.jq().find(FILTER_ENTRIES_SELECTOR));
  };

  /**
   * Returns a reference to the CommandButton for editing filter entry facets
   * @returns {CommandButton}
   * @private
   * @see bajaux/util/CommandButton
   */
  FacetsFilterEditor.prototype.$getEditFacetsCmdBtn = function () {
    return Widget["in"](this.jq().find(EDIT_CMD_SELECTOR));
  };

  /**
   * Initializes the editor with an editor for facets keys and
   * a command for editing facets
   * @param dom
   * @returns {Promise}
   */
  FacetsFilterEditor.prototype.doInitialize = function (dom) {
    var that = this,
      selectors = [FILTER_ENTRIES_SELECTOR, EDIT_CMD_SELECTOR].join(",");

    //prevent child editors from bubbling up events
    dom.addClass("FacetsFilterEditor").on([ENABLE_EVENT, DISABLE_EVENT, READONLY_EVENT, WRITABLE_EVENT, INITIALIZE_EVENT, LOAD_EVENT, DESTROY_EVENT].join(' '), selectors, false).on(MODIFY_EVENT, selectors, function () {
      that.setModified(true);
      return false;
    });
    return Promise.all([initFilterEntriesEditor(this), initEditFacetsCommand(this)]);
  };

  /**
   * Loads a bql:FacetsFilter instance
   * @param filter instance of bql:FacetsFilter
   * @param params
   * @returns {Promise}
   */
  FacetsFilterEditor.prototype.doLoad = function (filter, params) {
    if (!baja.hasType(filter, 'bql:FacetsFilter')) {
      return Promise.reject(new Error('editor must be loaded with bql:FacetsFilter'));
    }
    var that = this;
    this.$subscriber = new DepthSubscriber(2);
    return Promise.resolve(filter.isMounted() && this.$subscriber.subscribe(filter)).then(function () {
      var filterEntries = filter.getFilterEntries(),
        cursor = filterEntries.getSlots().is('bql:IBqlFilter'),
        encoding = cursor.toArray().join(', '),
        filterEntriesEd = that.$getFilterEntriesEditor();
      that.$filterEntries = filterEntries;
      return filterEntriesEd.load(encoding === "" ? "No facets" : encoding, params).then(function () {
        filterEntriesEd.jq().find('input').attr('size', filterEntriesEd.value().length);
      });
    });
  };

  /**
   * Returns a new bql:FacetsFilter initialized with the current list of
   * filter entries
   * @returns {baja.Component} A new `bql:FacetsFilter` instance
   */
  FacetsFilterEditor.prototype.doRead = function () {
    return baja.$('bql:FacetsFilter', {
      'filterEntries': this.$filterEntries.newCopy(true)
    });
  };

  /**
   * Saves the filter entries returned by the call to read()
   * @param filter
   * @param params
   * @returns {*}
   */
  FacetsFilterEditor.prototype.doSave = function (filter, params) {
    return save(this, filter, params);
  };

  /**
   * Sets the readonly state of the FacetsEditor
   * @param readonly
   * @returns {Promise}
   */
  FacetsFilterEditor.prototype.doReadonly = function (readonly) {
    return this.getChildWidgets().setAllReadonly(readonly);
  };

  /**
   * Sets the enabled state of the FacetsEditor
   * @param enabled
   * @returns {Promise}
   */
  FacetsFilterEditor.prototype.doEnabled = function (enabled) {
    return Promise.all([this.$getFilterEntriesEditor().setEnabled(false), this.$getEditFacetsCmdBtn().setEnabled(enabled)]);
  };

  /**
   * Destroys all child editors
   * @param enabled
   * @returns {Promise}
   */
  FacetsFilterEditor.prototype.doDestroy = function () {
    this.jq().removeClass("FacetsFilterEditor");
    if (this.$subscriber) {
      this.$subscriber.unsubscribe(this.value());
    }
    return this.getChildWidgets().destroyAll();
  };
  return FacetsFilterEditor;
});
