/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/console/commands/NotesCommand
 */
define(['baja!', 'lex!alarm', 'Promise', 'underscore', 'dialogs', 'bajaux/commands/Command', 'nmodule/alarm/rc/fe/AlarmNotesEditor', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/alarm/rc/util/alarmUtils', 'nmodule/alarm/rc/console/commands/alarmConsoleCommandMixIn'], function (baja, lexs, Promise, _, dialogs, Command, AlarmNotesEditor, feDialogs, alarmUtils, alarmConsoleCommandMixIn) {
  'use strict';

  var lex = lexs[0],
    ERROR_MSG_PREFIX = lex.get('alarm.console.acknowledge.insufficient.permissions');
  function getSelectedSubjects(table) {
    return _.invoke(table.getSelectedRows(), 'getSubject');
  }
  function getSelectedIndices(table) {
    var indices = [],
      selection = table.$getSelection(),
      rows = table.getModel().getRows();
    for (var i = 0; i < rows.length; i++) {
      if (selection.isSelected(i)) {
        indices.push(i);
      }
    }
    return indices;
  }
  function makeNoteParams(view, params) {
    var noteParams = {},
      table = view.$getAlarmTable(),
      singleSource = view.isSingleSourceView(),
      details = params && params.details,
      subject = details && params.subject;
    if (subject) {
      noteParams.ids = [subject.uuid];
    } else if (singleSource) {
      noteParams.ids = _.pluck(getSelectedSubjects(table), 'uuid');
    } else {
      noteParams.srcs = _.pluck(getSelectedSubjects(table), 'source');
    }
    return noteParams;
  }
  function getCurrentNotes(view, params) {
    var table = view.$getAlarmTable(),
      singleSource = view.isSingleSourceView(),
      details = params && params.details,
      subject = details && params.subject,
      notes,
      subjects;
    if (subject) {
      notes = subject.alarmData.notes || '';
    } else {
      subjects = getSelectedSubjects(table);
      if (subjects.length) {
        subject = subjects[0];
        if (subjects.length > 1 || !singleSource && subject.ackedCount + subject.unackedCount + subject.ackPendingCount > 1) {
          notes = lex.get('alarm.multipleAlarmNotes');
        } else {
          notes = subject.alarmData.notes || '';
        }
      }
    }
    return notes;
  }
  function reportError(failureList) {
    if (_.isArray(failureList) && failureList.length) {
      var errorMsg = _.escape(ERROR_MSG_PREFIX) + "<br/><br/>" + failureList.map(_.escape).join("<br/>");
      dialogs.showOk({
        content: errorMsg
      });
    }
    return failureList;
  }
  function addNotes(view, params) {
    var viewModel = view.$getViewModel(),
      noteParams = makeNoteParams(view, params),
      currentNotes = getCurrentNotes(view, params),
      showDialog = !(_.isEmpty(noteParams.srcs) && _.isEmpty(noteParams.ids)),
      table = view.$getAlarmTable(),
      selection = table.$getSelection(),
      selectedIndices = getSelectedIndices(table);
    if (!showDialog) {
      return Promise.resolve(false);
    }
    return NotesCommand.showNotesEditor(table.getSelectedRows(), currentNotes).then(function (notes) {
      if (notes && (!params || params.callThrough)) {
        noteParams.notes = notes;
        return dialogs.showLoading(0, viewModel.addNoteToAlarms(noteParams).then(reportError).then(function () {
          //restore selected elements
          _(selectedIndices).each(function (idx) {
            selection.select(idx);
          });
        }));
      } else {
        return notes;
      }
    });
  }

  /**
   * Adds notes to the selected alarms.
   *
   * @constructor
   * @param {module:nmodule/alarm/rc/console/AlarmConsole} view The alarm console view.
   * @param {Object} params
   * @extends {module:bajaux/commands/Command}
   * @mixes {module:nmodule/alarm/rc/console/commands/alarmConsoleCommandMixIn}
   */
  var NotesCommand = function NotesCommand(view, params) {
    Command.apply(this, [{
      module: 'alarm',
      lex: 'commands.notes',
      func: function func(noteParams) {
        return addNotes(view, _.extend(params || {
          callThrough: true
        }, noteParams));
      }
    }]);
    alarmConsoleCommandMixIn(this);
    this.$view = view;
    this.$showInMenu = true;
    this.$showInCommandBar = true;
  };
  NotesCommand.prototype = Object.create(Command.prototype);
  NotesCommand.prototype.constructor = NotesCommand;
  NotesCommand.prototype.selectionChanged = function () {
    this.setEnabled(!this.$view.isSelectionEmpty());
  };

  /**
   * Show the Notes Editor in dialog. This will also make the field editor readonly if the user does
   * not have operatorWrite on all Alarm Classes.
   *
   * @param {Array.<module:nmodule/webEditors/rc/wb/table/model/Row|baja.Struct>} rows An Array of Table Model Rows or
   * an Array of `baja.Struct` records which are each a `alarm:AlarmRecord`.
   * @param {String} currentNotes
   * @return {Promise.<String>}
   * @since Niagara 4.12
   */
  NotesCommand.showNotesEditor = function (rows, currentNotes) {
    return alarmUtils.hasOperatorWrite(rows).then(function (hasRequiredPermissions) {
      return feDialogs.showFor({
        value: currentNotes,
        formFactor: 'mini',
        title: lex.get('command.addNotes.label'),
        type: AlarmNotesEditor,
        readonly: !hasRequiredPermissions,
        buttons: [hasRequiredPermissions && 'ok', 'cancel']
      });
    });
  };
  return NotesCommand;
});
