/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/console/commands/FilterCommand
 */
define(['lex!alarm', 'Promise', 'underscore', 'bajaux/commands/Command', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/alarm/rc/console/commands/alarmConsoleCommandMixIn'], function (lexs, Promise, _, Command, feDialogs, alarmConsoleCommandMixIn) {
  'use strict';

  var lex = lexs[0],
    TITLE = lex.get('command.filter.label');

  /**
   * Let's the user configure what alarms should be shown in a pop up dialog box.
   *
   * @constructor
   * @param {module:nmodule/alarm/rc/console/AlarmConsole} view The alarm console view.
   * @extends {module:bajaux/commands/Command}
   * @mixes {module:nmodule/alarm/rc/console/commands/alarmConsoleCommandMixIn}
   */
  var FilterCommand = function FilterCommand(view) {
    var that = this;
    Command.apply(that, [{
      module: 'alarm',
      lex: 'commands.filter',
      func: function func() {
        return filter(that, view);
      }
    }]);
    alarmConsoleCommandMixIn(that);
    that.$showInCommandBar = true;
  };
  FilterCommand.prototype = Object.create(Command.prototype);
  FilterCommand.prototype.constructor = FilterCommand;
  function filter(cmd, view) {
    var viewModel = view.$getViewModel(),
      support = viewModel.$getSupport(),
      filterSet = viewModel.$getFilterSet(),
      isFilterSetEmpty = filterSet.getSlots().properties().getSize() === 0;
    return Promise.resolve(isFilterSetEmpty && support.loadDefaultFilterSet(filterSet)).then(function () {
      return feDialogs.showFor({
        title: TITLE,
        value: filterSet,
        formFactor: 'mini'
      }).then(function (filterSet) {
        if (!filterSet) {
          return;
        }
        filterSet = FilterCommand.$convertFilterSet(support, filterSet);
        cmd.$updateFilterIcon(filterSet);
        return viewModel.filterChanged(filterSet).then(function () {
          return filterSet;
        });
      });
    });
  }

  /**
   * Updates the command's icon based on the state of the filter set.
   *
   * @private
   * @inner
   * @since Niagara 4.6
   *
   * @param  filterSet The new filter set.
   */
  FilterCommand.prototype.$updateFilterIcon = function (filterSet) {
    var comps = filterSet.getSlots().toValueArray(),
      isActive = _.any(comps, function (filterEntry) {
        return filterEntry.getActive();
      });
    this.setIcon(lex.get('commands.filter' + (isActive ? 'ed' : '') + '.icon'));
  };

  /**
   * Checks the type of a filter set. If the type doesn't match then
   * create a new filter set and copy all over all dynamic properties.
   *
   * @private
   * @internal
   * @since Niagara 4.6
   *
   * @param  {module:nmodule/alarm/rc/console/baja/BajaAlarmConsoleSupport} support
   * The alarm console support object.
   * @param  filterSet The filter set to convert.
   * @returns The converted filter set.
   */
  FilterCommand.$convertFilterSet = function (support, filterSet) {
    var newFilterSet = support.getDefaultFilterSet();
    if (!filterSet.getType().is(newFilterSet.getType())) {
      filterSet.getSlots().properties().dynamic().each(function (prop) {
        newFilterSet.add({
          slot: prop.getName(),
          value: filterSet.get(prop).newCopy(/*exact*/true),
          flags: filterSet.getFlags(prop),
          facets: filterSet.getFacets(prop)
        });
      });
      filterSet = newFilterSet;
    }
    return filterSet;
  };
  return FilterCommand;
});
