/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 * @author Jeremy Narron
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/console/baja/BajaDataChannel
 */
define(['baja!', 'lex!alarm', 'underscore', 'Promise', 'nmodule/js/rc/tinyevents/tinyevents', 'nmodule/webEditors/rc/servlets/userData', 'nmodule/alarm/rc/console/audio/AlarmAudioHandler', 'nmodule/alarm/rc/console/audio/AlarmAudioQueue'], function (baja, lexs, _, Promise, tinyevents, userData, AlarmAudioHandler, AlarmAudioQueue) {
  'use strict';

  var lex = lexs[0],
    RECIPIENT_UPDATE_EVENT = 'recipientUpdate',
    SINGLE_SOURCE_UPDATE_EVENT = 'singleSourceUpdate',
    OPTIONS_CHANGED_EVENT = 'options';
  function space() {
    return baja.Ord.make('alarm:').get({
      lease: true
    });
  }
  function multiSourceSummaryHandler(channel, reg, update) {
    var rows = _.map(reg.getSources(), function (r) {
        return r;
      }),
      sourceCount = rows.length,
      alarmCount = reg.getAlarmCount(),
      db = channel.$db;
    return space().then(function (space) {
      return space.getDisplayNamesMap();
    }).then(function (map) {
      channel.emit(RECIPIENT_UPDATE_EVENT, {
        records: _.map(rows, _.partial(translateRow, map, db)),
        sourceCount: sourceCount,
        alarmCount: alarmCount
      }, !!update);
    });
  }
  function singleSourceSummaryHandler(channel, summary) {
    var db = channel.$db,
      records = _.map(summary.records, function (record) {
        return baja.bson.decodeValue(record);
      });
    return space().then(function (space) {
      return space.getDisplayNamesMap();
    }).then(function (map) {
      return {
        records: _.map(records, _.partial(translateRecord, map, db)),
        sourceCount: 1,
        alarmCount: summary.alarmCount,
        pageCount: summary.pageCount
      };
    });
  }
  function translateRow(map, db, row) {
    return _.extend(translateRecord(map, db, row.record), {
      ackedCount: row.acked,
      unackedCount: row.unacked,
      ackPendingCount: row.ackPending
    });
  }
  function createDateString(millis, timeZone) {
    var tf = baja.TimeFormat;
    return millis && baja.AbsTime.make({
      jsDate: new Date(millis)
    }).toDateTimeStringSync({
      timezone: timeZone,
      show: tf.SHOW_DATE + tf.SHOW_TIME + tf.SHOW_SECONDS + tf.SHOW_ZONE
    }) || lex.get('alarm.console.null');
  }
  function translateRecord(map, db, rec) {
    var ordList = rec.getSource(),
      source = ordList.size() ? ordList.encodeToString() : 'null',
      alarmData = _.mapObject(rec.getAlarmData().toObject(), function (val, key) {
        switch (key) {
          case 'TimeZone':
            return db.getTimeZone(val.getId());
          default:
            return val instanceof baja.Simple ? val.encodeToString() : val;
        }
      }),
      _alarmData = _.mapObject(rec.getAlarmData().toObject(), function (val) {
        var tz;
        if (val.getType().is("baja:AbsTime")) {
          tz = rec.getAlarmData().get("TimeZone");
          if (tz) {
            tz = db.getTimeZone(tz.getId());
          }
          return createDateString(val.getMillis(), tz);
        } else {
          return String(val);
        }
      });
    return {
      timestamp: rec.getTimestamp().getMillis(),
      uuid: rec.getUuid().encodeToString(),
      sourceState: rec.getSourceState().getDisplayTag(),
      _sourceState: rec.getSourceState().getTag(),
      ackState: rec.getAckState().getDisplayTag(),
      _ackState: rec.getAckState().getTag(),
      ackRequired: rec.getAckRequired(),
      source: source,
      alarmClass: map[rec.getAlarmClass()] || map.defaultAlarmClass,
      // 'alarmClass' is the display name for the AlarmClass, used for display in the console
      _alarmClass: rec.getAlarmClass(),
      // '_alarmClass' is the name for the AlarmClass, used as key for permissions map
      priority: rec.getPriority(),
      normalTime: rec.getNormalTime().getMillis(),
      ackTime: rec.getAckTime().getMillis(),
      user: rec.getUser(),
      alarmData: alarmData,
      _alarmData: _alarmData,
      alarmTransition: rec.getAlarmTransition().getDisplayTag(),
      _alarmTransition: rec.getAlarmTransition().getTag(),
      lastUpdate: rec.getLastUpdate().getMillis()
    };
  }
  function register(channel, params) {
    var handler = _.partial(multiSourceSummaryHandler, channel),
      filterSet = params && params.filterSet,
      timeRange = params && params.timeRange;
    return Promise.resolve(channel.$recipient && channel.$recipient.registerMultiSourceSummaryQuery(handler, filterSet, timeRange)).then(function (reg) {
      channel.$registration = reg;
    });
  }
  function initOptions(channel) {
    return userData.getJson('alarmConsole').then(function (options) {
      channel.$alarmConsoleOptions = options || channel.getDefaultAlarmConsoleOptions();
      channel.emit(OPTIONS_CHANGED_EVENT, channel.$alarmConsoleOptions);
    });
  }
  function initTimeZoneDb(channel) {
    return baja.TimeZoneDatabase.get().then(function (db) {
      channel.$db = db;
    });
  }

  /**
   * BajaDataChannel handles the underlying communication details with the
   * remote alarm recipient.
   * The channel is responsible for subscribing and responding to alarm
   * notifications from the recipient and for sending client commands for
   * various alarm management operation to the remote alarm database.
   *
   * @class
   * @alias module:nmodule/alarm/rc/console/baja/BajaDataChannel
   * @param {baja.Component} recipient - component representing the target alarm recipient
   */
  var BajaDataChannel = function BajaDataChannel(recipient) {
    this.$recipient = recipient;
    tinyevents(this);
  };

  ///////////////////////////////////////////////////////////////////////////////
  //Lifecycle
  ///////////////////////////////////////////////////////////////////////////////

  /**
   * Subscribes to the target recipient and gets user specific settings.
   *
   * @return {Promise} Promise resolved with initialized data channel
   */
  BajaDataChannel.prototype.init = function (params) {
    var that = this;
    return Promise.all([initTimeZoneDb(this), initOptions(this)]).then(function () {
      return register(that, params);
    });
  };

  /**
   * Unsubscribes from the target recipient.
   *
   * @return {Promise}
   */
  BajaDataChannel.prototype.destroy = function () {
    this.$db = null;
    this.removeAllListeners();
    return Promise.resolve(this.$registration && this.$registration.cancel());
  };

  ///////////////////////////////////////////////////////////////////////////////
  //Alarm Console Options support
  ///////////////////////////////////////////////////////////////////////////////

  /**
   * Returns the default alarm console options.
   *
   * @return {Object}
   */
  BajaDataChannel.prototype.getDefaultAlarmConsoleOptions = function () {
    return {
      soundOff: false,
      soundContinuous: false,
      multiSourceView: {
        sortByColumn: {},
        alarmDataColumns: ['msgText'],
        hiddenColumns: ['uuid', 'ackRequired', 'normalTime', 'ackTime', 'user', 'alarmData', 'alarmTransition', 'lastUpdate'],
        notesRequiredOnAck: false,
        soundDelay: AlarmAudioQueue.DEFAULT_ALARM_DELAY,
        soundFile: AlarmAudioHandler.DEFAULT_ALARM_SOUND,
        disableRowHighlight: false
      },
      singleSourceView: {
        sortByColumn: {},
        alarmDataColumns: ['msgText'],
        hiddenColumns: ['uuid', 'ackRequired', 'normalTime', 'ackTime', 'user', 'alarmData', 'alarmTransition', 'lastUpdate'],
        notesRequiredOnAck: false,
        pageSize: 25
      }
    };
  };

  /**
   * Returns the current alarm console options.
   *
   * @return {Object}
   */
  BajaDataChannel.prototype.getAlarmConsoleOptions = function () {
    return this.$alarmConsoleOptions;
  };

  /**
   * Updates the current alarm console options with a key/value pair. This
   * function updates the options on the client and also persists them to the
   * server.
   *
   * @param {String} key
   * @param {*} value
   * @param {Boolean} [singleSource] If true set the options for the single source
   * view rather than the multi source view. If this is undefined, the value will
   * be a generic alarm console option.
   * @return {Promise}
   */
  BajaDataChannel.prototype.setAlarmConsoleOption = function (key, value, singleSource) {
    var that = this;
    if (singleSource !== undefined) {
      if (singleSource) {
        that.$alarmConsoleOptions.singleSourceView[key] = value;
      } else {
        that.$alarmConsoleOptions.multiSourceView[key] = value;
      }
    } else {
      that.$alarmConsoleOptions[key] = value;
    }
    return userData.put('alarmConsole', JSON.stringify(that.$alarmConsoleOptions)).then(function () {
      that.emit(OPTIONS_CHANGED_EVENT, that.$alarmConsoleOptions);
    });
  };

  /**
   * Updates the current alarm console options of a particular view with a
   * group of key/value pairs. This function updates the options on the client
   * and also persists them to the server.
   *
   * @param {Object} options
   * @param {Boolean} singleSource if true set the options for the single source
   * view rather than the multi source view
   * @return {Promise}
   */
  BajaDataChannel.prototype.setAlarmConsoleOptions = function (options, singleSource) {
    var that = this;
    if (singleSource) {
      _.extendOwn(that.$alarmConsoleOptions.singleSourceView, options);
    } else {
      _.extendOwn(that.$alarmConsoleOptions.multiSourceView, options);
    }
    return userData.put('alarmConsole', JSON.stringify(this.$alarmConsoleOptions)).then(function () {
      that.emit(OPTIONS_CHANGED_EVENT, that.$alarmConsoleOptions);
    });
  };

  /**
   * @param {Boolean} singleSource if true get the options for the single source
   * view rather than the multi source view
   * @return {Promise}
   */
  BajaDataChannel.prototype.resetAlarmTableSettings = function (singleSource) {
    var that = this,
      defaults;
    if (singleSource) {
      defaults = this.getDefaultAlarmConsoleOptions().singleSourceView;
      this.$alarmConsoleOptions.singleSourceView.sortByColumn = defaults.sortByColumn;
      this.$alarmConsoleOptions.singleSourceView.alarmDataColumns = defaults.alarmDataColumns;
      this.$alarmConsoleOptions.singleSourceView.hiddenColumns = defaults.hiddenColumns;
    } else {
      defaults = this.getDefaultAlarmConsoleOptions().multiSourceView;
      this.$alarmConsoleOptions.multiSourceView.sortByColumn = defaults.sortByColumn;
      this.$alarmConsoleOptions.multiSourceView.alarmDataColumns = defaults.alarmDataColumns;
      this.$alarmConsoleOptions.multiSourceView.hiddenColumns = defaults.hiddenColumns;
    }
    return userData.put('alarmConsole', JSON.stringify(this.$alarmConsoleOptions)).then(function () {
      that.emit(OPTIONS_CHANGED_EVENT, that.$alarmConsoleOptions);
    });
  };

  ///////////////////////////////////////////////////////////////////////////////
  //Utilities
  ///////////////////////////////////////////////////////////////////////////////

  /**
   * Gets an array of common user defined alarm data keys for an alarm record.
   * Note that this array contains only the public static final String fields
   * in alarm:AlarmRecord.
   *
   * @return {Promise.<Array.<String>>} Promise resolved with a String array of alarm data fields
   */
  BajaDataChannel.prototype.getAlarmFields = function () {
    return space().then(function (space) {
      return space.getAlarmFields();
    });
  };

  /**
   * Loads the alarm summary from the data channel for the specified params.
   *
   * @param {Object} params - the object literal containing the method's arguments
   * @return {Promise} Promise resolved with an object literal containing the summary details
   */
  BajaDataChannel.prototype.loadAlarmSummary = function (params) {
    var that = this,
      restart = params && params.restart,
      isSingleSourceSummary = params && params.source && params.source !== 'null';
    return Promise.resolve(restart && that.$registration && that.$registration.cancel()).then(function () {
      if (restart || !that.$registration) {
        return register(that, params);
      }
    }).then(function () {
      if (isSingleSourceSummary) {
        return that.getSingleSourceSummary(params).then(function (summary) {
          that.emit(SINGLE_SOURCE_UPDATE_EVENT, summary);
        });
      } else {
        // If we haven't just restarted then fire an update event. This will
        // refresh the table.
        if (!restart) {
          return multiSourceSummaryHandler(that, that.$registration, /*update*/false);
        }
      }
    });
  };

  /**
   * Acknowledges alarms by source or uuid with optional notes.
   * @see {@link module:nmodule/alarm/rc/util/alarmUtils.ackAlarms}
   * for details regarding the expected parameter format.
   *
   * @param {Object} params - the object literal containing the method's arguments
   * @return {Promise.<Array.<String>>} Promise resolved with a String array of alarm classes for which
   * alarms failed to be acknowledged
   */
  BajaDataChannel.prototype.ackAlarms = function (params) {
    return space().then(function (space) {
      return space.ackAlarms(params);
    });
  };

  /**
   * Force clears alarms by source or uuid.
   * @see {@link module:nmodule/alarm/rc/util/alarmUtils.forceClearAlarms}
   * for details regarding the expected parameter format.
   *
   * @param {Object} params - the object literal containing the method's arguments
   * @return {Promise.<Array.<String>>} Promise resolved with a String array of alarm classes for which
   * alarms failed to be force cleared
   */
  BajaDataChannel.prototype.forceClearAlarms = function (params) {
    return space().then(function (space) {
      return space.forceClear(params);
    });
  };

  /**
   * Adds notes to alarms by source or uuid.
   * @see {@link module:nmodule/alarm/rc/util/alarmUtils.addNoteToAlarms}
   * for details regarding the expected parameter format.
   *
   * @param {Object} params - the object literal containing the method's arguments
   * @return {Promise.<Array.<String>>} Promise resolved with a String array of alarm classes for which
   * alarms failed to be force cleared
   */
  BajaDataChannel.prototype.addNoteToAlarms = function (params) {
    return space().then(function (space) {
      return space.addNoteToAlarms(params);
    });
  };

  /**
   * Gets the alarm summary for a single source for the specified params.
   *
   * @see {@link module:baja/alarm/AlarmSpace} for details regarding the expected parameter format.
   * @param {Object} params - the object literal containing the method's arguments
   * @return {Promise.<Object>} Promise resolved with an object literal containing the summary details
   */
  BajaDataChannel.prototype.getSingleSourceSummary = function (params) {
    params = params || {};
    params.source = params.source || 'null';
    params.ord = params.ord || this.$recipient && this.$recipient.getOrdInSession().encodeToString() || 'null';
    return space().then(function (space) {
      return space.getSingleSourceSummary(params);
    }).then(_.partial(singleSourceSummaryHandler, this));
  };

  /**
   * Gets the notes for an alarm record.
   *
   * @see {@link module:baja/alarm/AlarmSpace} for details regarding the expected parameter format.
   * @param {Object} params - the object literal containing the method's arguments
   * @return {Promise.<String>} Promise resolved with a notes String
   */
  BajaDataChannel.prototype.getNotes = function (params) {
    return space().then(function (space) {
      return space.getNotes(params);
    }).then(function (obj) {
      return obj && obj.notes || '';
    });
  };

  /**
   * Return a formatted date string.
   *
   * @param  {Number} millis Number of milliseconds since epoch relative to UTC.
   * @param  {baja.TimeZone} [timeZone] Optional time zone used for formatting.
   * @returns {String} The formatted string.
   */
  BajaDataChannel.createDateString = createDateString;

  ///////////////////////////////////////////////////////////////////////////////
  //Private
  ///////////////////////////////////////////////////////////////////////////////

  BajaDataChannel.prototype.$getRecipient = function () {
    return this.$recipient;
  };
  BajaDataChannel.prototype.$getRegistration = function () {
    return this.$registration;
  };
  BajaDataChannel.prototype.$getTimeZoneDb = function () {
    return this.$db;
  };
  return BajaDataChannel;
});
