/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/console/audio/AlarmAudioHandler
 */
define(["underscore"], function (_) {
  'use strict';

  var DEFAULT_ALARM_SOUND = "module://alarm/com/tridium/alarm/ui/sounds/smallBeep.wav";

  /**
   * Manages alarm sounds for the alarm console.
   *
   * @class
   * @alias module:nmodule/alarm/rc/console/audio/AlarmAudioHandler
   * @param {module:nmodule/alarm/rc/console/audio/AlarmAudioQueue} The alarm audio queue.
   */
  var AlarmAudioHandler = function AlarmAudioHandler(queue) {
    this.$sources = {};
    this.$queue = queue;
    this.$defaultSound = DEFAULT_ALARM_SOUND;
  };

  /**
   *  The default alarm sound to play.
   *
   * @type {String}
   */
  AlarmAudioHandler.DEFAULT_ALARM_SOUND = DEFAULT_ALARM_SOUND;

  /**
   * An unordered map of sound -> counts. Each increment
   * is a registered source that's interested in playing a sound.
   *
   * @internal
   * @private
   *
   * @type {Object}
   */
  AlarmAudioHandler.SOURCE_COUNTS = {};

  /**
   * Set the default alarm sound.
   *
   * @param {String} defaultSound the default alarm sound.
   */
  AlarmAudioHandler.prototype.setDefaultSound = function (defaultSound) {
    this.$defaultSound = defaultSound;
  };

  /**
   * Destroys the alarm sound player. After this is called no more sounds will
   * be played.
   */
  AlarmAudioHandler.prototype.destroy = function () {
    // This will silence any alarms associated with this sound play instance.
    this.update({
      records: []
    });
    this.$stopped = true;
  };

  /**
   * Returns true if the alarm in question is eligible to make a sound.
   *
   * @private
   *
   * @param  {Object}  record The alarm record to test.
   * @returns {Boolean} Returns true if eligible to make a sound.
   */
  AlarmAudioHandler.isAudibleAlarm = function (record) {
    return record._sourceState !== "normal" && record._ackState !== "acked";
  };

  /**
   * Returns an alarm sound for the alarm record.
   *
   * @private
   *
   * @param  {Object} record The alarm record to test for a sound.
   * @returns {String} The alarm sound to be used.
   */
  AlarmAudioHandler.prototype.toAlarmSound = function (record) {
    return record.alarmData.soundFile ? record.alarmData.soundFile : this.$defaultSound;
  };

  /**
   * Update the alarm sound player with some new alarm information.
   *
   * @param  {Object} summary An object that contains an alarm records array.
   */
  AlarmAudioHandler.prototype.update = function (summary) {
    // If the alarm player is stopped then don't process anymore alarm sounds.
    if (this.$stopped || !summary || !summary.records) {
      return;
    }
    var that = this,
      sourceCounts = AlarmAudioHandler.SOURCE_COUNTS,
      newSources = {},
      sources = that.$sources,
      toSound = [],
      toSilence = [],
      isAudibleAlarm = AlarmAudioHandler.isAudibleAlarm,
      toAlarmSound = _.bind(that.toAlarmSound, that),
      queue = that.$queue;

    // Convert the new alarm records array into a sources map.
    summary.records.forEach(function (rec) {
      newSources[rec.source] = rec;
    });
    function silenceIfCountIsZero(rec) {
      // If there are no sources at all or there are no sources after decrementing
      // then silence the specified sound.
      if (!sourceCounts[rec.source] || --sourceCounts[rec.source] <= 0) {
        delete sourceCounts[rec.source];
        toSilence.push(rec);
      }
    }

    // Find the new alarm sources that qualify for a sound and those
    // we need to silence.
    _.each(newSources, function (rec, source) {
      var oldRec = sources[source],
        oldRecAudible = oldRec && isAudibleAlarm(oldRec),
        newRecAudible = isAudibleAlarm(rec);
      if (!oldRecAudible && newRecAudible) {
        // If the sound for the source is not already playing then add it to the list of sounds
        // to play.
        if (!sourceCounts[rec.source]) {
          sourceCounts[rec.source] = 1;
          toSound.push(rec);
        } else {
          ++sourceCounts[rec.source];
        }
      } else if (oldRecAudible && !newRecAudible) {
        silenceIfCountIsZero(rec);
      }
    });

    // Find any other alarms that we need to silence by searching for those that
    // are missing and therefore closed.
    _.each(sources, function (rec, source) {
      if (isAudibleAlarm(rec) && !newSources[source]) {
        silenceIfCountIsZero(rec);
      }
    });

    // Update the cache of sources.
    that.$sources = newSources;

    // Convert to the alarm sounds we want to work with.
    toSound = toSound.map(toAlarmSound);
    toSilence = toSilence.map(toAlarmSound);

    // Remove any sounds to be silenced from the sound queue.
    toSilence.forEach(function (sound) {
      queue.removeFirst(sound);
    });

    // Add the sounds to be played to the queue.
    toSound.forEach(function (sound) {
      queue.add(sound);
    });

    // If there's no timer scheduled then play otherwise
    // the sound will just play in the not too distant future.
    if (!queue.isPlayScheduled() && toSound.length) {
      queue.play();
    }
  };
  return AlarmAudioHandler;
});
