/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/baja/ext/ConsoleRecipient
 */
define(['baja!', 'underscore'], function (baja, _) {
  'use strict';

  var Component = baja.Component,
    serverHandlerTypeSpec = "box:AlarmConsoleSummarySessionHandler",
    registrations = {},
    SUMMARY_UPDATE = "summaryUpdate";

  /**
   * Encapsulates an open alarms multi source registration.
   *
   * @inner
   * @private
   * @class
   * @alias module:nmodule/alarm/rc/baja/ext/Registration
   */
  var Registration = function Registration(id, recipient, callback) {
    var that = this,
      sub;
    baja.event.mixin(that);
    that.$id = id;
    that.$recipient = recipient;
    that.$sources = {};
    that.$lastUpdate = {};
    that.$loaded = false;

    // Listen for when the console recipient is removed from the Station.
    that.$subscriber = sub = new baja.Subscriber();
    sub.attach("unmount", function () {
      if (this === recipient) {
        that.cancel(/*fromUnmount*/true);
      }
    });
    sub.subscribe(recipient);
    that.attach(SUMMARY_UPDATE, callback);
  };

  /**
   * Return the id's registration.
   *
   * @returns {String} The registration's id.
   */
  Registration.prototype.getId = function () {
    return this.$id;
  };

  /**
   * Return the console recipient instance this registration is associated with.
   *
   * @returns {module:nmodule/alarm/rc/baja/ext/ConsoleRecipient} The console recipient.
   */
  Registration.prototype.getConsoleRecipient = function () {
    return this.$recipient;
  };

  /**
   * An unordered map of alarm source => row information.
   *
   * @returns {Object} An unordered map of sources to row information.
   */
  Registration.prototype.getSources = function () {
    return this.$sources;
  };

  /**
   * Return the total number of open alarms.
   *
   * @returns {Number} Total open alarms.
   */
  Registration.prototype.getAlarmCount = function () {
    var alarmCount = 0,
      sources = this.$sources,
      src;
    for (src in sources) {
      if (sources.hasOwnProperty(src)) {
        alarmCount += sources[src].acked + sources[src].unacked + sources[src].ackPending;
      }
    }
    return alarmCount;
  };

  /**
   * Cancel the registration.
   *
   * @returns {Promise} A promise that's resolved once the registration has been cancelled.
   */
  Registration.prototype.cancel = function (fromUnmount) {
    var that = this;

    // Short circuit to only cancel once.
    if (that.$cancelProm) {
      return that.$cancelProm;
    }
    that.$subscriber.detach();

    // Delete the registration up front.
    delete registrations[that.$id];

    // Detach all event handlers.
    that.detach();

    // Remove the server side handler.
    var cb = new baja.comm.Callback();
    baja.comm.removeServerHandler(that.$id, cb);
    return (that.$cancelProm = cb.promise()).then(function () {
      if (!fromUnmount) {
        // Unsubscribe all and detach all event handlers.
        return that.$subscriber.unsubscribeAll();
      }
    });
  };

  /**
   * Returns true if the registration is canceled.
   *
   * @returns {Boolean} Return true if canceled.
   */
  Registration.prototype.isCanceled = function () {
    return !!this.$cancelProm;
  };

  /**
   * BajaScript representation of an `alarm:ConsoleRecipient` value.
   *
   * @class
   * @extends baja.Component
   * @alias module:nmodule/alarm/rc/baja/ext/ConsoleRecipient
   */
  var ConsoleRecipient = function ConsoleRecipient() {
    Component.apply(this, arguments);
  };
  ConsoleRecipient.prototype = Object.create(Component.prototype);
  ConsoleRecipient.prototype.constructor = ConsoleRecipient;
  ConsoleRecipient.Registration = Registration;

  /**
   * The name of the key used to listen to new alarm updates.
   *
   * @type {String}
   */
  ConsoleRecipient.SUMMARY_UPDATE = SUMMARY_UPDATE;

  /**
   * An object that contains all open active registrations for
   * alarm information.
   *
   * @private
   *
   * @type {Object}
   */
  ConsoleRecipient.$registrations = registrations;

  /**
   * Find a unique server handler id we can use for a registration.
   *
   * @private
   *
   * @returns {String} A unique server handler id.
   */
  ConsoleRecipient.prototype.$findUniqueServerHandlerId = function () {
    var navOrd = this.getNavOrd(),
      baseId = String(this.getType()) + ":" + baja.SlotPath.escape(String(navOrd)),
      id = baseId;
    for (var i = 0; baja.comm.hasServerHandler(id); ++i) {
      id = baseId + ":" + i;
    }
    return id;
  };

  /**
   * Converts any JSON alarm records into BajaScript alarm records.
   *
   * @private
   *
   * @param  {Object} sources An object of alarm source => row information.
   * @returns {Promise.<object>} A promise that's resolved once all of the alarm records
   * have been converted to BajaScript objects. The result is a safe copy of the
   * input.
   */
  ConsoleRecipient.prototype.$convertAlarmRecords = function (sources) {
    return baja.bson.importUnknownTypes(sources).then(function () {
      // Decode all the alarm records to BajaScript objects.
      var src,
        val,
        result = Object.assign({}, sources);
      for (src in sources) {
        if (sources.hasOwnProperty(src) && sources[src].record) {
          result[src] = Object.assign({}, sources[src]);
          val = baja.bson.decodeValue(sources[src].record, baja.$serverDecodeContext);
          if (val) {
            result[src].record = val;
          }
        }
      }
      return result;
    });
  };

  /**
   * Registers a query for multi source summary alarm information. This method resolves to an
   * object that can be listened to for updating alarm event information.
   *
   * @param {Function} callback A function callback that's invoked anytime there are new events.
   * @param [filterSet] An optional filter set to query the alarms against.
   * @param [timeRange] An optional dynamic time range to be used in a filter.
   * @returns {Promise} A promise that resolves to an alarm multi source summary registration object.
   */
  ConsoleRecipient.prototype.registerMultiSourceSummaryQuery = function (callback, filterSet, timeRange) {
    // Lazily create server side call handler for managing live alarm console information.
    var that = this,
      id = that.$findUniqueServerHandlerId(),
      cb = new baja.comm.Callback(),
      // The parameters used to set up the server side session handler.
      params = {
        ord: String(that.getNavOrd())
      },
      // Holds all registration information.
      reg = new Registration(id, that, callback);
    if (filterSet) {
      params.filterSet = baja.bson.encodeValue(filterSet);
    }
    if (timeRange) {
      params.timeRange = baja.bson.encodeValue(timeRange);
    }
    baja.comm.makeServerHandler(id, serverHandlerTypeSpec, params, function (data) {
      return that.$convertAlarmRecords(data.sources).then(function (sources) {
        var row, newRow, src;

        // Iterate through each of the row updates.
        for (src in sources) {
          if (sources.hasOwnProperty(src)) {
            newRow = sources[src];

            // If there is already a row then update it.
            if (reg.$sources.hasOwnProperty(src)) {
              row = reg.$sources[src];
              if (newRow.record) {
                row.record = newRow.record;
              }
              row.acked += newRow.acked;
              row.unacked += newRow.unacked;
              row.ackPending += newRow.ackPending;
              row.acked = Math.max(row.acked, 0);
              row.unacked = Math.max(row.unacked, 0);
              row.ackPending = Math.max(row.ackPending, 0);
            } else if (newRow.open) {
              // Otherwise just add the new row.
              reg.$sources[src] = newRow;
            }
          }
        }

        // Remove sources that have zero alarms in them or if the alarm is force cleared.
        for (src in reg.$sources) {
          if (reg.$sources.hasOwnProperty(src) && reg.$sources[src].acked === 0 && reg.$sources[src].unacked === 0 && reg.$sources[src].ackPending === 0) {
            delete reg.$sources[src];
          }
        }

        // Fire the update
        reg.fireHandlers(SUMMARY_UPDATE, baja.error, that, reg, /*update*/true);
        return reg;
      });
    }, cb);
    function queryMultiSourceSummary() {
      var cb = new baja.comm.Callback(),
        moreAlarms = false;
      baja.comm.serverHandlerCall(id, "queryMultiSourceSummary", {}, cb);
      return cb.promise().then(function (data) {
        moreAlarms = data.moreAlarms;
        return that.$convertAlarmRecords(data.sources);
      }).then(function (sources) {
        var cb;
        _.extend(reg.$sources, sources);

        // Bail if the registration object has been canceled.
        if (!reg.isCanceled()) {
          // If there are more alarms then do another query.
          if (moreAlarms) {
            return queryMultiSourceSummary();
          } else {
            // All done loading so register for any new incoming alarms.
            cb = new baja.comm.Callback();
            baja.comm.serverHandlerCall(id, "completeInitialization", {}, cb);
            return cb.promise();
          }
        }
      });
    }
    return cb.promise().then(function () {
      return queryMultiSourceSummary();
    }).then(function () {
      registrations[id] = reg;

      // Fire the update for the first time.
      reg.fireHandlers(SUMMARY_UPDATE, baja.error, that, reg, /*update*/false);
      return reg;
    });
  };
  return ConsoleRecipient;
});
