/**
 * @copyright 2015, Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson.
 */

/* eslint-env browser */

/**
 * API Status: **Private**
 *
 * Full Screen Hx validation for the Shell Hx Profile.
 *
 * @module nmodule/hx/container/fullScreenUtils
 */
(function () {
  "use strict";

  var exports = {};

  /**
   * A static method that will remove the full screen parameter if 
   * this JavaScript is running in the top window that doesn't have a window name that begins with "fullScreen".
   * This will reload the page as necessary.
   *
   * @private
   *
   * @param [location] Optional browser window's location object. If not specified, the 
   * browser window will be used.
   * @param [isTopWindow] Optional flag to determine whether the current browser window
   * is the top widget. If not specified, the window detection will be made by this method.
   */
  exports.validate = function (location, isTopWindow) {
    location = location || window.location;
    var originalUri = location.href.replace("/ord?", "/ord/"),
      uri = originalUri,
      match,
      params;

    //once in fullScreen Popup window, stay in fullscreen Popup by ensuring fullScreen parameter is always present and true
    if (window.name && window.name.indexOf && window.name.indexOf("fullScreen") > -1) {
      if (window.hx && !window.hx.isFullScreen()) {
        match = /\?fullScreen(=|%3[dD])false(.+)$/.exec(location.href);
        if (match && match[1]) {
          //has fullScreen view parameter
          uri = uri.replace(/fullScreen(=|%3[dD])false/, "fullScreen=true");
        } else {
          //if no fullScreen view query, add it
          uri += "|view:?fullScreen=true";
        }
        if (uri !== originalUri) {
          location.replace(uri);
        }
      }
      return;
    }
    if (isTopWindow === undefined) {
      isTopWindow = window === window.top;
    }
    if (isTopWindow) {
      match = /\?(.+)$/.exec(uri);
      if (match && match[1]) {
        params = match[1];
        params = params.replace(/fullScreen(=|%3[dD])true/, "fullScreen=false");
        uri = uri.substring(0, match.index) + "?" + params;
      }
      if (uri !== originalUri) {
        location.replace(uri);
      }
    }
  };

  /**
   * If this page contains an iframe, wait for iframe's loadEvent to end.
   * @param {String} eventId
   */
  exports.loadTime = function (eventId) {
    var hx = window.hx;
    if (!window.performance || !window.performance.timing || !hx) {
      return; //browser does not support window.performance.timing or hx is not present
    }
    if (!hx.isFullScreen()) {
      require(["Promise", "jquery"], function (Promise, $) {
        hx.waitForLoadTime(function truthyFunction() {
          var iframe = $("#servletViewWidget")[0];
          return !iframe || iframe && iframe.contentWindow.performance.timing.loadEventEnd;
        }, function endTimeFunction() {
          var iframe = $("#servletViewWidget")[0];
          var endTime = iframe ? iframe.contentWindow.performance.timing.loadEventEnd : window.performance.timing.loadEventEnd;
          return endTime;
        });
        hx.loadTime(eventId);
      });
    }
  };
  window.fullScreenUtils = exports;
  if (typeof define === "function" && define.amd) {
    define("nmodule/hx/rc/container/fullScreenUtils", [], function () {
      return exports;
    });
  }
})();
