/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*jshint browser: true */ /* eslint-env browser */

/**
 * API Status: **Private**
 * @module nmodule/js/rc/asyncUtils/asyncUtils
 */
define(['Promise', 'underscore'], function (Promise, _) {
  'use strict';

  var compact = _.compact,
    each = _.each,
    every = _.every,
    isArray = _.isArray,
    isFunction = _.isFunction,
    negate = _.negate,
    once = _.once;
  var WAIT_FOR_TRUE_INTERVAL = 10;
  var WAIT_FOR_TRUE_TIMEOUT = 5000;

  /**
   * Utilities for working with promises and other async functions like
   * RequireJS imports and callbacks.
   *
   * @exports nmodule/js/rc/asyncUtils/asyncUtils
   */
  var exports = {};
  function resolveSequentially(layers) {
    var layer = layers && layers[0];
    return Promise.resolve(layer && exports.doRequire(layer).then(function () {
      return resolveSequentially(layers.slice(1));
    }));
  }
  function depsInvariant(deps) {
    if (deps) {
      var allArrays = every(deps, isArray);
      var noArrays = every(deps, negate(isArray));
      if (!allArrays && !noArrays) {
        throw new Error('deps must be all strings or all arrays');
      }
    }
    return deps;
  }
  function isArrayOfArrays(arr) {
    return isArray(arr) && isArray(arr[0]);
  }
  function toFulfilledResult(value) {
    return {
      status: 'fulfilled',
      value: value
    };
  }
  function toRejectedResult(reason) {
    return {
      status: 'rejected',
      reason: reason
    };
  }
  function isFulfilledResult(result) {
    return result.status === 'fulfilled';
  }
  function isRejectedResult(result) {
    return result.status === 'rejected';
  }
  function getResultValue(result) {
    return result.value;
  }
  function getResultReason(result) {
    return result.reason;
  }

  /**
   * @param {Array.<*|Promise.<*>>} args values, or promises to resolve
   * @returns {Promise.<Array>} an array of results with `status` of `fulfilled` or `rejected`, and
   * either a `value` or `reason` property as appropriate. You can also access the `resolved` and
   * `rejected` properties of the result for arrays of the resolved values and rejected reasons.
   * @since Niagara 4.13
   */
  exports.allSettled = function (args) {
    var settledResults;
    if (typeof Promise.allSettled === 'function') {
      settledResults = Promise.allSettled(args);
    } else {
      settledResults = Promise.all(args.map(function (arg) {
        return Promise.resolve(arg).then(toFulfilledResult, toRejectedResult);
      }));
    }
    return settledResults.then(function (result) {
      Object.defineProperty(result, 'resolved', {
        enumerable: false,
        get: function get() {
          return result.filter(isFulfilledResult).map(getResultValue);
        }
      });
      Object.defineProperty(result, 'rejected', {
        enumerable: false,
        get: function get() {
          return result.filter(isRejectedResult).map(getResultReason);
        }
      });
      return result;
    });
  };

  /**
   * Load some RequireJS dependencies in a promisified way.
   *
   * As of Niagara 4.6, this has support for ES6 modules transpiled to AMD using
   * Babel. Any `export default` present will be used as the value to resolve
   * `doRequire`; if `export default` is not used then you will receive the full
   * module exports. Passing the `resolveEntire` flag will resolve the full
   * module exports even if a default is present. (If this does not make sense,
   * you're probably not transpiling ES6 modules and you can safely ignore this
   * paragraph.)
   *
   * @param {String|Array.<String>} modules the RequireJS modules to retrieve.
   * If this is a single module ID, the promise will be resolved with the
   * result of the one resolved module. If this is an array of module IDs, the
   * promise will be resolved with an array of the resolved modules.
   * @param {object|Array.<string>} [params] parameters (can also be the array
   * of dependencies passed directly)
   * @param {Array.<string>|Array.<Array.<string>>} [params.deps] An optional array of dependencies
   * to preload before resolving the requested modules. This can also be an array of arrays of
   * dependencies (e.g. BIWebResource dependency groups): the dependencies in each array will be
   * resolved concurrently as a group before moving on to resolve the next array.If deps contains a
   * mixture of strings and arrays, an error will be thrown.
   * @param {boolean} [params.sequentialDeps] set to true to resolve dependencies sequentially
   * instead of in parallel.
   * @param {boolean} [params.resolveEntire=false] set to true to always resolve
   * the entire transpiled ES6 module, even if a default export is present. You
   * will need to refer to `yourResolvedModule.default` in your code directly.
   * @returns {Promise} promise to be resolved with the resolved module
   * or array of resolved modules, or rejected if any modules or dependencies
   * could not be found
   */
  exports.doRequire = function (modules, params) {
    var isArray = Array.isArray(modules);
    var deps = depsInvariant(Array.isArray(params) ? params : params && params.deps);
    var resolveEntire = params && params.resolveEntire;
    var resolveSequential = isArrayOfArrays(deps);
    if (!isArray) {
      modules = [modules];
    }
    var toRequire = [];
    var results = [];
    modules.forEach(function (module, i) {
      if (require.defined(module)) {
        results[i] = require(module);
      } else {
        toRequire[i] = module;
      }
    });
    function resolveDefault(module) {
      if (!resolveEntire && isEs6Transpiled(module) && 'default' in module) {
        return module["default"];
      }
      return module;
    }
    function getResults() {
      return isArray ? results.map(resolveDefault) : resolveDefault(results[0]);
    }
    return Promise.resolve(deps && (resolveSequential ? resolveSequentially(deps) : exports.doRequire(deps))).then(function () {
      if (compact(toRequire).length) {
        // eslint-disable-next-line promise/avoid-new
        return new Promise(function (resolve, reject) {
          require(toRequire, function () {
            each(arguments, function (obj, i) {
              if (obj !== undefined) {
                results[i] = obj;
              }
            });
            resolve(getResults());
          }, reject);
        });
      } else {
        return getResults();
      }
    });
  };
  function isEs6Transpiled(m) {
    //'__esModule' is injected by babel-transpiled es6 modules to mark them as such.
    return m && m.__esModule;
  }

  /**
   * Resolves all the given promises and resolves an array of their results.
   *
   * @deprecated just use Promise.all()
   * @param {Array.<jQuery.Promise>} promises array of promises (or raw values)
   * @returns {Promise} promise to be resolved with array of the results
   * of the resolved promises, or rejected if any of the promises reject
   */
  exports.whenAll = function (promises) {
    return Promise.all(promises);
  };

  /**
   * Invoke a function on an object or resolve a promise with undefined if the function or object does not exist.
   * Any additional parameters will be passed to the function when its called.
   *
   * @param {String} funcName the name of function to call
   * @param {*} object the object to call the func function on
   * @param {...*} args additional args will be passed to the function invocation
   * @returns {Promise} resolves a promise to the function result, or resolves to 'undefined' if the function or object does not exist.
    */
  exports.optionalInvoke = function (funcName, object) {
    var func;
    for (var _len = arguments.length, args = new Array(_len > 2 ? _len - 2 : 0), _key = 2; _key < _len; _key++) {
      args[_key - 2] = arguments[_key];
    }
    return Promise.resolve(object && (func = object[funcName]) && isFunction(func) && func.apply(object, args) || undefined);
  };

  /**
   * Maps the array of values to promises, resolves all the results, and
   * resolves an array of the results of the resolved promises.
   *
   * @deprecated use Promise.all(arr.map(func)), or Promise.all(_.invoke(arr))
   * @param {Array} arr
   * @param {Function} func a function that accepts each element of the array
   * and returns a promise (or a raw value).
   * @returns {Promise} promise to be resolved with an array of the
   * results of the resolved promises, or rejected if any of the promises
   * reject
   */
  exports.mapAsync = function (arr, func) {
    if (typeof func === 'string') {
      var funcName = func;
      var args = Array.prototype.slice.call(arguments, 2);
      func = function func(obj) {
        return obj[funcName].apply(obj, args);
      };
    }
    try {
      return Promise.all(arr.map(func));
    } catch (e) {
      return Promise.reject(e);
    }
  };

  /**
   * Run a promise, using `setTimeout` to check for the truthiness of the
   * condition function.
   *
   * @param {Function} func resolve the promise when this function returns a
   * truthy value, or a promise that resolves to a truthy value
   * @param {Number} [timeout=5000] the time, in milliseconds, after which to
   * give up waiting and reject
   * @returns {Promise.<*>} that resolves to the truthy value, as soon as it is
   * available
   */
  exports.waitForTrue = function (func, timeout) {
    if (typeof func !== 'function') {
      return Promise.reject(new Error('test function required'));
    }
    timeout = timeout || WAIT_FOR_TRUE_TIMEOUT;
    var start = +new Date();

    // eslint-disable-next-line promise/avoid-new
    return new Promise(function (resolve, reject) {
      (function test() {
        Promise.resolve(func()).then(function (result) {
          if (result) {
            return resolve(result);
          }
          if (new Date() - start < timeout) {
            exports.$setTimeout(test, WAIT_FOR_TRUE_INTERVAL);
          } else {
            reject(new Error('timed out after ' + timeout + 'ms'));
          }
        })["catch"](reject);
      })();
    });
  };

  /**
   * Wait approximately the given timeout and then resolve.
   * @param {number} timeout timeout in milliseconds
   * @returns {Promise}
   * @since Niagara 4.8
   */
  exports.waitInterval = function (timeout) {
    // eslint-disable-next-line promise/avoid-new
    return new Promise(function (resolve) {
      exports.$setTimeout(resolve, timeout);
    });
  };

  /**
   * This works like underscore's debounce function, but additionally ensures
   * that subsequent calls to the function will not execute until the previous
   * call's promise has resolved, even if the timeout elapses.
   *
   * @param {function} func the function to invoke
   * @param {number} [wait] how long to wait before the function
   * @param {boolean} [immediate]
   * @returns {function}
   * @since Niagara 4.10
   */
  exports.debounceAsync = function (func, wait, immediate) {
    var ticket;
    var prom;
    function execute(args) {
      function runFunction() {
        return (prom = Promise.resolve(func.apply(null, args))).then(function () {
          prom = 0;
          ticket = 0;
        });
      }
      return prom ? prom.then(runFunction) : runFunction();
    }
    return function () {
      for (var _len2 = arguments.length, args = new Array(_len2), _key2 = 0; _key2 < _len2; _key2++) {
        args[_key2] = arguments[_key2];
      }
      var inProgress = !!prom;
      clearTimeout(ticket);
      if (immediate && !inProgress) {
        return execute(args);
      } else {
        // eslint-disable-next-line promise/avoid-new
        return new Promise(function (resolve, reject) {
          ticket = exports.$setTimeout(function () {
            execute(args).then(resolve, reject);
          }, wait);
        });
      }
    };
  };

  /**
   * @private
   * @returns {{ resolve: function, reject: function, promise: Promise, state: string }}
   */
  exports.$deferred = function () {
    var df = {};
    var state = 'pending';
    // eslint-disable-next-line promise/avoid-new
    df.promise = new Promise(function (resolve, reject) {
      var settle = once(function (isResolve, arg) {
        state = isResolve ? 'resolved' : 'rejected';
        return (isResolve ? resolve : reject)(arg);
      });
      df.resolve = function (arg) {
        return settle(true, arg);
      };
      df.reject = function (arg) {
        return settle(false, arg);
      };
    });
    Object.defineProperty(df, 'state', {
      get: function get() {
        return state;
      }
    });
    return df;
  };

  /**
   * @private
   * @returns {boolean} true if we want waitInterval/waitForTrue to use the
   * Jasmine mock clock if installed - default false
   */
  exports.$respectMock = function () {
    return false;
  };

  /**
   * In most cases, combining the Jasmine mock clock with these Promise-based
   * functions does not work. In the case of waitInterval, for instance,
   * completing the setTimeout ticket won't make the Promise resolve, so you
   * still have to waitFor the resolution, which uses setTimeout...
   *
   * In 99.9% of cases you won't want setTimeout to affect the resolution of
   * Promises. When you want to introduce artificial delays to Promise
   * resolution, use spies in conjunction with waitInterval. But in the rare
   * case you actually do want a Promise resolution to rely on the mock clock
   * to resolve, spy on $respectMock to return true.
   *
   * @private
   * @param {function} func
   * @param {number} timeout
   * @return {number} timeout ticket
   */
  exports.$setTimeout = function (func, timeout) {
    if (!exports.$respectMock() && isMockClockInUse()) {
      func();
    } else {
      return setTimeout(func, timeout);
    }
  };
  function isMockClockInUse() {
    if (typeof jasmine === 'undefined') {
      return false;
    }
    return jasmine.Clock.installed === jasmine.Clock.defaultFakeTimer;
  }
  return exports;
});
