/**
 * @copyright 2022 Tridium, Inc. All Rights Reserved.
 */

define([], function (_, Promise) {
  'use strict';

  /**
   * A callback that gets called for each potential component that could be subscribed to
   * during a Depth Subscription. If returning true, then any child components will also
   * get a similar callback.
   *
   * @callback nmodule/webEditors/rc/wb/util/subscriptionUtil~SubscriptionFilter
   *
   * @param {Object} params The object literal for the parameters.
   * @param {baja.Component} params.component
   * @param {Number} params.depth  The depth of the component passed as the parameter.
   * Workbench and UX sometimes have differing definitions of subscription depth, so an
   * adjustment may be required:
   *
   *  BComponent.java#lease:
   *    If depth is greater than zero then the lease includes descendants
   *    (one is children, two is children and grandchildren, etc).
   *  DepthSubscriber.js#subscribeAll
   *     @param {Number} depth the depth to subscribe - 0 subscribes nothing,
   *     1 subscribes only the given components, 2 subscribes components
   *     and their kids, etc.
   *  subscriptionUtil.js#SubscriptionFilter
   *    @param {Number} depth the depth that is passed to the filter function is just like
   *    the depth from BComponent.java#lease, 1 for when the component is a child, and 2 for
   *    when it is a grandchild, etc. This function is available in Niagara 4.13 and higher.
   *
   * @returns {boolean}
   */

  /**
   * A callback that gets called for each subscribe from the DepthSubscriber.
   * This callback is helpful for managing additional subscriptions outside
   * any direct parent to child relationships from a `subscriptionFilter`.
   *
   * @callback nmodule/webEditors/rc/wb/util/subscriptionUtil~SubscribeCallback
   *
   * @param {Object} params the object literal for the parameters.
   * @param {baja.Component} params.component
   * @param {baja.Subscriber} params.subscriber
   * @returns {Promise}
   */

  /**
   * API Status: **Development**
   *
   * Utilities for working with subscription. The exported functions are can work with custom
   * DriverMgr.js subclasses to improve their subscription efficiency.
   *
   * @exports nmodule/webEditors/rc/wb/util/subscriptionUtil
   * @see module:nmodule/driver/rc/wb/mgr/DriverMgr
   * @since Niagara 4.13
   */
  var exports = {};

  /**
   * This basic subscription filter is useful for ensuring that a component and/or
   * its parent container are of certain types.
   *
   * @param {Object} params the object literal for the parameters.
   * @param {baja.Component} params.component
   * @param {Number} params.depth
   * @param {Number} [params.allowedDepth=1] Any depth greater than the allowed depth will return false.
   * @param {Array.<Type|String>|Type|String} [params.containerTypes]
   * @param {Array.<Type|String>|Type|String} [params.componentTypes]
   * @returns {boolean}
   */
  exports.basicSubscriptionFilter = function (_ref) {
    var component = _ref.component,
      depth = _ref.depth,
      _ref$allowedDepth = _ref.allowedDepth,
      allowedDepth = _ref$allowedDepth === void 0 ? 1 : _ref$allowedDepth,
      _ref$containerTypes = _ref.containerTypes,
      containerTypes = _ref$containerTypes === void 0 ? [] : _ref$containerTypes,
      _ref$componentTypes = _ref.componentTypes,
      componentTypes = _ref$componentTypes === void 0 ? [] : _ref$componentTypes;
    containerTypes = [].concat(containerTypes);
    componentTypes = [].concat(componentTypes);
    if (depth > allowedDepth) {
      return false;
    }
    var componentType = component.getType();
    if (componentTypes.length > 0) {
      if (componentTypes.filter(function (type) {
        return componentType.is(type);
      }).length === 0) {
        return false;
      }
    }
    if (containerTypes.length > 0) {
      var parent = component.getParent();
      if (!parent) {
        return false;
      }
      var parentType = parent.getType();
      if (containerTypes.filter(function (type) {
        return parentType.is(type);
      }).length === 0) {
        return false;
      }
    }
    return true;
  };
  return exports;
});
