/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * @module nmodule/webEditors/rc/wb/table/model/ComponentSource
 */
define(['baja!', 'nmodule/js/rc/tinyevents/tinyevents'], function (baja, tinyevents) {
  'use strict';

  /**
   * API Status: **Development**
   *
   * For use with a `ComponentTableModel`, acts as a source of components to
   * load into table rows.
   *
   * This should be overridden to provide components from different kinds of
   * sources; e.g. search results. Each subclass must obey the following
   * contract:
   *
   * - `getComponents()` returns an array of all available components
   * - `added` and `removed` events are emitted when the list of components
   *   changes (to notify the `ComponentTableModel` to add or remove rows).
   *   These should be fired with arrays of all components that were just
   *   added or removed.
   * - `changed` events are emitted when one of the components in the list
   *   has been changed (to notify the `ComponentTableModel` that row contents
   *   have changed). These should be fired once per component that changed,
   *   with two arguments: the component that changed, and the property on that
   *   component that changed.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/table/model/ComponentSource
   * @mixes tinyevents
   * @param {Object|baja.Component} obj params object, or the container itself
   * @param {Object} obj.container the `Component` container
   * @see module:nmodule/webEditors/rc/wb/table/model/ComponentTableModel
   */
  var ComponentSource = function ComponentSource(obj) {
    obj = baja.objectify(obj, 'container');
    this.$container = obj.container;
    tinyevents(this);
  };

  /**
   * The container object passed to the constructor.
   *
   * @returns {Object}
   */
  ComponentSource.prototype.getContainer = function () {
    return this.$container;
  };

  /**
   * Gets an array of all matching child Components of the container. Must
   * be implemented by subclasses.
   *
   * @abstract
   * @returns {Array.<baja.Component>}
   * @throws {Error} if not implemented
   */
  ComponentSource.prototype.getComponents = function () {
    throw new Error('getComponents() not implemented');
  };

  /**
   * Add new components to this component source. Must be implemented by
   * subclasses and emit `added` event as appropriate.
   *
   * @abstract
   * @param {Array.<baja.Component>} comps
   * @param {Array.<String>} [names] desired names for the components; may not
   * always be respected by subclasses
   * @returns {Promise.<Array.<baja.Component>>} as of Niagara 4.13, will be
   * resolved with the actual component instances after they are added
   * @throws {Error} if not implemented
   */
  ComponentSource.prototype.addComponents = function (comps, names) {
    throw new Error('addComponents() not implemented');
  };

  /**
   * Remove these components from this component source. Must be implemented
   * by subclasses and emit `removed` event as appropriate.
   *
   * @abstract
   * @param {Array.<baja.Component>} comps
   * @returns {Promise}
   * @throws {Error} if not implemented
   */
  ComponentSource.prototype.removeComponents = function (comps) {
    throw new Error('removeComponents() not implemented');
  };

  /**
   * Clean up event handlers associated with this component source.
   */
  ComponentSource.prototype.destroy = function () {};
  return ComponentSource;
});
