/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['Promise', 'underscore', 'nmodule/js/rc/asyncUtils/asyncUtils', 'nmodule/webEditors/rc/servlets/registry'], function (Promise, _, asyncUtils, registry) {
  'use strict';

  var doRequire = asyncUtils.doRequire,
    getEnabledMixinsOn = registry.getEnabledMixinsOn,
    compact = _.compact,
    first = _.first,
    flatten = _.flatten,
    invoke = _.invoke,
    map = _.map;
  function toTypeSpec(obj) {
    return obj.type;
  }

  /**
   * API Status: **Private**
   *
   * Functions for querying the registry about which manager columns to use for
   * particular Types.
   *
   * @exports nmodule/webEditors/rc/wb/mgr/model/mgrAgents
   */
  var mgrAgents = {};

  /**
   * This defines the JS-Based representation for a `IJavaScriptMgrAgent`.
   *
   * @typedef {Object} mgrAgents~managerAgentModule
   * @property {function(): Array.<module:nmodule/webEditors/rc/wb/mgr/model/MgrColumn>} getColumns
   * @property {function(baja.Component): Array.<baja.Component>} [getSubscriptionComponents]
   */

  /**
   * Called when an enabledMixInCallback is provided to `$getAgentsForMixinsOn`
   * @private
   * @callback mgrAgents~enabledMixInCallback
   * @param {Array.<module:nmodule/webEditors/rc/servlets/registry~TypeInfo>} enabledTypeMixins
   * @param {Array.<mgrAgents~managerAgentModule>} agentModules if the enabledTypeMixins of the same index does not have a registered agentModel,
   * then the value in the agentModules array will be undefined.
   *
   */

  /**
   * @private
   * @param {Type|String} type
   * @param {mgrAgents~enabledMixInCallback} [enabledMixInCallback] An optional callback that if provided, will be
   * given the enabledTypeMixins and the required MgrAgents and this callback will decide the results.
   * @returns {Promise.<Array.<mgrAgents~managerAgentModule>|*>} promise to be resolved
   * with an array of manager agent modules (call `getColumns` on each). If a
   * enabledMixInCallback is provided, the results to that function will be resolved.
   */
  mgrAgents.$getAgentsForMixinsOn = function (type, enabledMixInCallback) {
    var enabledTypeMixins;
    return getEnabledMixinsOn(type).then(function (results) {
      enabledTypeMixins = results;
      return registry.getAgentOnInfo(enabledTypeMixins.map(toTypeSpec), {
        is: 'webEditors:IJavaScriptMgrAgent'
      }).then(function (agentInfos) {
        return agentInfos.map(function (agentInfo) {
          return first(agentInfo);
        });
      });
    }).then(function (mgrAgents) {
      return Promise.all(mgrAgents.map(function (mgrAgentInfo) {
        if (mgrAgentInfo) {
          return doRequire(mgrAgentInfo.js.id, mgrAgentInfo.js.deps);
        }
      }));
    }).then(function (agentModules) {
      if (enabledMixInCallback) {
        return enabledMixInCallback(enabledTypeMixins, agentModules);
      }
      return compact(agentModules);
    });
  };

  /**
   * Query the registry for `IJavaScriptMgrAgent`s registered as mixins on the
   * given type, and resolve an array of all `MgrColumn`s to be added to a
   * manager.
   *
   * @param {Type|String} type
   * @returns {Promise.<Array.<module:nmodule/webEditors/rc/wb/mgr/model/MgrColumn>>}
   */
  mgrAgents.getMixInColumns = function (type) {
    return mgrAgents.$getAgentsForMixinsOn(type).then(function (agentModules) {
      return flatten(invoke(agentModules, 'getColumns'));
    });
  };

  /**
   * Query the registry for `IJavaScriptMgrAgent`s registered as mixins on the
   * given type, and resolve an array of all subscriptions resolved from calling
   * the optional `getSubscriptionComponents` function.
   *
   * @param {baja.Component} component
   * @returns {Promise.<Array.<baja.Component>>}
   * @since Niagara 4.13
   */
  mgrAgents.getMixInManagerSubscriptions = function (component) {
    var type = component.getType();
    var getSubscriptionComponents = function getSubscriptionComponents(enabledTypeMixins, agentModules) {
      return Promise.all(map(agentModules, function (agentModule, index) {
        if (agentModule && agentModule.getSubscriptionComponents) {
          var mixin = component.get(toSlotName(enabledTypeMixins[index].type));
          if (mixin) {
            return agentModule.getSubscriptionComponents(mixin);
          }
        }
        return [];
      })).then(function (components) {
        return flatten(components);
      });
    };
    return mgrAgents.$getAgentsForMixinsOn(type, getSubscriptionComponents);
  };
  function toSlotName(type) {
    return String(type).replace(':', '_');
  }
  return mgrAgents;
});
