function _slicedToArray(r, e) { return _arrayWithHoles(r) || _iterableToArrayLimit(r, e) || _unsupportedIterableToArray(r, e) || _nonIterableRest(); }
function _nonIterableRest() { throw new TypeError("Invalid attempt to destructure non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method."); }
function _unsupportedIterableToArray(r, a) { if (r) { if ("string" == typeof r) return _arrayLikeToArray(r, a); var t = {}.toString.call(r).slice(8, -1); return "Object" === t && r.constructor && (t = r.constructor.name), "Map" === t || "Set" === t ? Array.from(r) : "Arguments" === t || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(t) ? _arrayLikeToArray(r, a) : void 0; } }
function _arrayLikeToArray(r, a) { (null == a || a > r.length) && (a = r.length); for (var e = 0, n = Array(a); e < a; e++) n[e] = r[e]; return n; }
function _iterableToArrayLimit(r, l) { var t = null == r ? null : "undefined" != typeof Symbol && r[Symbol.iterator] || r["@@iterator"]; if (null != t) { var e, n, i, u, a = [], f = !0, o = !1; try { if (i = (t = t.call(r)).next, 0 === l) { if (Object(t) !== t) return; f = !1; } else for (; !(f = (e = i.call(t)).done) && (a.push(e.value), a.length !== l); f = !0); } catch (r) { o = !0, n = r; } finally { try { if (!f && null != t["return"] && (u = t["return"](), Object(u) !== u)) return; } finally { if (o) throw n; } } return a; } }
function _arrayWithHoles(r) { if (Array.isArray(r)) return r; }
/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/* jshint browser: true */

/**
 * API Status: **Private**
 *
 * Type providing functionality to support the learn table on a manager view. This is
 * intended to mix in functionality that is user interface specific, and should be
 * kept out of the `MgrLearn` mixin; i.e. this is intended for stuff that is private to
 * the view's implementation, and will not be part of the exposed learn API.
 *
 * This is not intended to be directly mixed-in by a manager implementation. The exported
 * functions will be applied to the manager at the same time that the `MgrLearn` mixin
 * is applied, and will be done automatically, so there's no need for concrete managers
 * to do it themselves.
 *
 * @module nmodule/webEditors/rc/wb/mgr/MgrLearnTableSupport
 * @mixin
 * @extends module:nmodule/webEditors/rc/wb/mgr/MgrLearn
 */
define(['baja!', 'lex!webEditors', 'log!nmodule.webEditors.rc.wb.mgr.MgrLearnTableSupport', 'bajaux/Widget', 'bajaux/dragdrop/dragDropUtils', 'Promise', 'jquery', 'underscore', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/util/ListSelection', 'nmodule/webEditors/rc/wb/job/JobBar', 'nmodule/webEditors/rc/wb/mixin/mixinUtils', 'nmodule/webEditors/rc/wb/table/menu/DefaultTableContextMenu', 'nmodule/webEditors/rc/wb/table/model/TableModel', 'nmodule/webEditors/rc/wb/table/tree/TreeTable', 'hbs!nmodule/webEditors/rc/wb/mgr/template/Manager', 'nmodule/js/rc/jquery/split-pane/split-pane', 'css!nmodule/js/rc/jquery/split-pane/split-pane'], function (baja, lexs, log, Widget, dragDropUtils, Promise, $, _, fe, feDialogs, ListSelection, JobBar, mixinUtils, DefaultTableContextMenu, TableModel, TreeTable, tplManager) {
  'use strict';

  var _lexs = _slicedToArray(lexs, 1),
    webEditorsLex = _lexs[0];
  var logError = log.severe.bind(log);
  var logFine = log.fine.bind(log);
  var MIXIN_NAME = '$MGR_LEARN_TABLE';
  var fromClipboard = dragDropUtils.fromClipboard;
  var exports = function exports(target, params) {
    var tableConstructor = params.tableCtor;
    var superDoInitialize = target.doInitialize;
    var superDoLayout = target.doLayout;
    var superDoLoad = target.doLoad;
    var superDoDestroy = target.doDestroy;
    var onLearnTableDblClickedOverride = target.onLearnTableDblClicked;
    var SPLIT_PANE_THROTTLE_MS = 200;
    var UPDATED_MODEL_THROTTLE_MS = 250;
    var softExtensions = {
      armLearnTableContextMenu: function armLearnTableContextMenu(table, jq, selector) {
        new DefaultTableContextMenu(table).arm(jq, selector);
      }
    };
    if (!mixinUtils.applyMixin(target, MIXIN_NAME, softExtensions)) {
      return;
    }

    ////////////////////////////////////////////////////////////////
    // Utility Functions
    ////////////////////////////////////////////////////////////////

    function isLearnModeCommandSelected(mgr) {
      var cmd = mgr.$getLearnModeCommand();
      return cmd && cmd.isSelected();
    }
    function setLearnModel(mgr, model) {
      mgr.$learnModel = model;
    }
    function getLearnModel(mgr) {
      return mgr.$learnModel;
    }

    ////////////////////////////////////////////////////////////////
    // Initialize / Layout / Load / Destroy
    ////////////////////////////////////////////////////////////////

    /**
     * Extend the base Manager's doInitialize() function, by adding the
     * discovery tables and the title elements (with the object counts).
     * This will also create the job bar widget that is used during discovery.
     *
     * @param {JQuery} dom
     * @param {Object} params
     * @returns {*|Promise}
     */
    target.doInitialize = function (dom, params) {
      var that = this;
      params = params || {};
      params = _.extend({
        html: exports.mgrLearnHtml()
      }, params);
      return superDoInitialize.call(that, dom, params).then(function () {
        return that.initializeJobBar(dom);
      });
    };

    /**
     * Initializes the JobBar widget.
     *
     * @param {JQuery} dom
     * @returns {Promise}
     */
    target.initializeJobBar = function (dom) {
      var jobBarEl = dom.find('.mgr-job-bar-container');
      if (!jobBarEl.length) {
        return Promise.resolve();
      }
      var jobBar = this.$jobBar = new JobBar();
      return jobBar.initialize(jobBarEl);
    };

    /**
     * Extends the base Manager type's doLayout() function to handle both the main
     * database table and the discovery table.
     *
     * @returns {Promise}
     */
    target.doLayout = function () {
      var that = this;

      // Call the super layout to set the bottom position of the database
      // table according to the action bar, then adjust the table heights.
      // We also adjust the position the split pane, but only if flagged to
      // do so.

      return Promise.resolve(superDoLayout.apply(that, arguments)).then(function () {
        if (that.$positionPanesOnLayout) {
          // This will resize the table heights when the positioning
          // is done.

          that.$positionSplitPane(isLearnModeCommandSelected(that));
        } else {
          return resizeTableHeights(that);
        }
      });
    };

    /**
     * Initialize the event handlers on the main manager model. This will be called after
     * the user interface has been constructed.
     *
     * @private
     * @param mgr
     */
    function initializeMainModelEvents(mgr) {
      var model = mgr.getModel();
      mgr.$mainModelChanged = _.throttle(function () {
        if (!mgr.isDestroyed()) {
          disableIconsForExisting(mgr);
          updateDbObjectCount(mgr);
        }
      }, UPDATED_MODEL_THROTTLE_MS, {
        leading: false
      });
      model.on('rowsAdded', mgr.$mainModelChanged);
      model.on('rowsRemoved', mgr.$mainModelChanged);
    }

    /**
     * Initialize the event handlers on the learn model. This will be called after
     * the user interface has been constructed.
     *
     * @private
     * @param mgr
     */
    function initializeLearnModelEvents(mgr) {
      var learnModel = getLearnModel(mgr);
      mgr.$learnModelChanged = _.throttle(function () {
        if (!mgr.isDestroyed()) {
          disableIconsForExisting(mgr);
          updateDiscoveryObjectCount(mgr);
        }
      }, UPDATED_MODEL_THROTTLE_MS, {
        leading: false
      });
      learnModel.on('rowsAdded', mgr.$learnModelChanged);
      learnModel.on('rowsRemoved', mgr.$learnModelChanged);
    }

    /**
     * Extension of the manager's load functionality. As well as loading the main table's
     * model, this will also load the model for the learn table.
     *
     * @param {module:nmodule/webEditors/rc/wb/mgr/model/MgrModel} model
     * @returns {*}
     */
    target.doLoad = function (model) {
      var that = this,
        jq = that.jq();
      return Promise.resolve(that.makeLearnModel()).then(function (learnModel) {
        setLearnModel(that, learnModel);
        return superDoLoad.call(that, model);
      }).then(function () {
        return buildLearnTable(that);
      }).then(function () {
        initializeSplitPanes(that, jq);
        initializeDragAndDrop(that, jq);
        initializeMainModelEvents(that);
        initializeLearnModelEvents(that);
        that.$registerOnLearnTableEvents();
        updateDbObjectCount(that);
        updateDiscoveryObjectCount(that);
        that.$toggleLearnPane(isLearnModeCommandSelected(that));
        return disableIconsForExisting(that);
      });
    };

    /**
     * Create and initialize the learn table for the manager. This will hook into the
     * table construction to allow the appropriate CSS to be set on the table. This
     * will also configure the show/hide menu for the new table.
     *
     * @private
     * @param mgr
     */
    function buildLearnTable(mgr) {
      var jq = mgr.jq();
      var tableContainer = jq.find('.discoveryTable');
      var learnModel = mgr.getLearnModel();
      var initParams = {};
      initParams.buildCell = _.bind(target.buildDiscoveryTableCell, mgr);
      initParams.destroyCell = _.bind(target.destroyDiscoveryTableCell, mgr);
      initParams.finishBuildingRow = _.bind(target.finishDiscoveryTableRow, mgr);
      var $resolveHandlers = TableModel.$resolveHandlers;
      var $constructorParams;
      if ($resolveHandlers) {
        $constructorParams = {
          selection: new ListSelection({
            $resolveHandlers: $resolveHandlers
          })
        };
      }
      return fe.buildFor({
        dom: tableContainer,
        type: tableConstructor,
        value: learnModel,
        $constructorParams: $constructorParams,
        initializeParams: initParams,
        hooks: {
          instantiated: mgr.$learnTableInstantiated
        },
        properties: {
          enableStriping: false
        }
      }).then(function (table) {
        return mgr.armLearnTableContextMenu(table, jq, '.mgr-show-hide-menu-discovery');
      });
    }

    /**
     * Remove the event handlers fromm the learn model, and delete the properties
     * on the manager that relate to it.
     *
     * @private
     * @param mgr
     * @returns {Promise}
     */
    function disposeLearnModel(mgr) {
      var model = getLearnModel(mgr),
        promise;
      if (model) {
        model.removeAllListeners('columnsFlagsChanged');
        model.removeListener('rowsAdded', mgr.$learnModelChanged);
        model.removeListener('rowsRemoved', mgr.$learnModelChanged);
        promise = model.removeColumns(model.getColumns());
      } else {
        promise = Promise.resolve();
      }
      return promise.then(function () {
        if (model) {
          delete mgr.$learnModel;
        }
        delete mgr.$learnModelChanged;
      });
    }

    /**
     * Extend the manager's destroy functionality to clean up event handlers
     * relating to the discovery table and model.
     *
     * @returns {*|Promise}
     */
    target.doDestroy = function () {
      var that = this,
        jq = that.jq(),
        model = that.getModel();
      jq.find('.split-pane').off();
      that.$removeLearnTableListeners();
      if (model) {
        model.removeListener('rowsAdded', that.$mainModelChanged);
        model.removeListener('rowsRemoved', that.$mainModelChanged);
      }
      delete that.$mainModelChanged;
      return superDoDestroy.apply(that, arguments).then(function () {
        return disposeLearnModel(that);
      });
    };

    ////////////////////////////////////////////////////////////////
    // Drag and Drop
    ////////////////////////////////////////////////////////////////

    /**
     * Initialize the event handling for the drag and drop events. This will call
     * functions to handle dragging rows from the learn table to the database table.
     *
     * @private
     * @param mgr the manager
     * @param dom the jquery object for the manager
     */
    function initializeDragAndDrop(mgr, dom) {
      dom.on('dragstart', '.mgr-discovery-row', handleLearnTableDragStart);
      dom.on('drop', '.mainTable', function (e) {
        return handleDrop(e, mgr);
      });
      dom.on('dragover', '.mainTable', handleDragOverMainTable);
    }

    /**
     * Event handler called when starting a drag from an item in the discovery table.
     * This will place some data on the dataTransfer object to indicate the drag operation.
     * When this is dropped, we shall invoke the AddCommand.
     *
     * @private
     * @param e the drag event
     */
    function handleLearnTableDragStart(e) {
      var row = $(e.target).data('mgr-discovered-row'),
        dataTransfer = e.originalEvent.dataTransfer;

      // NOTE 1: For this to work in the Workbench Java FX browser, it appears that it needs
      // to set any data on the event's dataTransfer object synchronously, before the event
      // handler returns. If it gets set asynchronously via a Promise, the data doesn't show
      // up in the drop event.

      // NOTE 2: The code for adding the items to the station is in the add command, as
      // items can be added without dragging and dropping. This means that we don't
      // actually place the discovery data in the DataTransfer object, just an indication
      // that something is being dragged. The AddCommand will pick up the discovery items
      // from the current list selection.

      if (row) {
        dataTransfer.setData('Text', JSON.stringify({
          mime: 'niagara/strings',
          data: [JSON.stringify({
            kind: 'mgr-drag-op',
            source: 'mgr-learn'
          })]
        }));
      }
    }

    /**
     * Event handler called when the item is over the main table.
     *
     * @param e the drag over event
     * @returns {boolean} returns false to allow a drop on to the table.
     */
    function handleDragOverMainTable(e) {
      e.originalEvent.dataTransfer.dropEffect = 'copy';
      return false;
    }

    /**
     * Event handler called when an item is dropped on the main table.
     * Items can be dropped from the discovery table of this manager,
     * the palette or the NavTree (dropping a mounted component). 
     * When dragging from the discovery table, this will check that the source 
     * of the drag was one or more rows in the discovery table, and invoke the 
     * operations to edit and add those items as new rows in the main manager table.
     *
     * @private
     * @param e the drop event
     * @param mgr the manager instance
     */
    function handleDrop(e, mgr) {
      var dataTransfer = e.originalEvent.dataTransfer;
      fromClipboard(dataTransfer).then(function (envelope) {
        switch (envelope.getMimeType()) {
          case 'niagara/navnodes':
            return Promise.all([envelope.toJson(), envelope.toValues()]).then(function (_ref) {
              var _ref2 = _slicedToArray(_ref, 2),
                json = _ref2[0],
                values = _ref2[1];
              var names = _.map(json, 'name');
              return mgr.$doInsert(names, values);
            });
          case 'niagara/strings':
            return envelope.toJson().then(function (arr) {
              var data = JSON.parse(arr[0]);
              if (data && data.kind === 'mgr-drag-op' && data.source === 'mgr-learn') {
                return mgr.$getAddCommand().invoke();
              }
            });
        }
      })["catch"](feDialogs.error);
      return false;
    }

    ////////////////////////////////////////////////////////////////
    // Table Content
    ////////////////////////////////////////////////////////////////

    /**
     * Update the object count text on one of the manager panes.
     *
     * @param model the model requiring the count to be updated; either database or discovery.
     * @param jq the jquery object for the element to contain the row count.
     */
    function updateRowCount(model, jq) {
      jq.text(webEditorsLex.get({
        key: 'mgr.titlePane.objects',
        args: [model.getRows().length]
      }));
    }

    /**
     * Update the count of objects in the title portion of the main database table pane.
     */
    function updateDbObjectCount(mgr) {
      var model = mgr.getModel();
      updateRowCount(model, mgr.jq().find('.main-table-object-count'));
    }

    /**
     * Update the count of objects in the title portion of the discovery table pane.
     */
    function updateDiscoveryObjectCount(mgr) {
      var model = mgr.getLearnModel();
      updateRowCount(model, mgr.jq().find('.discovery-table-object-count'));
    }
    function setIconOnDiscoveryRow(element, mgr, batch) {
      var dom = $(element),
        row = dom.data('mgr-discovered-row'),
        node = row && row.getSubject(),
        promise;
      if (node && (!node.isGroup || !node.isGroup())) {
        promise = mgr.getExisting(node, batch).then(function (exists) {
          return !!exists;
        });
      } else {
        promise = Promise.resolve(false);
      }
      return promise.then(function (existing) {
        dom.find('.IconEditor').toggleClass('mgr-learn-icon-disabled', existing);
      });
    }

    /**
     * Look through the rows in the learn table, and for any that match a component
     * in the station, disable the icon to show the item is already added.
     */
    function disableIconsForExisting(mgr) {
      var table = mgr.$getLearnTableElement(),
        batch = new baja.comm.Batch(),
        promises = [];
      table.find('.mgr-discovery-row').each(function (index, element) {
        promises.push(setIconOnDiscoveryRow(element, mgr, batch));
      });
      logFine('disableIconsForExisting: committing [' + promises.length + '] batched promises');
      return batch.commit(Promise.all(promises));
    }

    ////////////////////////////////////////////////////////////////
    // Table Layout
    ////////////////////////////////////////////////////////////////

    /**
     * Get the height of the table 'pane' area - that is the overall view height
     * minus the height of the action bar at the bottom.
     *
     * @returns {Number}
     */
    function getPaneAreaHeight(mgr) {
      var jq = mgr.jq(),
        managerHeight = jq.closest('.Manager').first().parent().outerHeight(),
        actionBarHeight = jq.find('.mgr-action-bar').outerHeight(),
        paneTop = jq.find('.mgr-pane-container').position().top;
      return managerHeight - actionBarHeight - paneTop;
    }

    /**
     * Function called to resize the divs containing the tables when the pane divider
     * is moved or the window size is changed. This needs to take into account a) the
     * divider position, b) the height of the command button container at the bottom
     * and c) the overall height of the view.
     *
     * @param mgr the `Manager` instance.
     * @returns {Promise}
     */
    function resizeTableHeights(mgr) {
      var paneAreaHeight,
        mainTableHeight,
        discoveryTableHeight,
        jq = mgr.jq(),
        divider = jq.find('.mgr-split-pane-divider'),
        mainTableJq = jq.find('.mainTable'),
        discoveryTableJq = jq.find('.discoveryTable'),
        actionBar = jq.find('.mgr-action-bar'),
        tableContainer = jq.find('.mgr-pane-container');
      paneAreaHeight = getPaneAreaHeight(mgr);
      tableContainer.outerHeight(paneAreaHeight);
      mainTableHeight = actionBar.offset().top - mainTableJq.offset().top;
      mainTableJq.outerHeight(mainTableHeight);
      discoveryTableHeight = divider.offset().top - discoveryTableJq.offset().top;
      discoveryTableJq.outerHeight(discoveryTableHeight);
      var mainTable = mgr.getMainTable();
      var learnTable = mgr.getLearnTable();
      return Promise.all([mainTable && mainTable.layout(), learnTable && learnTable.layout()]);
    }

    /**
     * Returns the TreeTable widget, used to display the discovered items.
     *
     * @function module:nmodule/webEditors/rc/wb/mgr/MgrLearn#getLearnTable
     * @returns {module:nmodule/webEditors/rc/wb/table/tree/TreeTable} the discovery table
     * @since Niagara 4.6
     */
    target.getLearnTable = function () {
      // noinspection JSValidateTypes
      return Widget["in"](this.$getLearnTableElement());
    };

    /**
     * Returns the jquery element for the discovery table.
     *
     * @private
     * @function module:nmodule/webEditors/rc/wb/mgr/MgrLearn#$getLearnTableElement
     * @returns {JQuery}
     */
    target.$getLearnTableElement = function () {
      return this.jq().find('.discoveryTable');
    };

    /**
     * Returns the job bar widget, used when the setJob() function is called on the
     * manager.
     *
     * @private
     * @function module:nmodule/webEditors/rc/wb/mgr/MgrLearn#$getJobBar
     * @returns {module:nmodule/webEditors/rc/wb/job/JobBar}
     */
    target.$getJobBar = function () {
      return this.$jobBar;
    };

    /**
     * Set up the split pane between the discovery and database tables. This will
     * resize the divs containing the tables in response to the events received when
     * the split pane divider is moved.
     *
     * @private
     * @param {module:nmodule/webEditors/rc/wb/mgr/Manager} mgr
     * @param {JQuery} dom the jQuery object for the manager view's DOM.
     */
    function initializeSplitPanes(mgr, dom) {
      dom.find('.split-pane').each(function () {
        var that = this,
          pane = $(that);
        pane.splitPane();
        pane.on('dividerdragend', _.throttle(function (event) {
          if (event.target === that) {
            mgr.layout()["catch"](logError);
          }
        }, SPLIT_PANE_THROTTLE_MS));
      });
    }

    /**
     * Set the position of the split panes for the discovery and main tables. This will
     * be called in response to the `LearnModeCommand` being toggled, or the manager state
     * being restored following a load.
     *
     * @private
     * @function module:nmodule/webEditors/rc/wb/mgr/MgrLearn#$positionSplitPane
     * @param {Boolean} show - true if the discovery pane is to be shown.
     */
    target.$positionSplitPane = function (show) {
      var that = this,
        jq = that.jq(),
        splitPane,
        size;
      if (jq.height() > 0) {
        // Fix the split pane position, such that when the learn table is shown, the splitter
        // divides the top and bottom tables equally. The table divs will then be resized
        // to accommodate those new heights. If the learn table is being hidden, set the height
        // of the first component to 0.

        splitPane = jq.find('.split-pane.horizontal-percent');
        size = show ? getPaneAreaHeight(that) / 2 : 0;
        _.defer(function () {
          if (!that.isDestroyed()) {
            splitPane.splitPane('firstComponentSize', size);
            resizeTableHeights(that)["catch"](logError);
          }
        });
        that.$positionPanesOnLayout = false;
      } else {
        // If we are loading the view and the $toggleLearnPane() function was called in
        // response to restoring the manager state, we might have to wait for the manager
        // widget to be laid out before we can calculate the required position for the
        // split pane. In this case, we'll set a flag to tell the layout method to call
        // this function, so it can try again when the UI is ready. The flag will be cleared
        // in the code above when doLayout() is called when the height > 0.

        that.$positionPanesOnLayout = true;
      }
    };

    /**
     * Function to show or hide the learn pane of the manager view. This involves a couple
     * of steps: first it needs to tell the split pane to collapse the first component, and
     * then needs to resize the table containers according to the new heights.
     *
     * @private
     * @function module:nmodule/webEditors/rc/wb/mgr/MgrLearn#$toggleLearnPane
     * @param {Boolean} show boolean value used to show or hide the learn pane.
     */
    target.$toggleLearnPane = function (show) {
      var that = this,
        jq = that.jq();

      // Hide or show both the top pane and the div used for the divider.

      jq.find('.mgr-pane-top').toggle(show);
      jq.find('.mgr-split-pane-divider').toggle(show);
      that.$positionSplitPane(show);
      _.defer(function () {
        if (!that.isDestroyed()) {
          resizeTableHeights(that)["catch"](logError);
        }
      });
    };

    ////////////////////////////////////////////////////////////////
    // Cell/Row Construction & Destruction
    ////////////////////////////////////////////////////////////////

    /**
     * Return a promise to build a cell in the discovery table. This will by default
     * just delegate to the table.
     *
     * @private
     * @param {module:nmodule/webEditors/rc/wb/table/model/Column} column
     * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
     * @param {JQuery} dom
     * @function module:nmodule/webEditors/rc/wb/mgr/MgrLearn#buildDiscoveryTableCell
     * @returns {Promise}
     */
    target.buildDiscoveryTableCell = function (column, row, dom) {
      return TreeTable.prototype.buildCell.call(this, column, row, dom);
    };

    /**
     * Return a promise to destroy a cell in the discovery table. This will by default
     * just delegate to the table.
     *
     * @private
     * @param {module:nmodule/webEditors/rc/wb/table/model/Column} column
     * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
     * @param {JQuery} dom
     * @function module:nmodule/webEditors/rc/wb/mgr/MgrLearn#destroyDiscoveryTableCell
     * @returns {Promise}
     */
    target.destroyDiscoveryTableCell = function (column, row, dom) {
      return TreeTable.prototype.destroyCell.call(this, column, row, dom);
    };

    /**
     * Finish the discovery row before it is added to the table.
     * This is where the CSS classes and other attributes are set.
     * We need to indicate that the row should be draggable. It
     * will also set a class to indicate that the row is for the
     * discovery table, which could be useful in the drag and drop code.
     *
     * @private
     * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
     * @param {JQuery} dom
     * @function module:nmodule/webEditors/rc/wb/mgr/MgrLearn#finishDiscoveryTableRow
     * @returns {Promise}
     */
    target.finishDiscoveryTableRow = function (row, dom) {
      dom.addClass('mgr-row mgr-discovery-row').prop('draggable', 'true').data('mgr-discovered-row', row);
      return Promise.resolve(dom);
    };

    /**
     * Registers listeners on the table events for the learn\discovery table
     * @since Niagara 4.14
     * @private
     */
    target.$registerOnLearnTableEvents = function () {
      var learnTable = this.getLearnTable();
      var that = this;
      if (learnTable) {
        var learnSelection = learnTable.$getSelection();
        this.$learnSelectionChangedCallBack = function () {
          return that.$tableSelectionChanged();
        };
        this.$learnDblClickedCallBack = function (event) {
          return that.$learnTableDblClicked(event);
        };
        learnSelection.on('changed', this.$learnSelectionChangedCallBack);
        that.jq().on('dblclick', '.discoveryTable tr', this.$learnDblClickedCallBack);
      }
    };

    /**
     * Removes the listeners registered on the table events for the learn\discovery table
     * @since Niagara 4.14
     * @private
     */
    target.$removeLearnTableListeners = function () {
      var learnTable = this.getLearnTable();
      var that = this;
      if (learnTable) {
        var learnSelection = learnTable.$getSelection();
        learnSelection.removeListener('changed', this.$learnSelectionChangedCallBack);
        that.jq().off('dblclick', '.discoveryTable tr', this.$learnDblClickedCallBack);
      }
    };

    /**
     * Gathers the necessary information needed to be passed to the double click event handler, mainly
     * the rows that were double-clicked.  Will then call the onLearnTableDlbClicked function which should
     * be implemented, when needed, to handle the double click event, typically this will invoke a
     * command on the selected records.  Takes as input the event and subjects that were double-clicked.
     * @since Niagara 4.14
     * @private
     * @param {JQuery.event} event the double-click event
     * @returns {Boolean} returns false always to stop the event at this point
     */
    target.$learnTableDblClicked = function (event) {
      var subjects = this.getLearnSubject($(event.target));
      this.onLearnTableDblClicked(event, subjects)["catch"](feDialogs.error);
      return false; //stop the event at this point
    };

    /**
     * Standard implementation that invokes the AddCommand on a double click of the learn table
     * @param {JQuery.Event} event the double click event
     * @param {Array.<*>} subjects the selected subject of the table being clicked
     * @returns {Promise|*}
     */
    target.onLearnTableDblClicked = function (event, subjects) {
      if (onLearnTableDblClickedOverride) {
        return Promise.resolve(onLearnTableDblClickedOverride.apply(this, arguments));
      }
      var addCommand = this.$getAddCommand();
      if (addCommand) {
        return Promise.resolve(addCommand.invoke());
      }
      return Promise.resolve();
    };

    /**
     * @param {JQuery} dom
     * @returns {Array|null} the selected subject of the table being clicked
     */
    target.getLearnSubject = function (dom) {
      var tableDom = dom.closest('.discoveryTable');
      if (!$.contains(this.jq()[0], tableDom[0])) {
        return null;
      }
      var table = Widget["in"](tableDom);
      if (!table) {
        return null;
      }
      return table.getSubject(dom);
    };

    /**
     * The default logic for onTableSelectionChanged for the MgrLearn has been implemented.
     * This will enable any `AddCommand`s if one of more rows were selected in the learn table.
     * And will enable the `MatchCommand` when one and only one row has been selected in each
     * table
     * @since Niagara 4.14
     * @param {Object} selectedSubjects an object that holds the array of the selected subjects from the tables
     * supported by the manager view.
     * @param {Array.<*>|undefined} selectedSubjects.mainTableSelection the current selected subjects in the main table
     * @param {Array.<*>|undefined} selectedSubjects.learnTableSelection the current selected subjects in the learn
     * or discovery table
     * @returns {Promise|*}
     */
    target.$defaultTableSelectionChanged = function (selectedSubjects) {
      var mainTableSelection = selectedSubjects.mainTableSelection,
        learnTableSelection = selectedSubjects.learnTableSelection;
      var that = this;
      if (that.$getAddCommands) {
        var addCmds = that.$getAddCommands();
        addCmds.forEach(function (addCmd) {
          addCmd.setEnabled(learnTableSelection.length > 0);
        });
      }
      var matchCmd = typeof that.$getMatchCommand === 'function' && that.$getMatchCommand(),
        quickMatchCmd = typeof that.$getQuickMatchCommand === 'function' && that.$getQuickMatchCommand();
      if ((matchCmd || quickMatchCmd) && mainTableSelection.length === 1 && learnTableSelection.length === 1) {
        return Promise.resolve(that.isMatchable(learnTableSelection[0], mainTableSelection[0])).then(function (matchable) {
          matchCmd && matchCmd.setEnabled(matchable);
          quickMatchCmd && quickMatchCmd.setEnabled(matchable);
        });
      } else {
        matchCmd && matchCmd.setEnabled(false);
        quickMatchCmd && quickMatchCmd.setEnabled(false);
      }
      return Promise.resolve();
    };
  };
  exports.mgrLearnHtml = function () {
    return tplManager({
      databaseTitle: webEditorsLex.get('mgr.titlePane.database'),
      discoveryTitle: webEditorsLex.get('mgr.titlePane.discovered')
    });
  };
  return exports;
});
