/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/mgr/BatchComponentEditor
 */
define(['baja!', 'log!nmodule.webEditors.rc.wb.mgr.BatchComponentEditor', 'bajaux/events', 'bajaux/Widget', 'Promise', 'underscore', 'nmodule/js/rc/switchboard/switchboard', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/fe/baja/util/ComplexDiff', 'nmodule/webEditors/rc/fe/baja/util/compUtils', 'nmodule/webEditors/rc/util/ListSelection', 'nmodule/webEditors/rc/wb/mgr/MgrSheet', 'nmodule/webEditors/rc/wb/mgr/mgrUtils', 'nmodule/webEditors/rc/wb/mgr/model/MgrColumn', 'nmodule/webEditors/rc/wb/mgr/model/MgrModelCompositeBuilder', 'nmodule/webEditors/rc/wb/mgr/model/columns/NameMgrColumn', 'nmodule/webEditors/rc/wb/mgr/model/columns/TypeMgrColumn', 'nmodule/webEditors/rc/wb/table/Table'], function (baja, log, events, Widget, Promise, _, switchboard, fe, BaseEditor, ComplexDiff, compUtils, ListSelection, MgrSheet, mgrUtils, MgrColumn, MgrModelCompositeBuilder, NameMgrColumn, TypeMgrColumn, Table) {
  'use strict';

  var newCopyComplete = compUtils.newCopyComplete;
  var MODIFY_EVENT = events.MODIFY_EVENT;
  var logError = log.severe.bind(log);
  var clearProposal = mgrUtils.clearProposal,
    clearProposals = mgrUtils.clearProposals,
    getProposedValue = mgrUtils.getProposedValue;

  /**
   * Editor intended for use with a `MgrModel`, allowing multiple components
   * to be edited simultaneously.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/mgr/BatchComponentEditor
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   */
  var BatchComponentEditor = function BatchComponentEditor() {
    var _this = this;
    BaseEditor.apply(this, arguments);
    var selection = this.$selection = new ListSelection();
    selection.select(0);
    selection.on('changed', function () {
      return _this.$onSelectionChanged()["catch"](logError);
    });
    switchboard(this, {
      '$onSelectionChanged': {
        allow: 'oneAtATime',
        onRepeat: 'preempt'
      }
    });
  };
  BatchComponentEditor.prototype = Object.create(BaseEditor.prototype);
  BatchComponentEditor.prototype.constructor = BatchComponentEditor;

  /**
   * Set up elements to show a component table and property sheet.
   *
   * Arm handlers to pick up on edits to the property sheet, and apply those
   * edits as proposed changes to the table model.
   *
   * Add `BatchComponentEditor` class.
   *
   * @param {JQuery} dom
   */
  BatchComponentEditor.prototype.doInitialize = function (dom) {
    var _this2 = this;
    dom.addClass('BatchComponentEditor').html("\n          <div class=\"editTable\">\n            <table class=\"ux-table no-stripe\"></table>\n          </div>\n          <div class=\"editSheet\">\n            <div></div>\n          </div>\n        ");
    dom.on(MODIFY_EVENT, '.editSheet', function (e, ed, modifiedEd) {
      _this2.$applySheetToModel(modifiedEd.jq().data('key'))["catch"](logError);
    });
  };

  /**
   * Load the component table and property sheet.
   *
   * @param {module:nmodule/webEditors/rc/wb/mgr/model/MgrModel} model
   * @returns {Promise}
   */
  BatchComponentEditor.prototype.doLoad = function (model) {
    return Promise.all([this.$buildSheet(model), this.$buildTable(model)]);
  };

  /**
   * Commit any proposed changes back to the underlying `MgrModel`.
   *
   * @returns {Promise}
   */
  BatchComponentEditor.prototype.doSave = function () {
    var sheet = this.$getPropertySheet(),
      builder = sheet.getBuilder(),
      model = this.value(),
      rows = model.getRows(),
      columns = model.getColumns();
    return Promise.resolve(builder.getKeys()).then(function (keys) {
      return Promise.all(keys.map(function (key) {
        var ed = builder.getEditorFor(key);
        return Promise.resolve(ed.shouldValidate() && ed.validate()).then(function () {
          var column = model.getColumn(key),
            data = rows.map(function (row) {
              return getProposedValue(row, key) || null;
            });
          return column.mgrValidate(model, data, {
            editor: ed
          });
        });
      }));
    }).then(function () {
      // we must batch together the column commits, waiting for the
      // COMMIT_READY callback for each one before we commit the batch.
      // this logic is almost identical to bajaux/mixin/batchSaveMixin.
      // With the addition of manager discovery support, this iteration
      // is now done by column name, rather than by keys from the builder,
      // as we want to commit rows with proposed values for non-editable
      // columns: these may have had their proposed values set from
      // adding a discovered object.

      var batch = new baja.comm.Batch(),
        results = _.flatten(columns.map(function (column) {
          //need this to run sync
          var key = column.getName();
          return _.compact(rows.map(function (row) {
            var data = getProposedValue(row, key);
            if (data !== undefined) {
              var ed = builder.getEditorFor(key);
              return batchCommit(column, data, row, ed, batch);
            }
          }));
        }), true);
      var savePromises = results.map(function (arr) {
        return arr[1];
      });
      var commitPromises = results.map(function (arr) {
        return arr[0];
      });
      var edSavePromises = results.map(function (arr) {
        return arr[2];
      });
      return Promise.all([Promise.all(commitPromises).then(function () {
        batch.commit();
      }), Promise.all(savePromises), Promise.all(edSavePromises)]);
    });
  };

  /**
   * Destroy child widgets.
   * @returns {Promise}
   */
  BatchComponentEditor.prototype.doDestroy = function () {
    var model = this.value();
    if (model) {
      var rows = model.getRows();
      rows.map(function (row) {
        clearProposals(row);
      });
    }
    this.jq().removeClass('BatchComponentEditor');
    return this.getChildWidgets().destroyAll();
  };
  BatchComponentEditor.prototype.doLayout = function () {
    return this.getChildWidgets().layoutAll();
  };

  /**
   * Take a field from the property sheet and apply the entered
   * values as proposed - not committed - changes to the table model.
   *
   * @private
   * @param {String} key the editor key
   * @returns {Promise} promise to be resolved when entered changes have
   * been applied as proposed changes to the table model
   */
  BatchComponentEditor.prototype.$applySheetToModel = function (key) {
    var _this3 = this;
    var builder = this.$getPropertySheet().getBuilder(),
      kid = builder.getEditorFor(key);
    if (!kid) {
      return Promise.resolve();
    }
    var selectedRows = builder.$getSelectedRows();
    var mgrModel = this.value();
    return kid.validate().then(function (value) {
      var column = mgrModel.getColumn(key),
        isArray = Array.isArray(value);
      return Promise.all(selectedRows.map(function (row, i) {
        return proposeNewCopy(_this3, isArray ? value[i] : value, row, column);
      }));
    }).then(function () {
      return mgrModel.emit('rowsChanged', selectedRows);
    });
  };

  /**
   * @private
   * @param {module:nmodule/webEditors/rc/wb/mgr/model/MgrModel} model
   * @returns {Promise}
   */
  BatchComponentEditor.prototype.$buildSheet = function (model) {
    var _this4 = this;
    var selection = this.$selection;
    return this.getOrdBase().then(function (ordBase) {
      return fe.buildFor({
        data: {
          builder: new MgrModelCompositeBuilder({
            model: model,
            selection: selection
          })
        },
        dom: _this4.$getPropertySheetElement(),
        value: model,
        properties: {
          allowHyperlink: false,
          nested: true,
          //NCCB-23224: Prevent PropertySheetSave on Enter Key
          ordBase: ordBase,
          showControls: false,
          showHeader: false
        },
        type: MgrSheet
      });
    });
  };

  /**
   * @private
   * @param {module:nmodule/webEditors/rc/wb/mgr/model/MgrModel} model
   * @returns {Promise}
   */
  BatchComponentEditor.prototype.$buildTable = function (model) {
    var selection = this.$selection;
    selection.put(_.range(0, model.getRows().length), true);
    return fe.buildFor({
      selection: selection,
      dom: this.$getTableElement(),
      properties: {
        hideUnseenColumns: false
      },
      value: model,
      type: Table
    });
  };

  /**
   * Update the property sheet with values coalesced from the selected rows.
   *
   * @private
   * @returns {Promise}
   */
  BatchComponentEditor.prototype.$onSelectionChanged = function () {
    return this.$getPropertySheet().getBuilder().buildAll();
  };

  /**
   * Get the element in which the component table is shown.
   *
   * @private
   * @returns {JQuery}
   */
  BatchComponentEditor.prototype.$getTableElement = function () {
    return this.jq().children('.editTable').children('table');
  };

  /**
   * Get the element in which the property sheet is shown.
   *
   * @private
   * @returns {JQuery}
   */
  BatchComponentEditor.prototype.$getPropertySheetElement = function () {
    return this.jq().children('.editSheet').children('div');
  };

  /**
   * Get the component table showing all the components to be edited.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/wb/table/Table}
   */
  BatchComponentEditor.prototype.$getTable = function () {
    return Widget["in"](this.$getTableElement());
  };

  /**
   * Get the property sheet allowing individual values to be edited.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/wb/PropertySheet}
   */
  BatchComponentEditor.prototype.$getPropertySheet = function () {
    return Widget["in"](this.$getPropertySheetElement());
  };
  function batchCommit(column, data, row, ed, batch) {
    var params = {
      editor: ed,
      batch: batch
    };

    // eslint-disable-next-line promise/avoid-new
    var readyToCommit = new Promise(function (resolve) {
      params.progressCallback = function (msg) {
        if (msg === MgrColumn.COMMIT_READY) {
          resolve();
        }
      };
    });

    //If an editor is supplied will call the editor save function, done as part of NCCB-53988
    //@since Niagara 4.14
    return [readyToCommit, column.commit(data, row, params), Promise.resolve(ed && ed.save(params)["catch"](logError))];
  }
  function newCopy(value, row, column) {
    if (value instanceof ComplexDiff) {
      var copy = newCopyComplete(column.getValueFor(row));
      return value.apply(copy);
    }
    return newCopyComplete(value);
  }
  function proposeNewCopy(ed, value, row, column) {
    var mgrProposed = newCopyComplete(getProposedValue(row, 'mgr.batch.proposed.' + column.getName()));
    return Promise.resolve(newCopy(value, row, column)).then(function (copy) {
      if (mgrProposed !== undefined) {
        copy = mgrProposed;
        clearProposal(row, 'mgr.batch.proposed.' + column.getName());
      }
      return column.propose(copy, row);
    }).then(function () {
      if (column instanceof NameMgrColumn) {
        mgrUtils.setHasUserDefinedNameValue(row, true);
      } else if (column instanceof TypeMgrColumn) {
        // If the row's type has changed, force an update of the property sheet
        ed.$onSelectionChanged()["catch"](logError);
      }
    });
  }
  return BatchComponentEditor;
});
