/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Tony Richards
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/links/SlotChooser
 */
define(['baja!', 'dialogs', 'jquery', 'underscore', 'Promise', 'nmodule/webEditors/rc/fe/BaseWidget', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/baja/IconEditor', 'nmodule/webEditors/rc/fe/baja/util/slotUtils', 'hbs!nmodule/webEditors/rc/wb/links/template/SlotChooser-row'], function (baja, dialogs, $, _, Promise, BaseWidget, fe, IconEditor, slotUtils, tplSlotChooserRow) {
  'use strict';

  var toCssClass = slotUtils.toCssClass,
    getSlotIcon = slotUtils.getSlotIcon;

  ////////////////////////////////////////////////////////////////
  // SlotChooser definition
  ////////////////////////////////////////////////////////////////

  /**
   * A callback used to determine what icon to display for a given slot.
   *
   * @callback module:nmodule/webEditors/rc/wb/links/SlotChooser~slotIcon
   *
   * @param {Object} params provides details from the SlotChooser to help determine
   * the displayed image.
   * @param {baja.Slot} params.slot the slot in question
   * @param {Boolean} params.isSelected true if the slot is selected, else false
   * @returns {baja.Icon|baja.Value|baja.Slot|Array.<String>} what to display
   */

  /**
   * A panel that displays slots of a component and allows the user
   * to choose one (or more) of the slots.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/BaseWidget
   * @alias module:nmodule/webEditors/rc/wb/links/SlotChooser
   * @param {Object} [params] - See BaseWidget params
   * @param {Object} [params.properties]
   * @param {Boolean} [params.properties.multiSelect=false] specifies whether or
   * not the SlotChooser should allow the user to select multiple slots.
   * @param {string} [params.properties.selectedSlot]
   * This property will let the SlotChooser know to pre-select a component's slot upon its doLoad.
   * Either source-component's slot or target-component's slot will be pre-selected, both cannot be pre-selected at a point of time by the SlotChooser.
   *
   * @example
   * Slot in16 will be selected upon load by the SlotChooser
   * selectedSlot: 'in16'
   *
   * @param {module:nmodule/webEditors/rc/wb/links/SlotChooser~slotIcon} [params.properties.slotIcon]
   * callback used to determine what the displayed icon should be. If not
   * specified, the editor will provide a default icon for each slot.
   */
  var SlotChooser = function SlotChooser(params) {
    BaseWidget.apply(this, arguments);
    this.$filter = function () {
      return true;
    };
    this.$selected = [];
    this.$selectableSlots = [];
  };
  SlotChooser.prototype = Object.create(BaseWidget.prototype);
  SlotChooser.constructor = SlotChooser;

  /**
   * @param {JQuery} dom
   */
  SlotChooser.prototype.doInitialize = function (dom) {
    var that = this;
    dom.addClass('SlotChooser');
    dom.on('click', '.SlotChooserRow-enabled', function (event) {
      event.preventDefault();
      var selectedSlots = that.getSelectedSlots();
      var domSlot = $(this).data('slot');
      var slotIndex = selectedSlots.indexOf(domSlot),
        isMultiSelect = that.properties().getValue("multiSelect");
      if (slotIndex > -1) {
        selectedSlots.splice(slotIndex, 1);
      } else if (isMultiSelect) {
        selectedSlots.push(domSlot);
      } else {
        selectedSlots = [domSlot];
      }
      that.setSelectedSlots(selectedSlots, true);
    });
    dom.on('click', '.SlotChooserRow-disabled', function (event) {
      event.preventDefault();
      var reason = $(this).attr('title');
      if (reason) {
        dialogs.showOk(reason);
      }
    });
    return BaseWidget.prototype.doInitialize.apply(that, arguments);
  };

  /**
   * Loads the slots for the specified components.  If more than one component is specified,
   *  the intersecting slots are loaded.
   *
   * @param {Array<baja.Component>} components an array of `baja:Component`
   * @returns {Promise|null} promise to be resolved with the requested components
   *  have been loaded
   */
  SlotChooser.prototype.doLoad = function (components) {
    var _this = this;
    var that = this;

    // These need to be leased at some point... but here / now?
    return baja.Component.lease(components).then(function () {
      that.$components = components;
      that.$slots = that.$getIntersectingSlots();
      return that.$loadSlots();
    }).then(function () {
      var selectedSlot = _this.properties().getValue('selectedSlot');
      if (selectedSlot) {
        that.setSelectedSlot(that.$components[0].getSlot(selectedSlot), true);
      }
    });
  };
  SlotChooser.prototype.doDestroy = function () {
    this.jq().removeClass('SlotChooser');
  };

  /**
   * Scrolls widget to the selected slot.
   */
  SlotChooser.prototype.scrollToSelectedSlot = function () {
    var selectedSlot = this.getSelectedSlot();
    selectedSlot && this.$scrollIntoView(".".concat(toCssClass(selectedSlot)));
  };

  /**
   * @private
   * @param {string} slotClass
   */
  SlotChooser.prototype.$scrollIntoView = function (slotClass) {
    this.jq().find(slotClass)[0].scrollIntoView({
      behavior: 'smooth',
      block: 'nearest',
      inline: 'nearest'
    });
  };

  /**
   * Sets the enable filter function to enable / disable slot selection.
   *
   * @param {function} filter function that is called once for each slot
   *  while (re)loading the slots into the DOM.  This filter function should
   *  return [true, ?, Promise.<LinkCheck>] to enable the slot for choosing, or
   *  [false, reason, Promise.<LinkCheck>] to disable the slot.
   *  reason is a string that is the reason the slot is not enabled
   *  Promise.<LinkCheck> (optional) is a Promise that resolve to a secondary LinkCheck.
   *  The first set of values in this return is "best guess" client-side link checking,
   *  whereas the LinkCheck third argument resolves to a definitive answer from the server.
   *  If the third element is null or missing then the first element Boolean is a definitive
   *  answer.
   */
  SlotChooser.prototype.setEnableFilter = function (filter) {
    this.$filter = filter;
    return this.$loadSlots();
  };

  /**
   * Reads the currently selected slots and turns them into an array of ords.
   *
   * @returns {Array.<String>} the currently selected slots. The
   * array will be empty if there are no selected slots.
   */
  SlotChooser.prototype.doRead = function () {
    var selectedSlots = this.getSelectedSlots();
    return selectedSlots.map(function (slot) {
      return slot.toString();
    });
  };

  /**
   * Set which slots are currently selected and optionally fire the
   * selectionChanged event.
   *
   * @param {Array.<baja.Slot>} selected - newly selected slot
   * @param {boolean} [fireEvents=false] Optional true if the selectionChanged event
   * should be fired
   */
  SlotChooser.prototype.setSelectedSlots = function (selected, fireEvents) {
    var wasSelected = this.$selected;
    this.$selected = selected;
    if (fireEvents) {
      // Fire an event so the container can respond to the change.
      this.emit('selectionChanged', selected, wasSelected, this);

      // Mark the widget modified
      this.setModified(true);
    }
  };

  /**
   * Set which slot is currently selected and optionally fire the
   * selectionChanged event.
   *
   * @param {baja.Slot} selected - newly selected slot
   * @param {boolean} [fireEvents=false] Optional true if the selectionChanged event
   * should be fired
   */
  SlotChooser.prototype.setSelectedSlot = function (selected, fireEvents) {
    this.setSelectedSlots([selected], fireEvents);
  };

  /**
   * Gets the slots that are available for selection in the SlotChooser.
   *
   * @returns {Array.<baja.Slot>} the slots displayed in the SlotChooser
   */
  SlotChooser.prototype.getSelectableSlots = function () {
    return this.$selectableSlots;
  };
  SlotChooser.prototype.$loadSlots = function () {
    var that = this,
      batch = new baja.comm.Batch(),
      slots = that.$slots.filter(that.$isValidSlotRow.bind(that)),
      rowPromises = slots.map(function (slot) {
        return that.$buildRowDom(slot, batch);
      });
    batch.commit();
    that.$selectableSlots = slots;
    if (rowPromises.length === 0) {
      that.jq().html([]);
      return Promise.resolve();
    }
    var unzipped = _.unzip(rowPromises),
      clientPromises = unzipped[0],
      serverPromises = unzipped[1];
    return Promise.all(clientPromises).then(function (rows) {
      that.jq().html(rows);
    }).then(function () {
      return Promise.all(serverPromises).then(function (rows) {
        that.jq().html(rows);
      });
    });
  };

  /**
   * Get the intersecting slots from all of the components.
   *
   * @private
   * @returns {Array} array of baja:Slot
   */
  SlotChooser.prototype.$getIntersectingSlots = function () {
    // TODO Implement; current implementation returns the slots from
    // the first component
    return this.$components[0].getSlots().toArray();
  };

  /**
   * @returns {Array.<baja.Slot>} The currently selected slot, or empty array if
   * no slot has been selected
   */
  SlotChooser.prototype.getSelectedSlots = function () {
    return this.$selected || [];
  };

  /**
   * @returns {baja.Slot} The currently selected slot, or null if no slot
   * has been selected.
   */
  SlotChooser.prototype.getSelectedSlot = function () {
    var selectedSlots = this.getSelectedSlots();
    return selectedSlots.length ? selectedSlots[0] : null;
  };

  /**
   * Creates a table row to hold an instance of `SlotChooserRow`.
   *
   * @private
   * @param {Slot} slot - Slot for which to build a row DOM.
   * @param {Batch} batch object for batching the server side link checks.
   *
   * @returns {Array.<Promise.<JQuery>>} the table row into which to load the field editor.
   * The first entry in the array resolves to the row dom based on the client link check,
   * and the second entry is based on the server (definitive) link check.
   */
  SlotChooser.prototype.$buildRowDom = function (slot, batch) {
    var slotIcon = this.properties().getValue("slotIcon");
    var that = this,
      buildRow = function buildRow(slot, enabled, reason, server) {
        var selected;
        if (that.getSelectedSlots().includes(slot)) {
          // If the slot was previously selected but now it's no longer enabled
          // then mark it so that it's not selected. (but only do this if "enabled"
          // is coming from a server checkLink.
          if (!enabled && server) {
            that.setSelectedSlots([], true);
          } else {
            selected = true;
          }
        }
        var name = that.$components[0].getDisplayName(slot),
          row = $(tplSlotChooserRow({
            slotClass: toCssClass(slot),
            slotType: getSlotType(slot),
            name: that.$components[0].getDisplayName(slot),
            enabled: enabled,
            selected: selected,
            tooltip: reason || name
          }));

        // Attach the slot to the row data to make it handy for referencing later
        row.data('slot', slot);
        var iconParams = {
          isSelected: that.getSelectedSlots().includes(slot),
          slot: slot
        };
        return fe.buildFor({
          dom: row.children('.SlotChooserRow-icon'),
          value: slotIcon ? slotIcon(iconParams) : getSlotIcon(slot),
          type: IconEditor
        }).then(function () {
          return row;
        });
      },
      clientPromise,
      // eslint-disable-next-line promise/avoid-new
      serverPromise = new Promise(function (resolve, reject) {
        // Construct the HTML, specifying the slot style, name, icon
        clientPromise = Promise.resolve(that.$filter(slot, batch)).then(function (filterResponse) {
          var enabled, reason;
          if (Array.isArray(filterResponse)) {
            reason = filterResponse.length > 1 ? filterResponse[1] : '';
            enabled = filterResponse[0];
            if (filterResponse.length > 2 && filterResponse[2]) {
              resolve(filterResponse[2].then(function (checkLink) {
                return buildRow(slot, checkLink.isValid(), checkLink.getInvalidReason(), true);
              }));
            } else {
              resolve(buildRow(slot, enabled, reason));
            }
          } else {
            enabled = filterResponse;
            resolve(buildRow(slot, enabled, reason));
          }

          //TODO buildRow gets called twice - why?
          return buildRow(slot, enabled, reason);
        });
      });
    return [clientPromise, serverPromise];
  };

  /**
   * Checks to see if the specified slot is a valid slot row
   *
   * @private
   * @param {baja.Slot} slot to determine if it should be displayed in this chooser
   * @returns {boolean} true if the slot is valid, otherwise false
   */
  SlotChooser.prototype.$isValidSlotRow = function (slot) {
    var that = this;

    // Not a valid slot if it's hidden
    if (slot.getFlags() & baja.Flags.HIDDEN) {
      return false;
    }
    if (slot.isProperty()) {
      // Not valid if it's a Component (unless it's a Vector), Link, or WsAnnotation
      var v = that.$components[0].get(slot);
      if (baja.hasType(v, 'baja:Component') && !baja.hasType(v, 'baja:Vector')) {
        return false;
      }
      // TODO If a Link is not valid, why would a Relation be valid?
      if (baja.hasType(v, 'baja:Link')) {
        return false;
      }
      if (baja.hasType(v, 'baja:WsAnnotation')) {
        return false;
      }
      return true;
    } else {
      return true;
    }
  };

  /**
   * Get slot type for styling purposes - action, topic, relation, or property.
   * @param {baja.Slot} slot
   * @returns {string}
   */
  function getSlotType(slot) {
    if (slot.isAction()) {
      return 'action';
    }
    if (slot.isTopic()) {
      return 'topic';
    }
    if (slot.getType().is('baja:Relation')) {
      return 'relation';
    }
    return 'property';
  }
  return SlotChooser;
});
