function _slicedToArray(r, e) { return _arrayWithHoles(r) || _iterableToArrayLimit(r, e) || _unsupportedIterableToArray(r, e) || _nonIterableRest(); }
function _nonIterableRest() { throw new TypeError("Invalid attempt to destructure non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method."); }
function _unsupportedIterableToArray(r, a) { if (r) { if ("string" == typeof r) return _arrayLikeToArray(r, a); var t = {}.toString.call(r).slice(8, -1); return "Object" === t && r.constructor && (t = r.constructor.name), "Map" === t || "Set" === t ? Array.from(r) : "Arguments" === t || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(t) ? _arrayLikeToArray(r, a) : void 0; } }
function _arrayLikeToArray(r, a) { (null == a || a > r.length) && (a = r.length); for (var e = 0, n = Array(a); e < a; e++) n[e] = r[e]; return n; }
function _iterableToArrayLimit(r, l) { var t = null == r ? null : "undefined" != typeof Symbol && r[Symbol.iterator] || r["@@iterator"]; if (null != t) { var e, n, i, u, a = [], f = !0, o = !1; try { if (i = (t = t.call(r)).next, 0 === l) { if (Object(t) !== t) return; f = !1; } else for (; !(f = (e = i.call(t)).done) && (a.push(e.value), a.length !== l); f = !0); } catch (r) { o = !0, n = r; } finally { try { if (!f && null != t["return"] && (u = t["return"](), Object(u) !== u)) return; } finally { if (o) throw n; } } return a; } }
function _arrayWithHoles(r) { if (Array.isArray(r)) return r; }
/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/*eslint-env browser */ /*jshint browser: true */

define(['baja!', 'baja!baja:JobState', 'lex!baja,webEditors', 'jquery', 'underscore', 'Promise', 'dialogs', 'nmodule/webEditors/rc/wb/table/model/Column', 'nmodule/webEditors/rc/wb/table/model/Row', 'nmodule/webEditors/rc/wb/table/model/TableModel'], function (baja, types, lexs, $, _, Promise, dialogs, Column, Row, TableModel) {
  "use strict";

  /**
   * API Status: **Private**
   *
   * Helper functions relating to jobs, including functions for reading a job log and processing
   * the items contained within. This provides common functionality used by the job bar and the
   * live job dialog.
   *
   * @exports nmodule/webEditors/rc/wb/job/jobSupport
   */
  var exports = {};
  var bajaLex = lexs[0],
    webEditorsLex = lexs[1],
    JOB_STATE_ENUM = baja.$('baja:JobState'),
    MESSAGE_ICON_ORD = 'module://icons/x16/info.png',
    STATE_ICON_ORDS = {
      unknown: 'module://icons/x16/job/unknown.png',
      running: 'module://icons/x16/job/running.png',
      canceled: 'module://icons/x16/job/failed.png',
      canceling: 'module://icons/x16/job/failed.png',
      success: 'module://icons/x16/job/success.png',
      failed: 'module://icons/x16/job/failed.png'
    },
    TABLE_TIME_FORMAT_ARGS = {
      show: baja.TimeFormat.SHOW_DATE | baja.TimeFormat.SHOW_TIME,
      lex: bajaLex
    },
    DETAIL_TIME_FORMAT_ARGS = {
      show: baja.TimeFormat.SHOW_DATE | baja.TimeFormat.SHOW_TIME | baja.TimeFormat.SHOW_SECONDS,
      lex: bajaLex
    };

  /**
   * Utility function used to filter out empty lines when parsing the job log.
   */
  function notEmpty(s) {
    return !!s;
  }

  /**
   * Return a baja:JobState enum from the status id string in the
   * log item, or null if there was no associated state and the log entry
   * was a message.
   */
  function jobStateForLogItem(item) {
    var tag = item.id.toLowerCase();
    return JOB_STATE_ENUM.getRange().isTag(tag) ? JOB_STATE_ENUM.get(tag) : null;
  }

  /**
   * Decode a single job log item from a string produced by the Java method
   * JobLogItem.encode(). Fields are separated by the '|' character, with '\'
   * as the escape character. Returns an array of the field values obtained
   * from the string. This does not handle the formatting of the messages
   * obtained from the logs, that will be done later.
   */
  function splitFields(encoded) {
    var buf = '',
      escaped = false,
      decoded = [];
    _.each(encoded, function (ch) {
      if (escaped) {
        buf += ch === 'n' ? '\n' : ch;
        escaped = false;
      } else if (ch === '\\') {
        escaped = true;
      } else if (ch === '|') {
        decoded.push(buf);
        buf = '';
      } else {
        buf += ch;
      }
    });
    decoded.push(buf);
    while (decoded.length < 4) {
      decoded.push('');
    }
    return decoded;
  }

  /**
   * Convert an array of parsed fields values into an object with those field values
   * as its properties. This will return a Promise that will resolve to an object
   * representing the log item, as it may involve asynchronous lexicon requests
   * due to the message format string.
   *
   * @param {Array} fields An array of parsed job log item fields.
   * @returns {Promise}
   */
  function toObjects(fields) {
    var id = fields[0].toUpperCase(),
      timestamp = baja.AbsTime.DEFAULT.decodeFromString(fields[1]),
      msgPattern = fields[2],
      detailsPattern = fields[3];
    return Promise.all([baja.Format.format(msgPattern), baja.Format.format(detailsPattern)]).then(function (_ref) {
      var _ref2 = _slicedToArray(_ref, 2),
        message = _ref2[0],
        details = _ref2[1];
      return {
        'id': id,
        'timestamp': timestamp,
        'message': message,
        'details': details
      };
    });
  }

  /**
   * Takes an encoded job log string obtained from a BJob's 'readLog' action, and
   * converts it to an array of objects representing the items in the log. Each object
   * will have properties for the id, timestamp, message and a property
   * for details, which may be empty.
   */
  function parseJobLog(encoded) {
    var fields = _.chain(encoded.split('\n')).filter(notEmpty).map(splitFields).value();
    return Promise.all(fields.map(toObjects));
  }

  /**
   * Truncate a string to a maximum length, replacing the removed portion of the
   * string with an ellipsis character.
   */
  function truncateString(str, len) {
    if (str && str.length > len) {
      str = str.substring(0, len - 1) + '…';
    }
    return str;
  }

  /**
   * Return the first line of a multiline string, to be used as a summary
   * in the table.
   */
  function firstLine(str) {
    return str ? _.head(str.split(/[\r\n]/g)) : '';
  }

  /**
   * Return a string to be used in a table cell, taking just the first line
   * of a multi-line string and optionally truncating it.
   */
  function getStringForTableCell(str, truncateLength) {
    str = firstLine(str);
    if (truncateLength) {
      str = truncateString(str, truncateLength);
    }
    return str;
  }

  ////////////////////////////////////////////////////////////////
  // LogFieldColumn
  ////////////////////////////////////////////////////////////////

  /**
   * Internal table column type, used to display a field from a JobLogItem.
   *
   * @private
   * @constructor
   */
  var LogFieldColumn = function LogFieldColumn(name, lexKey, truncateLength) {
    Column.call(this, name, {
      displayName: webEditorsLex.get(lexKey || name)
    });
    this.$truncateLength = truncateLength;
  };
  LogFieldColumn.prototype = Object.create(Column.prototype);
  LogFieldColumn.prototype.constructor = LogFieldColumn;

  /**
   * Get the value to be displayed in the column for the given row. This
   * will be an appropriately localized value.
   */
  LogFieldColumn.prototype.getValueFor = function (row) {
    var name = this.getName(),
      subject = row.getSubject();
    if (name === 'id') {
      return bajaLex.get(subject.id.toLowerCase());
    } else if (name === 'timestamp') {
      return subject.timestamp.toDateTimeStringSync(TABLE_TIME_FORMAT_ARGS);
    } else {
      // If we have a multi-line string, display the first line in the table.
      // To see the full content, the user can double-click the cell to get the
      // full details in a message box. The caller can also choose to truncate
      // the messages in the dialog to a maximum length.

      return getStringForTableCell(subject[name], this.$truncateLength);
    }
  };

  /**
   * Set the length at which to truncate strings displayed in the the main table.
   * @private
   * @param length
   */
  LogFieldColumn.prototype.$setTruncateLength = function (length) {
    this.$truncateLength = length;
  };

  /**
   * Add the status icon on a cell in the id column.
   */
  LogFieldColumn.prototype.buildCell = function (row, dom) {
    var that = this;
    return Column.prototype.buildCell.apply(that, arguments).then(function () {
      var img;
      if (that.getName() === 'id') {
        img = $('<img>');
        img.addClass('job-log-item-icon');
        img.attr('src', _.first(row.getIcon().getImageUris()));
        dom.prepend(img);
      }
    });
  };

  ////////////////////////////////////////////////////////////////
  // Exports
  ////////////////////////////////////////////////////////////////

  /**
   * Return a promise to read the log for the given job, parse it, and resolve to an array
   * of objects representing the items in the log.
   *
   * @param {baja.Component} job a BJob component
   * @returns {Promise} a `Promise` that will resolve to an array of parsed log items.
   */
  exports.readLogItems = function (job) {
    return job.readLog().then(function (encodedLog) {
      return parseJobLog(encodedLog);
    });
  };

  /**
   * Parse a string of encoded JobLogItems encoded by the Java JobLogItem.encode() method,
   * delimited by new line characters. Returns a `Promise` that will resolve to an array
   * of objects containing the decoded items.
   *
   * @param {String} encoded - a new line delimited string of encoded JobLog items.
   * @returns {Promise}
   */
  exports.parseEncodedLog = function (encoded) {
    return parseJobLog(encoded);
  };

  /**
   * Parse a string encoded by the Java JobLogItem.encode() method, and return a
   * `Promise` that will resolve to an object containing the parsed properties.
   *
   * @param {String} encoded - a single encoded job log item.
   * @returns {Promise}
   */
  exports.parseEncodedLogItem = function (encoded) {
    return toObjects(splitFields(encoded));
  };

  /**
   * Make a `TableModel` instance for displaying the job log.
   *
   * @param {Array.<module:nmodule/webEditors/rc/wb/table/model/Row>} rows an
   * array of `Row`s to be used as the initial contents of the table.
   * @param {Object} [params] optional object literal containing parameters for configuring
   * the model.
   * @param {Boolean} [params.sortable] optional boolean indicating whether the table should
   * allow sorting of the columns. This might be useful in cases where the log entries are needed
   * to be preserved in timestamp order.
   * @param {Number} [params.truncateLength] optional number allowing the text in the table
   * to be truncated to a maximum length.
   *
   * @returns {TableModel}
   */
  exports.makeLogModel = function (rows, params) {
    var columns = [new LogFieldColumn('id', 'status'), new LogFieldColumn('timestamp'), new LogFieldColumn('message'), new LogFieldColumn('details')];
    rows = rows || [];
    params = _.defaults(params || {}, {
      sortable: true
    });
    _.each(columns, function (col) {
      col.setSortable(params.sortable);
      if (params.truncateLength) {
        col.$setTruncateLength(params.truncateLength);
      }
    });
    return new TableModel({
      rows: rows,
      columns: columns
    });
  };

  /**
   * Return a `Row` instance for the given log entry, with an appropriate icon for the
   * status field of the item.
   *
   * @param {nmodule/webEditors/rc/wb/job/jobSupport~jobLogItem} item an object representing an item parsed from a job's log.
   * @returns {module:nmodule/webEditors/rc/wb/table/model/Row} row
   */
  exports.makeLogTableRow = function (item) {
    var state = jobStateForLogItem(item),
      icon = baja.Icon.make(state ? exports.stateToIconOrd(state) : MESSAGE_ICON_ORD);
    return new Row(item, icon);
  };

  /**
   * Show the details dialog for a single item in the log. If the text had been
   * truncated in the main table, this can be used to see the full text, and also
   * display multi-line text.
   *
   * @param {nmodule/webEditors/rc/wb/job/jobSupport~jobLogItem} item an item parsed from the job log.
   */
  exports.showItemDetailsDialog = function (item) {
    var id = bajaLex.get(item.id.toLowerCase()),
      str = id + ' [' + item.timestamp.toDateTimeStringSync(DETAIL_TIME_FORMAT_ARGS) + '] ' + item.message;
    str = _.escape(str);
    if (item.details) {
      str = str + '\n\n' + _.escape(item.details);
    }

    // Show the details in an 'ok' dialog. The act of double clicking on an item
    // in the table will have caused the text underneath the mouse to become
    // selected, so we'll deselect the text, as that's not what was intended.

    window.getSelection().removeAllRanges();
    dialogs.showOk({
      title: webEditorsLex.get('details'),
      content: '<div class="job-details-dlg">' + str + '</div>'
    });
  };

  /**
   * Return an ord string for an icon to represent the given state.
   *
   * @param state a baja:JobState enum value.
   * @returns {String} the ord for an icon corresponding to the state argument.
   */
  exports.stateToIconOrd = function (state) {
    return STATE_ICON_ORDS[state.getTag()];
  };
  return exports;

  /**
   * Represents a single log item
   * @typedef {object} nmodule/webEditors/rc/wb/job/jobSupport~jobLogItem
   * @property {string} id
   * @property {baja.AbsTime} timestamp
   * @property {string} message
   * @property {string} [details]
   */
});
