/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/commands/transfer/TransferCommand
 */
define(['bajaux/commands/Command', 'jquery', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/wb/mixin/mixinUtils'], function (Command, $, Promise, _, feDialogs, mixinUtils) {
  'use strict';

  var extend = _.extend,
    omit = _.omit,
    result = _.result;

  /**
   * Command to execute a transfer operation.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/commands/transfer/TransferCommand
   * @extends {module:bajaux/commands/Command}
   */
  var TransferCommand = function TransferCommand(owner, transferOp) {
    var params = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
    this.$owner = owner;
    this.$op = transferOp;
    this.$params = params;
    var displayName = params.displayName,
      lex = params.lex,
      module = params.module,
      subject = params.subject,
      enabled = params.enabled;
    var transferOpEnabled = subject ? owner.isTransferOpEnabled(transferOp, subject) : true;
    var supported = subject ? owner.isTransferOpSupported(transferOp, subject) : true;
    Command.call(this, {
      displayName: displayName,
      module: module,
      lex: lex,
      enabled: transferOpEnabled && supported && enabled !== false,
      /**
       * @name module:nmodule/webEditors/rc/wb/commands/transfer/TransferCommand#undoable
       * @function
       * @param {object} [params]
       * @param {Array} [params.subject] optionally specify a subject to perform
       * the transfer op on, regardless of what was passed to `make()`
       * @param {JQuery} [params.dom] optionally specify which DOM element
       * provides the subject, instead of passing it directly as `params.subject`
       * @returns {Promise}
       */
      undoable: function undoable() {
        var params = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
        return Promise.resolve(params.subject || subject || getSubject(owner, params.dom)).then(function (subject) {
          if (!subject) {
            throw new Error('subject required');
          }
          if (!owner.isTransferOpSupported(transferOp, subject)) {
            throw new Error(transferOp + ' op not supported');
          }
          if (owner.isTransferOpEnabled(transferOp, subject)) {
            return Promise["try"](function () {
              return owner.doTransferOp(transferOp, subject, omit(params, 'subject', 'dom'));
            }).then(function (undoable) {
              if (Command.isUndoable(undoable)) {
                return undoable;
              }
            });
          }
        });
      }
    });
  };
  TransferCommand.prototype = Object.create(Command.prototype);
  TransferCommand.prototype.constructor = TransferCommand;

  /**
   * Create an instance of `TransferCommand`. The resolved instance, when
   * invoked, will perform the transfer operation on the owner's subject.
   *
   * @param {*} owner - must have `TransferSupport` mixin
   * @param {string} transferOp
   * @param {object} [params]
   * @param {string} [params.displayName]
   * @param {string} [params.module]
   * @param {string} [params.lex]
   * @param {Array} [params.subject] specified subject - otherwise will be
   * derived from the owner using `getSubject()`
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/commands/transfer/TransferCommand>}
   * @see module:nmodule/webEditors/rc/wb/mixin/TransferSupport
   */
  TransferCommand.make = function (owner, transferOp, params) {
    if (!owner || !mixinUtils.hasMixin(owner, 'transfer')) {
      return Promise.reject(new Error('owner with TransferSupport required'));
    }
    if (!transferOp) {
      return Promise.reject(new Error('transfer op required'));
    }
    return Promise.resolve(new TransferCommand(owner, transferOp, params || {}));
  };

  /**
   * Two `TransferCommands` will merge together if they are performing the same
   * transfer op on the subject. The subject on which the new `TransferCommand`
   * will be performed is the merged subject of both commands.
   *
   * @param {module:bajaux/commands/Command} cmd
   * @returns {module:nmodule/webEditors/rc/wb/commands/transfer/TransferCommand|null}
   */
  TransferCommand.prototype.merge = function (cmd) {
    var owner = this.$owner;
    var op = this.$op;
    var params = this.$params;
    if (!(cmd instanceof TransferCommand)) {
      return null;
    }
    if (cmd.$op !== op || cmd.$owner !== owner) {
      return null;
    }
    params.enabled = cmd.isEnabled() && this.isEnabled();
    var mySubject = params.subject || [];
    var hisSubject = cmd.$params.subject || [];
    return new TransferCommand(owner, op, extend({}, params, {
      subject: mySubject.concat(hisSubject)
    }));
  };
  function getSubject(owner, dom) {
    return Promise.resolve(owner.getSubject(dom || result(owner, 'jq') || $()));
  }
  return TransferCommand;
});
