/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Tony Richards
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/commands/LinkCommand
 */
define(['lex!webEditors', 'Promise', 'underscore', 'bajaux/commands/Command', 'nmodule/webEditors/rc/fe/baja/util/compUtils', 'nmodule/webEditors/rc/fe/baja/util/typeUtils', 'nmodule/webEditors/rc/wb/baja/linkCheckUtil', 'nmodule/webEditors/rc/wb/links/LinkPad', 'nmodule/webEditors/rc/wb/util/TransferDataManager'], function (lexs, Promise, _, Command, compUtils, typeUtils, linkCheckUtil, LinkPad, TransferDataManager) {
  'use strict';

  var all = _.all,
    extend = _.extend;
  var formatDisplayNames = compUtils.formatDisplayNames;
  var isComponent = typeUtils.isComponent;
  var checkLinks = linkCheckUtil.checkLinks;
  var webEditorsLex = lexs[0];
  var transferDataManager = TransferDataManager.getInstance();

  ////////////////////////////////////////////////////////////////
  // Exports
  ////////////////////////////////////////////////////////////////

  /**
   * @class
   * @extends module:bajaux/commands/Command
   * @alias module:nmodule/webEditors/rc/wb/commands/LinkCommand
   * @param {object} params
   * @param {string} params.lex
   * @param {baja.Component|Array.<baja.Component>} params.subject the
   * components on which the command will be invoked
   * @param {baja.Component|Array.<baja.Component>} params.object the components
   * on the other end of the links to be created
   * @param {string} [params.direction='from'] the direction the link will go -
   * `from` means link goes out from object to subject (subject receives the
   * `Link`), `to` means link goes out from subject to object (object receives
   * the `Link`). In both cases, the object is what you called Link Mark on, the
   * subject is what you called Link From or Link To on. To keep it straight in
   * your head, imagine the end result of two linked components on the
   * wiresheet: []----[] - `from` means i did Link Mark on the left and Link
   * From on the right, `to` means i did Link Mark on the right and Link To on
   * the left. Or: if i invoke Link To, my Link Marked component gets a Link from
   * the guy i invoked the command on; if i invoke Link From, the guy i invoked
   * the command on gets a Link from my Link Marked component. Note that if the
   * user is prompted for the other properties, they will also be able to change
   * the direction.
   * @param {string} [params.sourceSlotName] if known beforehand, provide the
   * source slot, otherwise the user will be prompted
   * @param {string} [params.targetSlotName] if known beforehand, provide the
   * target slot, otherwise the user will be prompted
   */
  var LinkCommand = function LinkCommand() {
    var _this = this;
    var params = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    var _params$direction = params.direction,
      direction = _params$direction === void 0 ? 'from' : _params$direction,
      lex = params.lex,
      object = params.object,
      sourceSlotName = params.sourceSlotName,
      subject = params.subject,
      targetSlotName = params.targetSlotName;
    if (!subject) {
      throw new Error('subject required');
    }
    this.$direction = direction;
    this.$lex = lex;
    this.$object = object ? [].concat(object) : [];
    this.$sourceSlotName = sourceSlotName;
    this.$subject = [].concat(subject);
    this.$targetSlotName = targetSlotName;
    this.setEnabled(false);
    Command.call(this, {
      module: 'webEditors',
      lex: lex,
      undoable: function undoable() {
        return _this.$makeUndoable();
      }
    });
  };
  LinkCommand.prototype = Object.create(Command.prototype);
  LinkCommand.prototype.constructor = LinkCommand;

  /**
   * Give the command an opportunity to initialize.
   *
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/commands/LinkCommand>} resolves
   * to the initialized command
   */
  LinkCommand.prototype.init = function () {
    var _this2 = this;
    // enable this command when a linkMark is available
    return getLinkMark().then(function (linkMark) {
      _this2.setEnabled(!!(linkMark && linkMark.length > 0));
      return _this2;
    });
  };

  /**
   * Resolves to the params to be used to create the actual link. These may be
   * specified in the constructor, otherwise the user will be prompted for them.
   * @private
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/links/LinkPad~LinkCheckParams>}
   */
  LinkCommand.prototype.$getLinkCheckParams = function () {
    var linkMarkedComponents = this.$object;
    var sourceSlot = this.$sourceSlotName;
    var invokeTargets = this.$subject;
    var invokeTargetsOnLeft = this.$direction === 'to';
    var targetSlot = this.$targetSlotName;
    if (linkMarkedComponents.length > 0) {
      if (sourceSlot && targetSlot) {
        var result;
        if (invokeTargetsOnLeft) {
          result = {
            sources: invokeTargets,
            targets: linkMarkedComponents,
            sourceSlot: sourceSlot,
            targetSlot: targetSlot
          };
        } else {
          result = {
            sources: linkMarkedComponents,
            targets: invokeTargets,
            sourceSlot: sourceSlot,
            targetSlot: targetSlot
          };
        }
        return Promise.resolve(result);
      } else {
        // show the LinkPad if no slots are specified
        return LinkPad.promptForLinkCheck({
          sources: invokeTargetsOnLeft ? invokeTargets : linkMarkedComponents,
          targets: invokeTargetsOnLeft ? linkMarkedComponents : invokeTargets,
          sourceSlot: sourceSlot,
          targetSlot: targetSlot,
          properties: {
            editIncoming: true,
            editOutgoing: true,
            useHandles: true
          }
        });
      }
    }
    return this.$getLinkMarkedComponents().then(function (components) {
      return LinkPad.promptForLinkCheck({
        sources: invokeTargetsOnLeft ? invokeTargets : components,
        targets: invokeTargetsOnLeft ? components : invokeTargets,
        properties: {
          editIncoming: true,
          editOutgoing: true,
          useHandles: true
        }
      });
    });
  };

  /**
   * @private
   * @returns {Promise.<Array.<baja.Component>>}
   */
  LinkCommand.prototype.$getLinkMarkedComponents = function () {
    return getLinkMark().then(function (linkMark) {
      return Promise.all(linkMark.map(function (navNode) {
        return navNode.getNavOrd().get({
          lease: true
        });
      }));
    });
  };

  /**
   * @private
   * @param {module:nmodule/webEditors/rc/wb/links/LinkPad~LinkCheckParams} linkCheckParams
   * @returns {Promise} to be resolved after the links are added
   */
  LinkCommand.prototype.$doAddLinks = function (linkCheckParams) {
    return checkLinks(extend({
      addLink: true
    }, linkCheckParams));
  };

  /**
   * @private
   * @returns {Promise.<module:bajaux/commands/Command~Undoable>}
   */
  LinkCommand.prototype.$makeUndoable = function () {
    var _this3 = this;
    return this.$getLinkCheckParams().then(function (linkCheckParams) {
      if (!linkCheckParams) {
        return;
      }
      var sources = linkCheckParams.sources,
        targets = linkCheckParams.targets;
      var sourceNames = formatDisplayNames(sources);
      var targetNames = formatDisplayNames(targets);
      var allSourcesAndTargetsMounted = function allSourcesAndTargetsMounted() {
        return all(sources.concat(targets), function (c) {
          return isComponent(c) && c.isMounted();
        });
      };
      var linkChecks;
      return {
        undo: function undo() {
          return removeAddedLinks(sources, targets, linkChecks);
        },
        redo: function redo() {
          return _this3.$doAddLinks(linkCheckParams).then(function (l) {
            linkChecks = l;
          });
        },
        undoText: function undoText() {
          return webEditorsLex.get({
            key: 'commands.link.undoText',
            args: [sourceNames, targetNames]
          });
        },
        redoText: function redoText() {
          return webEditorsLex.get({
            key: 'commands.link.redoText',
            args: [sourceNames, targetNames]
          });
        },
        canUndo: allSourcesAndTargetsMounted,
        canRedo: allSourcesAndTargetsMounted
      };
    });
  };

  /**
   * Get the LinkMark display names for each NavNode.
   *
   * @private
   * @returns {Promise.<Array.<String>>} resolves to an array of strings of marked component
   * display names
   */
  LinkCommand.prototype.$resolveLinkMarkDisplayNames = function () {
    return getLinkMark().then(function (linkMarks) {
      if (linkMarks) {
        return _.map(linkMarks, function (linkMark) {
          return linkMark.getNavDisplayName();
        });
      }
      return [''];
    });
  };

  /**
   * @return {Promise.<String>} resolves to a string that is the display name of this
   * command.
   */
  LinkCommand.prototype.toDisplayName = function () {
    var _this4 = this;
    return this.$resolveLinkMarkDisplayNames().then(function (linkMarkDisplayNames) {
      return webEditorsLex.get({
        key: _this4.$lex + '.displayName',
        args: [formatDisplayNames(linkMarkDisplayNames)]
      });
    });
  };
  function removeAddedLinks(sources, targets, linkChecks) {
    return Promise.all(sources.map(function (source, i) {
      return Promise.all(targets.map(function (target, j) {
        if (!target.isMounted()) {
          throw new Error('Target removed');
        }
        var linkCheck = linkChecks[i][j];
        return target.remove(linkCheck.$addedSlot);
      }));
    }));
  }

  /**
   * @returns {Promise.<Array.<baja.NavNode>>}
   */
  function getLinkMark() {
    return transferDataManager.getLinkMark();
  }
  return LinkCommand;
});
