/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

define(['baja!', 'Promise', 'underscore', 'nmodule/webEditors/rc/wb/baja/clientLinkCheck'], function (baja, Promise, _, clientLinkCheck) {
  'use strict';

  var LinkCheck = require('bajaScript/baja/comp/LinkCheck');
  var Callback = require('bajaScript/baja/comm/Callback');

  /**
   * API Status: **Private**
   * @exports nmodule/webEditors/rc/wb/baja/linkCheckUtil
   */
  var linkCheckUtil = {};

  /**
   * Perform client-side link checking (and optionally follow up with a server side link check)
   *
   * @param {Object} ref
   * @param {baja.Component} [ref.source]
   * @param {baja.Slot}      [ref.sourceSlot]
   * @param {baja.Component} [ref.target]
   * @param {baja.Slot}      [ref.targetSlot]
   *
   * @param {Boolean}   [checkServer=true] - true if the server check link should also be performed.
   * If server check is performed, the client LinkCheck will have a new Promise on attribute
   * LinkCheck.serverLinkCheck.  This secondary `Promise<module:baja/comp/LinkCheck>` is the result
   * of calling ref.targetComponent.checkLink(). If false, only a client-side
   * check will be performed.
   *
   * @returns {Promise.<module:baja/comp/LinkCheck|null>} resolves to a LinkCheck, or null
   * if `checkServer` is false and there is not enough information on the client
   * side to determine whether the link is valid.
   */
  linkCheckUtil.checkLink = function checkLink(ref) {
    var checkServer = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : true;
    var target = ref.target;
    if (checkServer) {
      var serverLinkCheck = target.checkLink(ref);
      return clientLinkCheck(ref).then(function (linkCheck) {
        if (linkCheck) {
          linkCheck.serverLinkCheck = serverLinkCheck;
        } else {
          linkCheck = new LinkCheck(false, 'Cannot determine validity.');
          linkCheck.serverLinkCheck = serverLinkCheck;
        }
        return linkCheck;
      });
    } else {
      return clientLinkCheck.apply(this, arguments);
    }
  };

  /**
   * Perform link checking for multiple sources and targets. Server-side link checking will be done in batch for
   * multiple mounted components. If there is a desire to check a link with an unmounted components, the individual 'checkLink`
   * method is available for creating one link at a time.
   * If the `addLink` parameter is true, no client side link checks will be completed and if there are no server side
   * linking problems, the links will be added to the targets to fully establish the links.
   * When the `addLink` parameter is false, each client LinkCheck that is resolved will have a new Promise on the
   * attribute LinkCheck.serverLinkCheck. This secondary `Promise<<module:baja/comp/LinkCheck>>` is the result from the
   * server side link check.
   *
   * @param {Object} ref
   * @param {baja.Component|Array.<baja.Component>} ref.sources
   * @param {Slot|String} ref.sourceSlot
   * @param {baja.Component|Array.<baja.Component>} ref.targets
   * @param {Slot|String} ref.targetSlot
   * @param {boolean} [ref.addLink=false]
   *
   * @returns {Promise.<Array.<Array.<module:baja/comp/LinkCheck>>>} resolves to a matrix of LinkChecks where the first
   * dimension is sources and the second dimension is targets.
   */
  linkCheckUtil.checkLinks = function checkLinks(ref) {
    var _this = this;
    var addLink = !!ref.addLink;
    var sources = [].concat(ref.sources);
    var targets = [].concat(ref.targets);
    var serverLinkCheck = Promise["try"](function () {
      var cb = new Callback();
      var firstSource = sources[0];
      var space = firstSource.getComponentSpace();
      if (!space) {
        throw new Error("All sources and targets should be mounted");
      }
      space.getCallbacks().checkLinks(ref.sources, ref.sourceSlot.toString(), ref.targets, ref.targetSlot.toString(), cb, ref.addLink);
      return cb.promise();
    });
    if (addLink) {
      return serverLinkCheck;
    }
    return Promise.all(_.map(sources, function (source, i) {
      var sourceSlot = source.getSlot(ref.sourceSlot);
      return Promise.all(_.map(targets, function (target, j) {
        var targetSlot = target.getSlot(ref.targetSlot);
        var args = {
          source: source,
          sourceSlot: sourceSlot,
          target: target,
          targetSlot: targetSlot,
          multipleSources: sources.length > 1
        };
        return clientLinkCheck.apply(_this, [args]).then(function (linkCheck) {
          if (!linkCheck) {
            linkCheck = new LinkCheck(false, 'Cannot determine validity.');
          }
          linkCheck.serverLinkCheck = serverLinkCheck.then(function (linkChecks) {
            return linkChecks[i][j];
          });
          return linkCheck;
        });
      }));
    }));
  };

  /**
   * @typedef {object} module:nmodule/webEditors/rc/wb/baja/linkCheckUtil~LinkConfig
   * @property {Array.<baja.Component>} sources components to link from
   * @property {baja.Slot|string} sourceSlot slot name to link from
   * @property {Array.<baja.Component>} targets components to link to
   * @property {baja.Slot|string} targetSlot slot name to link to
   */

  return linkCheckUtil;
});
