/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/baja/clientLinkCheck
 */
define(['baja!', 'Promise', 'underscore'], function (baja, Promise, _) {
  'use strict';

  var LinkCheck = require('bajaScript/baja/comp/LinkCheck');

  /**
   * @alias module:nmodule/webEditors/rc/wb/baja/clientLinkCheck
    * @param {Object} [obj] the Object Literal used to specify the method's
   * @param {baja.Component} obj.source
   * @param {Slot}      [obj.sourceSlot]
   * @param {baja.Component} obj.target
   * @param {Slot}      [obj.targetSlot]
   * @param {boolean}      [obj.multipleSources=false] Set this parameter to true if multiple Sources are planned.
   * @return {Promise}
   */
  var clientLinkCheck = function clientLinkCheck() {
    var _ref = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
      source = _ref.source,
      sourceSlot = _ref.sourceSlot,
      target = _ref.target,
      targetSlot = _ref.targetSlot,
      _ref$multipleSources = _ref.multipleSources,
      multipleSources = _ref$multipleSources === void 0 ? false : _ref$multipleSources;
    if (!isSubscribed(source)) {
      return Promise.reject(new Error('subscribed source required'));
    }
    if (!isSubscribed(target)) {
      return Promise.reject(new Error('subscribed target required'));
    }
    sourceSlot = source.getSlot(sourceSlot);
    targetSlot = target.getSlot(targetSlot);
    if (!sourceSlot) {
      return invalid('source slot not found');
    }
    if (!targetSlot) {
      return invalid('target slot not found');
    }
    var func = toMakeFunc(sourceSlot, targetSlot);
    return Promise.resolve(func(source, sourceSlot, target, targetSlot, multipleSources) || null);
  };
  var makeFunctions = {
    property: {
      property: propToProp,
      action: propToAction,
      topic: propToTopic
    },
    action: {
      property: actionToProp,
      action: actionToAction,
      topic: actionToTopic
    },
    topic: {
      property: topicToProp,
      action: topicToAction,
      topic: topicToTopic
    }
  };
  function toMakeFunc(sourceSlot, targetSlot) {
    return makeFunctions[slotType(sourceSlot)][slotType(targetSlot)];
  }
  function slotType(slot) {
    if (slot.isAction()) {
      return 'action';
    }
    if (slot.isTopic()) {
      return 'topic';
    }
    return 'property';
  }
  function propToProp(source, sourceSlot, target, targetSlot, multipleSources) {
    var targetFlags = target.getFlags(targetSlot);
    if (targetFlags & baja.Flags.READONLY) {
      return invalid('linkcheck.propReadonly');
    } else if (isComponent(sourceSlot) && !isVector(sourceSlot)) {
      return invalid('linkcheck.propComponent');
    } else if (isLinkTarget(target, targetSlot) && !(targetFlags & baja.Flags.FAN_IN)) {
      return invalid('linkcheck.propAlreadyLinked');
    } else if (multipleSources && !(targetFlags & baja.Flags.FAN_IN)) {
      return invalid('linkcheck.propNoFanIn');
    }
    return checkTypes(sourceSlot.getType(), targetSlot.getType());
  }
  function propToAction(source, sourceSlot, target, targetSlot) {
    var targetType = targetSlot.getParamType();
    if (!targetType || checkTypes(sourceSlot.getType(), targetType)) {
      return valid();
    }
  }
  function propToTopic(source, sourceSlot, target, targetSlot) {
    return invalid('linkcheck.propToTopic');
  }
  function actionToProp(source, sourceSlot, target, targetSlot) {
    return invalid('linkcheck.actionToProp');
  }
  function actionToAction(source, sourceSlot, target, targetSlot) {
    if (is(sourceSlot, 'baja:CompositeAction')) {
      return valid();
    }
    var sourceParam = sourceSlot.getParamType(),
      targetParam = targetSlot.getParamType();
    if (!targetParam) {
      return valid();
    }
    if (!sourceParam) {
      return invalid('linkcheck.actionToActionNoArg');
    }
    return checkTypes(sourceParam, targetParam);
  }
  function actionToTopic(source, sourceSlot, target, targetSlot) {
    if (is(targetSlot, 'baja:CompositeTopic')) {
      return valid();
    }
    var sourceParam = sourceSlot.getParamType(),
      targetEvent = targetSlot.getEventType();
    if (!targetEvent) {
      return valid();
    }
    if (!sourceParam) {
      return invalid('linkcheck.actionToTopicNoEvent');
    }
    return checkTypes(sourceParam, targetEvent);
  }
  function topicToProp(source, sourceSlot, target, targetSlot) {
    return invalid('linkcheck.topicToProp');
  }
  function topicToAction(source, sourceSlot, target, targetSlot) {
    var sourceEvent = sourceSlot.getEventType(),
      targetParam = targetSlot.getParamType();
    if (!targetParam) {
      return valid();
    }
    if (!sourceEvent) {
      return invalid('linkcheck.topicToActionNoEvent');
    }
    return checkTypes(sourceEvent, targetParam);
  }
  function topicToTopic(source, sourceSlot, target, targetSlot) {
    if (is(targetSlot, 'baja:CompositeTopic')) {
      return valid();
    }
    var sourceEvent = sourceSlot.getEventType(),
      targetEvent = targetSlot.getEventType();
    if (!targetEvent) {
      return valid();
    }
    if (!sourceEvent) {
      return invalid('linkcheck.topicToTopicNoEvent');
    }
    return checkTypes(sourceEvent, targetEvent);
  }
  function valid() {
    return LinkCheck.makeValid();
  }
  function invalid(reason) {
    return baja.lex({
      module: 'baja'
    }).then(function (lex) {
      return LinkCheck.makeInvalid(lex.get({
        key: reason,
        def: reason
      }));
    });
  }
  function is(c, typeSpec) {
    return c && c.getType && c.getType().is(typeSpec);
  }
  function isComponent(c) {
    return is(c, 'baja:Component');
  }
  function isVector(c) {
    return is(c, 'baja:Vector');
  }
  function isSubscribed(c) {
    return isComponent(c) && c.isSubscribed();
  }
  function checkTypes(sourceType, targetType) {
    //TODO: CompositeLink/converter-agent-on support
    if (sourceType.is(targetType)) {
      return valid();
    }
  }
  function isLinkTarget(comp, slot) {
    return comp.getLinks(slot).length > 0;
  }
  return clientLinkCheck;
});
