/**
 * @copyright 2021 Tridium, Inc. All Rights Reserved.
 */

/* global URL */

define([], function () {
  "use strict";

  /**
   * API Status: **Private**
   *
   * Utilities for validating the format of ip addresses and host name
   *
   * @exports nmodule/webEditors/rc/util/ipAddressUtil
   * @private
   * @since Niagara 4.13
   */
  var exports = {};

  /**
   * @param {string} address
   * @returns {boolean} if address is an ipv4, mapped ipv4, or ipv6 address
   * @since Niagara 4.14
   */
  exports.isNumericAddress = function (address) {
    return exports.isIpv4(address) || exports.isIpv4Mapped(address) || exports.isIpv6(address);
  };

  /**
   * @param {string} address
   * @returns {boolean} if address is a valid numeric address or a valid hostname
   * @since Niagara 4.14
   */
  exports.isValidHost = function (address) {
    return exports.isNumericAddress(address) || exports.isHostName(address);
  };

  /**
   * Validates an Ipv4 address to see if it is in the correct format, not that it exist
   * @param {string} address the address to be validated
   * @returns {boolean} true if valid false if not
   */
  exports.isIpv4 = function (address) {
    if (!address) {
      return false;
    }
    if (address.indexOf(".") !== -1) {
      var addressParts = address.split(".");
      var count = addressParts.length;
      if (count !== 4) {
        return false;
      }
      for (var i = 0; i < count; i++) {
        var regExp = /^\+?[0-9]\d*$/;
        var part = addressParts[i];
        if (!regExp.test(part)) {
          return false;
        }
        if (!part || part > 255 || part < 0) {
          return false;
        }
      }
    } else {
      if (address < 0 || address > 4294967295) {
        return false;
      }
    }
    try {
      new URL('http://' + address);
    } catch (_) {
      return false;
    }
    return true;
  };

  /**
   * Validates an Ipv6 address to see if it is in the correct format, not that it exist
   * @param {string} address the address to be validated
   * @returns {boolean} true if valid false if not
   */
  exports.isIpv6 = function (address) {
    // check for if the address is blank, null or undefined
    if (!address) {
      return false;
    }

    // check to make sure the address has ":"
    if (!address.includes(":")) {
      return false;
    }

    //check to see if last char is not a single ":", double ":" is allowed
    var addrLength = address.length;
    if (address.charAt(addrLength - 1) === ":" && address.charAt(addrLength - 2) !== ":") {
      return false;
    }

    // if there is more than one "::" it is invalid
    var dblColonCnt = (address.match(/::/g) || []).length;
    if (dblColonCnt > 1) {
      return false;
    }

    // let us split out the scope for future processing
    var addressParts = address.split("%");

    // can have only one'%'
    if (addressParts.length > 2) {
      return false;
    }
    address = addressParts[0];
    var scope = addressParts.length === 2 ? addressParts[1] : "";

    // okay now lets split the rest of the address into its segments
    addressParts = address.split(":");
    var addrPartsLength = addressParts.length;
    var colonCnt = addrPartsLength - 1;

    // can not have more than 8 segements or 7 colons, and must have a minimum of 2
    if (colonCnt > 7 || colonCnt < 2) {
      return false;
    }

    // if we have a double colon then we must have less than 7 colons.
    if (dblColonCnt > 1 && colonCnt < 7) {
      return false;
    }

    //let us check the scope by first determining the type
    if (scope) {
      if (isNaN(parseInt(scope, 10))) {
        // not numeric
        return true;
      } else {
        //if the scope is numeric then it must be an integer starting at 1
        var scopeRegexp = /^\+?[1-9]\d*$/;
        if (!scopeRegexp.test(scope)) {
          return false;
        }
      }
    }
    for (var entry = 0; entry < addrPartsLength; entry++) {
      var regExp = /^[0-9a-fA-F]+$/g;
      var addrPart = addressParts[entry];

      // an empty part is valid at this point, so no need to check.  Just check if there is something
      if (addrPart) {
        // maximum length of each part is 4 characters
        if (addrPart.length > 4) {
          return false;
        }

        // the regeExp should check for hex values
        if (!regExp.test(addrPart)) {
          return false;
        }
      }
    }
    return true;
  };

  /**
   * Validates a host name to see if it is in the correct format, not that it exist
   * @param {string} hostname the hostname to be validated
   * @returns {boolean} true if valid false if not
   */
  exports.isHostName = function (hostname) {
    if (!hostname) {
      return false;
    }
    if (hostname.length > 253) {
      return false;
    }
    if (hostname.startsWith("\\") || hostname.startsWith("/")) {
      return false;
    }
    var hostnameParts = hostname.split(".");
    for (var entry = 0; entry < hostnameParts.length; entry++) {
      var regExp = /^[-0-9a-zA-Z]+$/g;
      var part = hostnameParts[entry];
      if (part.length < 1 || part.length > 63) {
        return false;
      }
      if (part.startsWith("-") || part.endsWith("-")) {
        return false;
      }

      // the regeExp should check for valid values
      if (!regExp.test(part)) {
        return false;
      }
    }
    return true;
  };

  /**
   * Validates an IPv4Mapped address to see if it is in the correct format, not that it exist
   * @param {string} address the address to be validated
   * @returns {boolean} true if valid false if not
   */
  exports.isIpv4Mapped = function (address) {
    if (!address) {
      return false;
    }
    if (!address.toLowerCase().startsWith("::ffff:") || address.length < 8) {
      return false;
    }
    return exports.isIpv4(address.substring(7));
  };
  return exports;
});
