/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['baja!', 'lex!webEditors', 'dialogs', 'Promise', 'nmodule/webEditors/rc/fe/BaseWidget'], function (baja, lexs, dialogs, Promise, BaseWidget) {
  'use strict';

  var webEditorsLex = lexs[0];
  function toDisplayName(ed) {
    var value = ed.value();
    if (baja.hasType(value, 'baja:Complex')) {
      return Promise.resolve(value.getDisplayName());
    } else {
      return ed.toDisplayName();
    }
  }
  function showConfirmAbandonChangesDialog(ed) {
    return toDisplayName(ed).then(function (displayName) {
      // eslint-disable-next-line
      return new Promise(function (resolve, reject) {
        dialogs.show({
          title: webEditorsLex.get('dialogs.confirmClose.title'),
          text: webEditorsLex.get({
            key: 'dialogs.confirmHyperlink.content',
            args: [displayName]
          }),
          buttons: [{
            name: 'save',
            displayName: webEditorsLex.get('commands.save.displayName'),
            handler: function handler() {
              return resolve(ed.save());
            }
          }, {
            name: 'abandon',
            displayName: webEditorsLex.get('dialogs.confirmAbandon.abandon'),
            handler: resolve
          }, {
            name: 'cancel',
            displayName: webEditorsLex.get('dialogs.cancel'),
            handler: reject
          }]
        });
      });
    });
  }

  /**
   * API Status: **Private**
   * @exports nmodule/webEditors/rc/util/commonDialogs
   */
  var exports = {};

  /**
   * Shows a dialog that will prompt the user to either save or abandon changes
   * to a modified editor.
   *
   * If the editor is undefined or is not modified, no dialog will be shown.
   *
   * If the editor is modified, clicking Save will save the editor and then
   * resolve the promise to continue the workflow. Clicking Abandon will resolve
   * the promise without saving. Clicking Cancel will reject the promise to
   * stop the workflow.
   *
   * @param {module:nmodule/webEditors/rc/fe/BaseWidget} ed
   * @returns {Promise} promise to be resolved if the editor is not
   * modified or if the user clicked Save or Abandon, or rejected if the user
   * clicked Cancel or the editor failed to save
   */
  exports.confirmAbandonChanges = function (ed) {
    if (ed && !(ed instanceof BaseWidget)) {
      return Promise.reject(new Error('BaseWidget required'));
    } else if (!ed || !ed.isModified()) {
      return Promise.resolve(); //nothing to save
    }
    return showConfirmAbandonChangesDialog(ed);
  };
  return exports;
});
