function _slicedToArray(r, e) { return _arrayWithHoles(r) || _iterableToArrayLimit(r, e) || _unsupportedIterableToArray(r, e) || _nonIterableRest(); }
function _nonIterableRest() { throw new TypeError("Invalid attempt to destructure non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method."); }
function _unsupportedIterableToArray(r, a) { if (r) { if ("string" == typeof r) return _arrayLikeToArray(r, a); var t = {}.toString.call(r).slice(8, -1); return "Object" === t && r.constructor && (t = r.constructor.name), "Map" === t || "Set" === t ? Array.from(r) : "Arguments" === t || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(t) ? _arrayLikeToArray(r, a) : void 0; } }
function _arrayLikeToArray(r, a) { (null == a || a > r.length) && (a = r.length); for (var e = 0, n = Array(a); e < a; e++) n[e] = r[e]; return n; }
function _iterableToArrayLimit(r, l) { var t = null == r ? null : "undefined" != typeof Symbol && r[Symbol.iterator] || r["@@iterator"]; if (null != t) { var e, n, i, u, a = [], f = !0, o = !1; try { if (i = (t = t.call(r)).next, 0 === l) { if (Object(t) !== t) return; f = !1; } else for (; !(f = (e = i.call(t)).done) && (a.push(e.value), a.length !== l); f = !0); } catch (r) { o = !0, n = r; } finally { try { if (!f && null != t["return"] && (u = t["return"](), Object(u) !== u)) return; } finally { if (o) throw n; } } return a; } }
function _arrayWithHoles(r) { if (Array.isArray(r)) return r; }
/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*eslint-env browser */ /*jshint browser: true */

define(['baja!', 'lex!webEditors', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/baja/util/compUtils', 'nmodule/webEditors/rc/fe/baja/util/typeUtils'], function (baja, lexs, Promise, _, compUtils, typeUtils) {
  'use strict';

  var _lexs = _slicedToArray(lexs, 1),
    webEditorsLex = _lexs[0];
  var any = _.any,
    initial = _.initial,
    last = _.last;
  var RPC_TYPE_SPEC = 'web:ChangeUserPasswordRpc';
  var getNavOrd = compUtils.getNavOrd,
    getParentComponent = compUtils.getParentComponent;
  var isComplex = typeUtils.isComplex,
    isComponent = typeUtils.isComponent;

  /** @param {String} err */
  function reject(err) {
    return Promise.reject(new Error(err));
  }
  function changeUserPasswords(usernames, passwords, params) {
    return baja.rpc({
      typeSpec: RPC_TYPE_SPEC,
      method: 'changeUserPasswords',
      args: [usernames, _.map(passwords, btoaUtf8)],
      batch: params && params.batch
    })["catch"](function (err) {
      baja.error(err);
      handleJSONParseableException(err);
    });
  }
  function validatePasswords(schemes, usernames, passwords, params) {
    return baja.rpc({
      typeSpec: RPC_TYPE_SPEC,
      method: 'validatePasswords',
      args: [usernames, _.map(passwords, btoaUtf8), schemes],
      batch: params && params.batch
    })["catch"](function (err) {
      baja.error(err);
      handleJSONParseableException(err, webEditorsLex.get('password.validateError'));
    });
  }
  function setPassword(pwd, pwdSlotName, baseOrd, params) {
    return baja.rpc({
      typeSpec: "web:PasswordRpc",
      method: 'setPassword',
      args: [pwd, pwdSlotName, baseOrd, params.useDefault],
      batch: params && params.batch
    })["catch"](function (err) {
      baja.error(err);
      throw err;
    });
  }
  function setCertificateAliasAndPassword(alias, pwd, slotName, baseOrd, params) {
    return baja.rpc({
      typeSpec: "web:PasswordRpc",
      method: 'setCertificateAliasAndPassword',
      args: [alias, pwd, slotName, baseOrd, params.useDefault, params.setAlias, params.setPassword],
      batch: params && params.batch
    })["catch"](function (err) {
      baja.error(err);
      throw err;
    });
  }

  /**
   * Encode the string to base64 using the UTF-8 charset
   *
   * @inner
   * @param str the source string
   * @returns {string} the base64 encoded string
   */
  function btoaUtf8(str) {
    return btoa(encodeURIComponent(str).replace(/%([0-9A-F]{2})/g, function toSolidBytes(match, p1) {
      return String.fromCharCode(parseInt(p1, 16));
    }));
  }
  function handleJSONParseableException(err, messageSummary) {
    var errs,
      msgs = [];
    try {
      errs = JSON.parse(err.message);
    } catch (err2) {
      //Error message is not JSON so no additional processing is required
      var message = err.message;
      if (messageSummary) {
        message = messageSummary + '\n' + message;
      }
      throw new Error(message);
    }
    if (messageSummary) {
      msgs.push(messageSummary + '\n');
    }
    var length = Object.keys(errs).length;
    _.each(errs, function (err, name) {
      if (err) {
        if (name && length > 1) {
          msgs.push(name + ': ' + err);
        } else {
          msgs.push(err);
        }
      }
    });
    throw new Error(msgs.join('\n'));
  }

  /**
   * API Status: **Private**
   *
   * Utilities relating to password changes. These include some client-side
   * security checks, but they should be treated as feel-good measures purely
   * for UI warnings and the like. Anything sent up from this module in a
   * non-HTTPS environment is subject to man in the middle attacks.
   *
   * This module communicates with the following servlets:
   *
   * - `com.tridium.web.servlets.ChangeUserPasswordServlet`
   *
   * @exports nmodule/webEditors/rc/servlets/password
   */
  var exports = {};

  /**
   * Check to see if we are running in a secured environment. Note that
   * this is a "feel-good" measure: if we're in an insecure environment no
   * amount of JS checks will make it secure for real.
   *
   * @private
   * @returns {Promise}
   */
  exports.$isSecure = function () {
    return Promise.resolve(baja.bson.$canEncodeSecurely());
  };
  function requireSecure() {
    //noinspection JSAccessibilityCheck
    return exports.$isSecure().then(function (secure) {
      if (!secure) {
        return reject('insecure environment');
      }
    });
  }

  /**
   * Change the passwords for the given users.
   *
   * @param {Array.<String>} usernames array of usernames
   * @param {Array.<String>} passwords array of plaintext passwords
   * @param {baja.Complex} base a base Complex to use to resolve the
   * `WebService`
   * @param {Object} params
   * @param {baja.comm.Batch} [params.batch]
   * @returns {Promise} promise to be resolved if the passwords are
   * successfully changed. Rejects if the passwords cannot be changed, or if
   * called from a non-SSL environment (no POST to the station will occur).
   */
  exports.changePasswords = function (usernames, passwords, base, params) {
    var reserve = baja.comm.Batch.reserve(params);
    return requireSecure().then(function () {
      if (!_.isArray(usernames) || !_.isArray(passwords)) {
        return reject('usernames and passwords required');
      }
      if (usernames.length !== passwords.length) {
        return reject('different numbers of usernames and passwords');
      }
      if (!isComplex(base)) {
        return reject('base required');
      }
      return reserve.commit(changeUserPasswords(usernames, passwords, {
        batch: reserve
      }));
    });
  };

  /**
   * Verify the given passwords against the specified authentication schemes.
   *
   * @param {Array.<String>} schemes array of authentication scheme names
   * @param {Array.<String>} usernames array of usernames
   * @param {Array.<String>} passwords array of plaintext passwords
   * @param {baja.Complex} base a base Complex to use to resolve the
   * `WebService`
   * @param {Object} params
   * @param {baja.comm.Batch} [params.batch]
   * @returns {Promise} promise to be resolved if the passwords are valid for
   * their corresponding authentication schemes (meet strength requirements
   * etc.). Rejects if any passwords are invalid, or if called from a non-SSL
   * environment (no POST to the station will occur). The message of the
   * rejection Error can be JSON-parsed as a username -> reason map.
   */
  exports.validatePasswords = function (schemes, usernames, passwords, base, params) {
    var reserve = baja.comm.Batch.reserve(params);
    return requireSecure().then(function () {
      if (!(_.isArray(schemes) && _.isArray(passwords) && _.isArray(usernames))) {
        return reject('schemes, usernames and passwords required');
      }
      if (schemes.length !== passwords.length) {
        return reject('different numbers of schemes and passwords');
      }
      if (schemes.length !== usernames.length) {
        return reject('different numbers of schemes and usernames');
      }
      if (!isComplex(base)) {
        return reject('base required');
      }
      return reserve.commit(validatePasswords(schemes, usernames, passwords, {
        batch: reserve
      }));
    });
  };

  /**
   * Set the password property of component (resolved via baseOrd).
   *
   * This is not needed in most cases. Since Niagara 4.14, BajaScript and BOX have built-in support
   * for setting passwords on the station.
   *
   * @param {baja.Simple|String} pwd - The password to set, as string or `baja:Password`
   * @param {baja.Slot|String} pwdSlot - The frozen slot to set the password on
   * @param {baja.Complex|baja.Ord} base - The base complex or ord
   * @param {Object} [params]
   * @param {baja.comm.Batch} [params.batch]
   * @param {boolean} [params.useDefault] - Indicates to use the default password, defaults to false.
   * @returns {Promise} Promise to be resolved for successful password set
   * or reject if insecure
   */
  exports.setPassword = function (pwd, pwdSlot, base, params) {
    var reserve = baja.comm.Batch.reserve(params);
    var useDefault = params && params.useDefault || false;
    return requireSecure().then(function () {
      if (isComplex(base)) {
        return getNavOrd(base);
      } else if (baja.hasType(base, 'baja:Ord')) {
        return base;
      } else {
        throw new Error('base required');
      }
    }).then(function (ord) {
      return reserve.commit(setPassword(pwd.encodeToString(), String(pwdSlot), String(ord), {
        batch: reserve,
        useDefault: useDefault
      }));
    });
  };

  /**
   * Set the `baja:CertificateAliasAndPassword` property of component (resolved via baseOrd)
   * @param {String} alias - The alias to set
   * @param {baja.Simple|String} pwd - The password to set, as string or `baja:Password`
   * @param {baja.Slot|String} slot - The slot to set the `baja:CertificateAliasAndPassword` on.
   * @param {baja.Complex|baja.Ord} base - The base complex or ord
   * @param {Object} params
   * @param {baja.comm.Batch} [params.batch]
   * @param {boolean} [params.useDefault=false] - Indicates to use the default password.
   * @param {boolean} [params.setAlias=false] - Indicates to set the alias.
   * @param {boolean} [params.setPassword=false] - Indicates to use a password.
   * @returns {Promise} Promise to be resolved for a successful set
   * or reject if insecure
   * @since Niagara 4.13U2
   */
  exports.setCertificateAliasAndPassword = function (alias, pwd, slot, base, params) {
    var reserve = baja.comm.Batch.reserve(params);
    var useDefault = params && params.useDefault || false;
    var setAlias = params && params.setAlias || false;
    var setPassword = params && params.setPassword || false;
    return requireSecure().then(function () {
      if (isComplex(base)) {
        return compUtils.getNavOrd(base);
      } else if (baja.hasType(base, 'baja:Ord')) {
        return base;
      } else {
        throw new Error('base required');
      }
    }).then(function (ord) {
      return reserve.commit(setCertificateAliasAndPassword(alias, pwd.encodeToString(), String(slot), String(ord), {
        batch: reserve,
        useDefault: useDefault,
        setAlias: setAlias,
        setPassword: setPassword
      }));
    });
  };

  /**
   * If the complex has a password slot, reset it to the default value so
   * BajaScript can send the instance up to the station, and hold onto it so it
   * can be set via RPC afterward.
   *
   * @param {baja.Complex} comp an unmounted component ready to be sent up to
   * the station
   * @param {string|baja.Slot} [slot] the slot containing the password; if
   * omitted, the first `baja:Password` slot will be used
   * @returns {{password: string, slot: string}|undefined} the password and slot
   * name of the component's password slot, if it had one; otherwise undefined.
   *
   * @since Niagara 4.13
   * @deprecated since Niagara 4.14
   */
  exports.stripPasswordForLater = function (comp, slot) {
    slot = slot || comp.getSlots().is('baja:Password').first();
    if (slot) {
      var password = comp.get(slot);
      comp.set({
        slot: slot,
        value: baja.$('baja:Password')
      });
      return {
        slot: String(slot),
        password: password.encodeToString()
      };
    }
  };

  /**
   * Evaluate fips status that was set on the server side, and resolves
   * either fips password strength or default password strength.
   *
   * @returns {Promise.<baja.Struct>}
   */
  exports.getDefaultPasswordStrength = function () {
    return baja.rpc({
      typeSpec: "web:PasswordRpc",
      method: "getDefaultPasswordStrength",
      args: []
    });
  };

  /**
   * Resolves the fips password strength when isFips is true,
   * and default password strength when isFips is false.
   *
   * @param {boolean} isFips
   * @returns {Promise.<baja.Struct>}
   */
  exports.getPasswordStrength = function (isFips) {
    return baja.rpc({
      typeSpec: "web:PasswordRpc",
      method: "getPasswordStrength",
      args: [isFips]
    });
  };

  /**
   * When adding new Complex instances to the station, if those instances
   * contain passwords, BajaScript will prevent them from being sent across the
   * wire.
   *
   * As a workaround, this method will strip out all the passwords from the
   * complexes, add the complexes, and then re-set the passwords via RPC after
   * the fact.
   *
   * @param {Array.<baja.Complex>} comps complexes to add. These may not be
   * mounted components, and they may not be Users (use one of the User-centric
   * methods for these).
   * @param {function} doAdd performs the work of adding the components to the
   * station. It may return a promise. It *must* resolve to the newly added,
   * live values (BajaScript does not keep Complex instance-equal after adding
   * them up to the station).
   * @returns {Promise} to be resolved after the components are added, and the
   * passwords have been set.
   *
   * @since Niagara 4.13
   * @deprecated since Niagara 4.14, BajaScript and BOX have built-in support for setting passwords
   * on the station.
   */
  exports.safelyAdd = function (comps, doAdd) {
    return Promise["try"](function () {
      return comps.map(function (comp) {
        if (isComponent(comp) && comp.isMounted()) {
          throw new Error('cannot call on a mounted component');
        }
        if (comp.getType().is('baja:User')) {
          throw new Error('cannot call on a User');
        }
        return sockAwayPasswords(comp);
      });
    }).then(function (sockedPasswords) {
      var hasPasswordsToSet = any(sockedPasswords, function (map) {
        return Object.keys(map).length;
      });
      return Promise.resolve(hasPasswordsToSet && requireSecure()).then(doAdd).then(function (newComps) {
        if (!Array.isArray(newComps)) {
          throw new Error('doAdd callback must resolve to an array of Complexes');
        }
        return Promise.all(sockedPasswords.map(function (passwordMap, i) {
          var comp = newComps[i];
          return Promise.all(Object.keys(passwordMap).map(function (path) {
            var password = passwordMap[path];
            path = path.split('/');
            var toSet = fromSlotPath(comp, initial(path));
            var slot = last(path);
            var parent = getParentComponent(toSet);
            if (parent && parent.isMounted()) {
              return exports.setPassword(password, slot, toSet);
            } else {
              return toSet.set({
                slot: slot,
                value: baja.$('baja:Password', password)
              });
            }
          }));
        }));
      });
    });
  };

  /**
   * @param {baja.Component} comp
   * @returns {object.<string, string>} a map from
   * '/'-delimited slot path to the password stripped from that slot
   */
  function sockAwayPasswords(comp) {
    var map = {};
    (function doSock(comp, path) {
      comp.getSlots().properties().each(function (prop) {
        var myPath = path.concat(prop);
        var kid = comp.get(prop);
        if (kid.getType().is('baja:Password')) {
          map[myPath.join('/')] = exports.stripPasswordForLater(comp, prop).password;
        } else if (kid.getType().isComplex()) {
          doSock(kid, myPath);
        }
      });
    })(comp, []);
    return map;
  }
  function fromSlotPath(comp, path) {
    return path.reduce(function (comp, prop) {
      return comp.get(prop);
    }, comp);
  }
  return exports;
});
