/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*global setTimeout: false, clearTimeout: false */

define(['baja!', 'log!nmodule.webEditors.rc.servlets.navMonitor', 'bajaux/events', 'Promise', 'underscore', 'nmodule/js/rc/asyncUtils/asyncUtils', 'nmodule/js/rc/tinyevents/tinyevents'], function (baja, log, events, Promise, _, asyncUtils, tinyevents) {
  'use strict';

  /**
   * API Status: **Private**
   *
   * Functions for the NavMonitor mechanism that performs `Fw.TOUCH` operations
   * on mounted nodes in order to keep them alive within a space.
   *
   * @exports nmodule/webEditors/rc/servlets/navMonitor
   */
  var exports = {};
  var listeners = [],
    touchTicket,
    doRequire = asyncUtils.doRequire,
    logFine = log.fine.bind(log),
    logSevere = log.severe.bind(log),
    INITIALIZE_EVENT = events.INITIALIZE_EVENT,
    DESTROY_EVENT = events.DESTROY_EVENT,
    TOUCH_DELAY = 500,
    DEFAULT_SCAN_RATE = 20000,
    lastParams;

  ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  var getSystemProperties = _.once(function () {
    return doRequire('niagaraSystemProperties')["catch"](_.constant({}));
  });
  function getScanRate(params) {
    var scanRate = params && params.scanRate;
    if (typeof scanRate !== 'undefined') {
      return Promise.resolve(scanRate);
    }
    return getSystemProperties().then(function (systemProperties) {
      scanRate = systemProperties['niagara.nav.touch.scanRate'];
      return typeof scanRate === 'undefined' ? DEFAULT_SCAN_RATE : scanRate;
    });
  }
  function allOrds() {
    return Promise.all(_.map(listeners, function (listener) {
      return listener.getTouchableOrds();
    })).then(_.flatten);
  }
  function doTouch() {
    return allOrds().then(exports.touchOrds);
  }
  function hasMixin(ed) {
    return ed.hasMixIn('navMonitor');
  }

  // TODO: BOX fix? niagaraVirtuals fix?
  // resolving the ORD for the virtuals container itself causes the BatchResolve
  // to fail station-side.
  function isTouchable(ordStr) {
    return ordStr.match(/virtual:\/.+/);
  }

  ////////////////////////////////////////////////////////////////
  // exports
  ////////////////////////////////////////////////////////////////

  /**
   * An editor with the `NavMonitorSupport` mixin should emit this `tinyevent`
   * to request that a nav monitor touch call be immediately made.
   * @type {string}
   */
  exports.REQUEST_TOUCH_EVENT = 'requestTouch';

  /**
   * Starts up the nav monitor to periodically make touch requests to the
   * station.
   *
   * @private
   * @param {Object} [params]
   * @param {Number} [params.scanRate=20000] touch requests will be made at this
   * rate (in ms). If omitted, the `niagara.nav.touch.scanRate` system property
   * from the station will be used, if present.
   * @returns {Promise} promise to be resolved when monitoring has started up
   */
  exports.$startMonitor = function (params) {
    lastParams = params;
    return getScanRate(lastParams || params).then(function (scanRate) {
      if (scanRate <= 0) {
        logFine('navMonitor disabled with scanRate ' + scanRate);
        return;
      } else {
        logFine('navMonitor enabled with scanRate ' + scanRate);
      }
      clearTimeout(touchTicket);
      touchTicket = setTimeout(function scheduleTouch() {
        doTouch().then(function () {
          if (touchTicket !== null) {
            touchTicket = setTimeout(scheduleTouch, scanRate);
          }
        })["catch"](logSevere);
      }, TOUCH_DELAY);
    });
  };

  /**
   * Immediately stop making touch requests to the station.
   */
  exports.$stopMonitor = function () {
    clearTimeout(touchTicket);
    //set to null to signal "don't restart" after a current doTouch resolves
    touchTicket = null;
  };

  /**
   * Un-register a widget from the nav monitor process.
   * @param {module:bajaux/Widget|Object} obj
   */
  exports.deregister = function (obj) {
    listeners = _.without(listeners, obj);
    obj.removeListener(exports.REQUEST_TOUCH_EVENT, obj.$requestTouchListener);
  };

  /**
   * Register a widget with the nav monitor process. When a scan is made, this
   * object will be queried for ORDs to touch by calling its `getTouchableOrds`
   * function.
   * @param {module:bajaux/Widget|Object} obj
   * @throws {Error} if `getTouchableOrds()` not implemented
   */
  exports.register = function (obj) {
    if (!(obj && typeof obj.getTouchableOrds === 'function')) {
      throw new Error('getTouchableOrds() not implemented');
    }
    listeners.push(obj);
    var listener = obj.$requestTouchListener = function () {
      exports.requestImmediateTouch();
    };
    tinyevents(obj);
    obj.on(exports.REQUEST_TOUCH_EVENT, listener);
  };

  /**
   * Request that a touch request be made immediately, effectively resetting the
   * scan rate interval back to 0.
   */
  exports.requestImmediateTouch = function () {
    exports.$startMonitor();
  };

  /**
   * Register `bajaux` event listeners on this element to automatically
   * register/unregister `Widget`s that implement that have the
   * `NavMonitorSupport` mixin.
   * @param {JQuery} elem
   */
  exports.armElement = function (elem) {
    elem.on(INITIALIZE_EVENT, function (e, ed) {
      if (hasMixin(ed)) {
        exports.register(ed);
      }
    }).on(DESTROY_EVENT, function (e, ed) {
      if (hasMixin(ed)) {
        exports.deregister(ed);
      }
    });
  };

  /**
   * Call up to the station to request a TOUCH operation on an array of ORDs.
   * This will keep alive any ORDs that require TOUCH-ing to do so (e.g.
   * virtual components).
   *
   * @param {Array.<baja.Ord|String>} ords the ORDs to touch. (Any duplicates in
   * this list will be removed before sending the network call.)
   * @param {Object} [params]
   * @param {String|baja.Ord} [params.base] to be used station-side to resolve
   * the given ORDs to `BISpaceNode`s. If omitted, the absolute ord of the
   * Station root will be used.
   * @param {baja.comm.Batch} [params.batch]
   * @returns {Promise}
   */
  exports.touchOrds = function (ords, params) {
    if (!_.isArray(ords)) {
      return Promise.reject(new Error('ords array required'));
    }
    var base = String(params && params.base || baja.station.getAbsoluteOrd());
    ords = _.chain(ords).map(String).unique().filter(isTouchable).value();
    if (!ords.length) {
      return Promise.resolve();
    }
    return baja.rpc({
      typeSpec: 'baja:NavMonitorRpc',
      method: 'touchOrds',
      args: [base, ords],
      batch: params && params.batch
    });
  };
  return exports;
});
