/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Tony Richards
 */
/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/mixin/zoomMixIn
 */
define(['bajaux/Widget', 'Promise', 'underscore'], function (Widget, Promise, _) {
  'use strict';

  var MIXIN_NAME = 'zoom';

  /**
   * Applies the `zoom` mixin to the target Widget.
   *
   * @function
   * @alias module:bajaux/mixin/zoomMixIn
   * @param {module:bajaux/Widget} target The widget to apply the mixin to.
   * @param {Object} params
   * @param {Array<string>} params.zoomTargets array of css selectors that should handle pinch
   *    to zoom touch events.  It is required that this selector returns an element that can have
   *    its height changed the element must have a parent that has overflow-y: scroll.
   * @param {Number} params.zoomFactor (Optional)  a zoom factor to scale the pinch zoom in / out.
   *  default is 2.0 if not specified
   */
  var zoomMixIn = function zoomMixIn(target, params) {
    if (!isWidget(target)) {
      throw new Error("Zoom + Scroll mixin only applies to instances or sub-classes of Widget");
    }
    if (!_.isObject(params)) {
      throw new Error("Zoom + Scroll mixin: No zoom / scroll parameters");
    }
    if (!Array.isArray(params.zoomTargets)) {
      throw new Error("Zoom mixin: No Zoom Targets specified");
    }
    var mixins = target.$mixins,
      initialize = target.initialize,
      destroy = target.destroy;
    if (!_.contains(mixins, MIXIN_NAME)) {
      mixins.push(MIXIN_NAME);
    }
    var lastY1 = 0;
    var lastY2 = 0;
    var zoomFactor = params.zoomFactor || 2.0;
    var handleStart = function handleStart(event) {
      if (event.touches.length === 2) {
        var topY, bottomY;
        if (event.touches[0].clientY < event.touches[1].clientY) {
          topY = 0;
          bottomY = 1;
        } else {
          topY = 1;
          bottomY = 0;
        }
        lastY1 = event.touches[topY].clientY;
        lastY2 = event.touches[bottomY].clientY;
      } else {
        if (event.touches[0]) {
          lastY1 = event.touches[0].clientY;
        }
      }
    };
    var applyStyles = function applyStyles(deltaTop, deltaBottom) {
      var constrain = false;
      params.zoomTargets.forEach(function (selector) {
        var element = target.jq().find(selector);
        var parentHeight = element.parent().height();
        var originalHeight = element.height();
        var scrollTop = element.parent().scrollTop();
        var newHeight = originalHeight + deltaTop - deltaBottom;
        if (newHeight < parentHeight) {
          element.height(parentHeight);
          element.parent().scrollTop(0);
          constrain = true;
        } else {
          element.height(newHeight);
          element.parent().scrollTop(scrollTop + deltaTop);
        }
      });
      return constrain;
    };
    var handleZoom = function handleZoom(event) {
      var deltaTop = 0,
        deltaBottom = 0;
      if (event.touches.length === 1 && event.shiftKey) {
        var curY = event.touches[0].clientY;
        var scroll = (curY - lastY1) * 2;
        lastY1 = curY;
        scroll *= 2;
        deltaBottom = scroll;
      } else if (event.touches.length > 1) {
        var topY, bottomY;
        if (event.touches[0].clientY < event.touches[1].clientY) {
          topY = 0;
          bottomY = 1;
        } else {
          topY = 1;
          bottomY = 0;
        }
        var Y1 = event.touches[topY].clientY;
        deltaTop = (lastY1 - Y1) * zoomFactor;
        lastY1 = Y1;
        var Y2 = event.touches[bottomY].clientY;
        deltaBottom = (lastY2 - Y2) * zoomFactor;
        lastY2 = Y2;
      } else {
        return;
      }
      if (!applyStyles(deltaTop, deltaBottom)) {
        event.preventDefault();
      }
    };
    target.initialize = function (dom) {
      return Promise.resolve(initialize.apply(target, arguments)).then(function () {
        params.zoomTargets.forEach(function (selector) {
          target.jq().on('touchstart', selector, handleStart);
          target.jq().on('touchmove', selector, handleZoom);
        });
      });
    };
    target.destroy = function () {
      params.zoomTargets.forEach(function (selector) {
        target.jq().off('touchstart', selector, handleStart);
        target.jq().off('touchmove', selector, handleZoom);
      });
      return destroy.apply(target, arguments);
    };
  };
  function isWidget(target) {
    return target instanceof Widget;
  }
  return zoomMixIn;
});
