/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Tony Richards
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/TabbedCompositeEditor
 */
define(['jquery', 'underscore', 'Promise', 'bajaux/util/CommandButtonGroup', 'bajaux/commands/ToggleCommandGroup', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/CompositeEditor'], function ($, _, Promise, CommandButtonGroup, ToggleCommandGroup, fe, CompositeEditor) {
  'use strict';

  /**
   * TabbedCompositeEditor - a composite editor where each child editor is on a
   * separate tab.
   *
   * When deriving from this class, you should implement `makeBuilder` in the same
   * way as any other `CompositeEditor`, with an additional method `getTabFor`,
   * which should return the params passed to a `ToggleCommand` / `Command` constructor.
   *
   * @example
   * builder.getTabFor = function(key) {
   *  return {
   *    displayName: getDisplayName(key)
   *  };
   * };
   *
   * builder.getTabFor = function(key) {
   *  return {
   *    module: "myModule",
   *    lex: "myCommand" + key
   *  };
   * };
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/CompositeEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/TabbedCompositeEditor
   */
  var TabbedCompositeEditor = function TabbedCompositeEditor(params) {
    var that = this;
    that.$tabs = {};
    CompositeEditor.apply(this, arguments);
  };
  TabbedCompositeEditor.prototype = Object.create(CompositeEditor.prototype);
  TabbedCompositeEditor.prototype.constructor = TabbedCompositeEditor;

  /**
   * Initialize the editor.
   *
   * @param {JQuery} dom - DOM element that the editor will use to render itself.
   *
   * @returns {Promise} Resolves when the editor is initialized
   */
  TabbedCompositeEditor.prototype.doInitialize = function (dom) {
    var that = this,
      builder = that.getBuilder(),
      builderKeys,
      commands = [],
      originalArgs = arguments;
    dom.html('<div class="ux-tab-bar"></div>');
    if (typeof builder.getTabFor !== 'function') {
      throw new Error('TabbedCompositeEditor requires a builder that implements getTabFor');
    }
    return Promise.resolve(builder.getKeys()).then(function (keys) {
      builderKeys = keys;
      keys.forEach(function (key) {
        var tab = builder.getTabFor(key);
        if (tab) {
          var command = _.extend({}, tab, {
            value: key
          });
          commands.push(command);
          that.$tabs[key] = command;
        }
      });
      that.$toggleCmdGrp = new ToggleCommandGroup({
        onChange: function onChange(value) {
          that.$showTabPanel(value);
        },
        commands: commands
      });
      return CompositeEditor.prototype.doInitialize.apply(that, originalArgs);
    }).then(function () {
      return fe.buildFor({
        dom: $('.ux-tab-bar', dom),
        type: CommandButtonGroup,
        value: that.$getToggleCommandGroup(),
        initializeParams: {
          toggleGroup: true
        }
      });
    }).then(function (commandButtonGroup) {
      that.$cmdButtonGrp = commandButtonGroup;
      if (!that.$getToggleCommandGroup().isEmpty()) {
        // Show the first tab panel
        that.$showTabPanel(builderKeys[0]);
        // Make the tab bar indicate that the first tab is selected
        that.$getToggleCommandGroup().get(0).setSelected(true);
      }
    });
  };
  TabbedCompositeEditor.prototype.doLayout = function () {
    var activeWidget = this.getActiveWidget();
    if (activeWidget) {
      return activeWidget.layout();
    }
  };
  TabbedCompositeEditor.prototype.doDestroy = function () {
    return this.getChildWidgets().destroyAll();
  };

  /**
   * getActiveWidget gets the widget currently being displayed.
   *
   * Only one child widget at a time is displayed, based on which of the tabs (i.e. ToggleCommand
   * buttons) is selected.
   *
   * @returns {Widget} active Widget, which is the Widget being displayed due to a tab
   * being selected.
   */
  TabbedCompositeEditor.prototype.getActiveWidget = function () {
    var selectedTab = this.$getToggleCommandGroup().getSelected();
    if (selectedTab) {
      var dom = this.getBuilder().getDomFor(selectedTab.$value);
      if (dom) {
        return dom.data('widget');
      }
    }
  };
  TabbedCompositeEditor.prototype.$getToggleCommandGroup = function () {
    return this.$toggleCmdGrp;
  };
  TabbedCompositeEditor.prototype.$showTabPanel = function (choice) {
    var that = this,
      needsLayout;
    _.each(that.$tabs, function (tab, key) {
      var editorDom = that.getBuilder().getDomFor(key);
      if (tab.value === choice) {
        editorDom.show();
        needsLayout = editorDom.data('widget');
      } else {
        editorDom.hide();
      }
    });
    if (needsLayout) {
      return needsLayout.layout();
    }
  };
  return TabbedCompositeEditor;
});
