/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/PermissionsRowEditor
 */
define(['baja!', 'log!nmodule.webEditors.rc.fe.baja.PermissionsRowEditor', 'jquery', 'Promise', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'hbs!nmodule/webEditors/rc/fe/baja/template/PermissionsRow'], function (baja, log, $, Promise, BaseEditor, tplRow) {
  'use strict';

  var logError = log.severe.bind(log),
    CELL_DISABLED_STYLE_CLASS = 'permissionCellDisabled';

  /**
   * A field editor for working with baja.Permission.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/PermissionsRowEditor
   *
   * @param {baja.Permissions} [params.maxPermissions] the maximum permissions level allowed.
   *           Used to disable the checkboxes and show them as greyed out.
   *           If not supplied, baja.Permissions.all is assumed.
   * @param {String} [params.rowDescription] a description for this permissions row. If no row description is supplied,
   *           the description cell will not be rendered. Defaults to blank.
   * @param {Number} [params.rowId] used to identify a row in the grid. Defaults to -1;
   */
  var PermissionsRowEditor = function PermissionsRowEditor(params) {
    var that = this;
    BaseEditor.call(that, $.extend(true, {
      keyName: 'PermissionsRowEditor'
    }, params));
    that.$maxPermissions = params && params.maxPermissions || baja.Permissions.all;
    that.$rowDescription = params && params.rowDescription || '';
    that.$rowId = params && params.rowId || -1;
  };
  PermissionsRowEditor.prototype = Object.create(BaseEditor.prototype);
  PermissionsRowEditor.prototype.constructor = PermissionsRowEditor;

  /**
   * The cell style indicating that a user does not have permissions to edit
   * @private
   */
  PermissionsRowEditor.CELL_DISABLED_STYLE_CLASS = CELL_DISABLED_STYLE_CLASS;
  PermissionsRowEditor.prototype.$getMaxPermissions = function () {
    return this.$maxPermissions;
  };

  /**
   * A description for this permissions row
   *
   * @returns {String}
   */
  PermissionsRowEditor.prototype.getRowDescription = function () {
    return this.$rowDescription;
  };

  /**
   * An identifier for this permissions row
   *
   * @returns {Number}
   */
  PermissionsRowEditor.prototype.getRowId = function () {
    return this.$rowId;
  };

  /**
   * Creates a PermissionsMap element.
   *
   * @param {JQuery} dom
   */
  PermissionsRowEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.on('change', 'input:checkbox', function (ev) {
      that.$handleChangeEvent(ev)["catch"](logError);
      return false;
    });
  };
  PermissionsRowEditor.prototype.$handleChangeEvent = function (event) {
    var that = this,
      newValue;
    that.setModified(true);

    // on the basis of the new value ,we may need to set other boxes
    if (event && event.currentTarget) {
      newValue = event.currentTarget.checked;
      switch (event.currentTarget.className) {
        case 'adInvoke':
          // I => i
          that.$getOperatorInvokeElement().prop('checked', newValue);
          break;
        case 'adWrite':
          // W => w, R (& r)
          that.$getOperatorWriteElement().prop('checked', newValue);
          that.$getAdminReadElement().prop('checked', newValue);
        /* falls through */ // fall through to set r

        case 'adRead':
        // (R => r) fall through

        case 'opWrite':
          // R => r
          that.$getOperatorReadElement().prop('checked', newValue);
          break;
      }
    }
    return that.read().then(function (value) {
      return that.$update(value);
    });
  };

  /**
   * Loads the given baja.Permissions into the editor.
   *
   * @param {baja.Permissions} permissions
   * @returns {Promise}
   */
  PermissionsRowEditor.prototype.doLoad = function (permissions) {
    var that = this;
    return that.$update(permissions);
  };

  /**
   * update the checkbox values and formatting in this editor using the
   * passed baja.Permissions and the current user's maximum allowed permissions.
   * Checkboxes are disabled if its permission is implied by another (eg AdminRead implies OperatorRead).
   * If the user doesn't have the permissions to a checkbox, an additional style is applied to the cell
   * (by default greyed out)
   *
   * @private
   * @param {baja.Permissions} permissions
   * @returns {Promise}
   */
  PermissionsRowEditor.prototype.$update = function (permissions) {
    var that = this,
      max = that.$getMaxPermissions();
    var adReadImplied = permissions.hasAdminWrite(),
      // W => R
      opInvokeImplied = permissions.hasAdminInvoke(),
      // I => i
      opWriteImplied = permissions.hasAdminWrite(),
      // W => w
      opReadImplied = permissions.hasOperatorWrite() || permissions.hasAdminRead() || opWriteImplied,
      // W, R, w => r
      opReadEnabled = false,
      opWriteEnabled = false,
      opInvokeEnabled = false,
      adReadEnabled = false,
      adWriteEnabled = false,
      adInvokeEnabled = false;
    if (that.isEnabled() && !that.isReadonly()) {
      opReadEnabled = max.hasOperatorRead() || max.hasAdminRead() || permissions.hasOperatorRead();
      opWriteEnabled = max.hasOperatorWrite() || max.hasAdminWrite() || permissions.hasOperatorWrite();
      opInvokeEnabled = max.hasOperatorInvoke() || max.hasAdminInvoke() || permissions.hasOperatorInvoke();
      adReadEnabled = max.hasAdminRead() || permissions.hasAdminRead();
      adWriteEnabled = max.hasAdminWrite() || permissions.hasAdminWrite();
      adInvokeEnabled = max.hasAdminInvoke() || permissions.hasAdminInvoke();
    }
    var updatedRow = tplRow({
      rowDescription: that.getRowDescription(),
      opRead: opReadImplied || permissions.hasOperatorRead(),
      opReadDisabled: opReadImplied || !opReadEnabled,
      opReadCellStyle: opReadEnabled ? '' : CELL_DISABLED_STYLE_CLASS,
      opWrite: opWriteImplied || permissions.hasOperatorWrite(),
      opWriteDisabled: opWriteImplied || !opWriteEnabled,
      opWriteCellStyle: opWriteEnabled ? '' : CELL_DISABLED_STYLE_CLASS,
      opInvoke: opInvokeImplied || permissions.hasOperatorInvoke(),
      opInvokeDisabled: opInvokeImplied || !opInvokeEnabled,
      opInvokeCellStyle: opInvokeEnabled ? '' : CELL_DISABLED_STYLE_CLASS,
      adRead: adReadImplied || permissions.hasAdminRead(),
      adReadDisabled: adReadImplied || !adReadEnabled,
      adReadCellStyle: adReadEnabled ? '' : CELL_DISABLED_STYLE_CLASS,
      adWrite: permissions.hasAdminWrite(),
      adWriteDisabled: !adWriteEnabled,
      adWriteCellStyle: adWriteEnabled ? '' : CELL_DISABLED_STYLE_CLASS,
      adInvoke: permissions.hasAdminInvoke(),
      adInvokeDisabled: !adInvokeEnabled,
      adInvokeCellStyle: adInvokeEnabled ? '' : CELL_DISABLED_STYLE_CLASS
    });
    return Promise.resolve(that.$updateRowDom(updatedRow));
  };

  /**
   * construct a baja.Permissions instance from the checkbox values in this editor.
   *
   * @private
   * @returns {baja.Permissions}
   */
  PermissionsRowEditor.prototype.$rowToPermissions = function () {
    var that = this;
    var mask = baja.Permissions.none.getMask(),
      // default to no permissions
      opRead = toBoolean(that.$getOperatorReadElement()),
      opWrite = toBoolean(that.$getOperatorWriteElement()),
      opInvoke = toBoolean(that.$getOperatorInvokeElement()),
      adRead = toBoolean(that.$getAdminReadElement()),
      adWrite = toBoolean(that.$getAdminWriteElement()),
      adInvoke = toBoolean(that.$getAdminInvokeElement());
    if (opRead) {
      mask |= baja.Permissions.OPERATOR_READ;
    }
    if (opWrite) {
      mask |= baja.Permissions.OPERATOR_WRITE;
    }
    if (opInvoke) {
      mask |= baja.Permissions.OPERATOR_INVOKE;
    }
    if (adRead) {
      mask |= baja.Permissions.ADMIN_READ;
    }
    if (adWrite) {
      mask |= baja.Permissions.ADMIN_WRITE;
    }
    if (adInvoke) {
      mask |= baja.Permissions.ADMIN_INVOKE;
    }
    return baja.Permissions.make(mask);
  };
  PermissionsRowEditor.prototype.$getOperatorReadElement = function () {
    return this.jq().find('.opRead');
  };
  PermissionsRowEditor.prototype.$getOperatorWriteElement = function () {
    return this.jq().find('.opWrite');
  };
  PermissionsRowEditor.prototype.$getOperatorInvokeElement = function () {
    return this.jq().find('.opInvoke');
  };
  PermissionsRowEditor.prototype.$getAdminReadElement = function () {
    return this.jq().find('.adRead');
  };
  PermissionsRowEditor.prototype.$getAdminWriteElement = function () {
    return this.jq().find('.adWrite');
  };
  PermissionsRowEditor.prototype.$getAdminInvokeElement = function () {
    return this.jq().find('.adInvoke');
  };
  function toBoolean(htmlCollection) {
    return !!(htmlCollection.length === 1 && htmlCollection[0] && htmlCollection[0].checked);
  }

  /**
   * Update this row's DOM with the passed row content.
   *
   * @param {JQuery} updatedRow
   * @private
   */
  PermissionsRowEditor.prototype.$updateRowDom = function (updatedRow) {
    this.jq().html(updatedRow);
  };

  /**
   * Read the individual child editors and assemble them into a `Permissions`
   *
   * @returns {Promise.<baja.Permissions>} promise to be resolved with a `baja.Permissions` value
   */
  PermissionsRowEditor.prototype.doRead = function () {
    return Promise.resolve(this.$rowToPermissions());
  };
  return PermissionsRowEditor;
});
