/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author JJ Frankovich
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/FileInputEditor
 */
define(['jquery', 'lex!webEditors', 'nmodule/webEditors/rc/fe/baja/BaseEditor'], function ($, lexicons, BaseEditor) {
  'use strict';

  var $lastUploadedFile = null,
    webEditorsLex = lexicons[0];
  //TODO: when refactoring this editor, be very careful to go re-test subclasses
  //e.g. CertificateEditor

  /**
   * FileInputEditor provides a reference to a javascript file chosen by the user.
   * Calls to save and read must conform to baja types (which can't be a javascript file) so
   * those calls return the fileName (for informational purposes). The reference to the actual file
   * is obtained via FileInputEditor.getLastUploadedFile() after FileInputEditor.doSave() has been called.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/FileInputEditor
   */
  var FileInputEditor = function FileInputEditor(params) {
    BaseEditor.call(this, $.extend({
      keyName: 'FileInputEditor'
    }, params));
  };
  FileInputEditor.prototype = Object.create(BaseEditor.prototype);
  FileInputEditor.prototype.constructor = FileInputEditor;

  /**
   * Return the input element.
   *
   * @private
   * @returns {jQuery}
   */
  FileInputEditor.prototype.$getInput = function () {
    return this.jq().children('input');
  };
  FileInputEditor.prototype.$getClearButton = function () {
    return this.jq().find('.clearButton');
  };

  /**
   * Creates a file input
   * @param {jQuery} dom
   */
  FileInputEditor.prototype.doInitialize = function (dom) {
    var that = this,
      input = $('<input type="file">'),
      clearButton = $('<input />').attr({
        type: 'button',
        "class": 'clearButton',
        value: webEditorsLex.get('FileInputEditor.clearButton')
      });
    dom.html(input);
    dom.append(clearButton);
    input.on('change', function (event) {
      that.$updateFile(event);
    });
    clearButton.on('click', function () {
      that.clearButton(); // override this function in the subclasses to clear specific/custom fields
    });
  };

  /**
   * update the $lastChangedFile from the last change event
   * @param {Object} event
   */
  FileInputEditor.prototype.$updateFile = function (event) {
    var that = this,
      files = event.target.files;
    that.$lastChangedFile = files[0];
    that.setModified(true);
  };

  /**
   * Returns the file Contents from the last uploaded file that has been Saved.
   * @returns {Object} Javascript file reference
   */
  FileInputEditor.getLastUploadedFile = function () {
    return $lastUploadedFile;
  };

  /**
   * Read the File name from the widget.
   * @returns {String} FileName of Javascript file if its been set
   */
  FileInputEditor.prototype.doRead = function () {
    var that = this;
    $lastUploadedFile = that.$lastChangedFile;
    if (that.$lastChangedFile) {
      return that.$lastChangedFile.name;
    }
  };

  /**
   * Enables or disables the input element.
   *
   * @param {Boolean} enabled
   */
  FileInputEditor.prototype.doEnabled = function (enabled) {
    this.$getInput().prop('disabled', this.isReadonly() || !enabled);
    this.$getClearButton().prop('disabled', this.isReadonly() || !enabled || this.clearButtonDisabled());
  };

  /**
   * Disables or enables the input element.
   *
   * @param {Boolean} readonly
   */
  FileInputEditor.prototype.doReadonly = function (readonly) {
    this.$getInput().prop('disabled', !this.isEnabled() || readonly);
    this.$getClearButton().prop('disabled', !this.isEnabled() || readonly || this.clearButtonDisabled());
  };

  /**
   * Removes the File Content from the Widget.
   */
  FileInputEditor.prototype.doDestroy = function () {
    this.$lastChangedFile = null;
  };

  /**
   * Save the changed file to the static entry available from FileInputEditor.getLastUploadedFile()
   * Saves the fileName as the String returned
   * @return {String}
   */
  FileInputEditor.prototype.doSave = function () {
    var that = this,
      name = that.$lastChangedFile ? that.$lastChangedFile.name : "";
    $lastUploadedFile = that.$lastChangedFile;
    return name;
  };

  /**
   * Allow FileInputEditor implementations to clear the file selection
   * and related fields
   *
   * @abstract
   */
  FileInputEditor.prototype.clearButton = function () {
    throw new Error('not implemented');
  };

  /**
   * Determine if this editor should have the clear button enabled or disabled.
   * Sub-classes can override this function to control the 'disabled' property of the
   * clear button.
   * @return {boolean} true to enable the button; false otherwise
   */
  FileInputEditor.prototype.clearButtonDisabled = function () {
    return true;
  };
  return FileInputEditor;
});
