/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Aaron Laniewicz
 */

/**
 * API Status: **Private**
 * @since Niagara 4.3
 * @module nmodule/control/rc/fe/TriggerModeEditor
 */
define(['baja!', 'bajaux/commands/ToggleCommandGroup', 'bajaux/events', 'bajaux/util/CommandButtonGroup', 'bajaux/Widget', 'hbs!nmodule/control/rc/fe/templates/TriggerModeEditor', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/fe/baja/DisplayOnlyEditor', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/servlets/registry', 'Promise', 'underscore'], function (baja, ToggleCommandGroup, events, CommandButtonGroup, Widget, template, BaseEditor, DisplayOnlyEditor, fe, registry, Promise, _) {
  'use strict';

  var DESTROY_EVENT = events.DESTROY_EVENT,
    INITIALIZE_EVENT = events.INITIALIZE_EVENT,
    LOAD_EVENT = events.LOAD_EVENT,
    ENABLE_EVENT = events.ENABLE_EVENT,
    DISABLE_EVENT = events.DISABLE_EVENT,
    MODIFY_EVENT = events.MODIFY_EVENT,
    READONLY_EVENT = events.READONLY_EVENT,
    WRITABLE_EVENT = events.WRITABLE_EVENT;

  /**
   * Field editor for working with `control:TriggerMode` instances.
   * The following two properties are supported in the arguments:
   * - `properties.hideTriggerModeList` a String list of trigger modes that are to be hidden in the format of
   * "module1:type1;module2:type2;...".
   * - `properties.showDay|showDays` if either is set to true the Interval and Daily Editors will show
   * days as well as hours in their RelTimeEditor.
   * @class
   * @alias module:nmodule/control/rc/fe/TriggerModeEditor
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   */
  var TriggerModeEditor = function TriggerModeEditor() {
    var params = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    var that = this;
    BaseEditor.apply(this, arguments);
    if (that.getFormFactor() === Widget.formfactor.mini) {
      DisplayOnlyEditor.$mixin(this);
    } else {
      that.validators().add(function () {
        return that.$getValueEditor().validate();
      });
    }
  };
  TriggerModeEditor.prototype = Object.create(BaseEditor.prototype);
  TriggerModeEditor.prototype.constructor = TriggerModeEditor;

  /**
   * Initialize the `TriggerModeEditor`.
   *
   * @param {jQuery} dom
   */
  TriggerModeEditor.prototype.doInitialize = function (dom) {
    var that = this;

    //Disable events from bubbling up
    dom.on([DESTROY_EVENT, INITIALIZE_EVENT, LOAD_EVENT, ENABLE_EVENT, DISABLE_EVENT, READONLY_EVENT, WRITABLE_EVENT].join(' '), '.bajaux-initialized', false);
    dom.html(template());
    dom.on(MODIFY_EVENT, '.bajaux-initialized', function () {
      that.setModified(true);
      return false;
    });
    that.$commandList = [];
    return registry.getTypes({
      targetType: 'control:TriggerMode',
      showAbstract: false,
      showInterface: false
    }).then(function (types) {
      that.$triggerTypes = _.pluck(types, 'type');
      return baja.importTypes(that.$triggerTypes);
    }).then(function () {
      that.$toggleCommandGroup = new ToggleCommandGroup({
        onChange: function onChange(value) {
          return that.$loadValue(baja.$(that.$triggerTypes[value])).then(function () {
            return that.setModified(true);
          });
        },
        commands: that.$triggerTypes.map(function (type, index) {
          that.$commandList[index] = type;
          return {
            displayName: baja.$(type).getDisplayName(),
            value: index
          };
        })
      });
    }).then(function () {
      return fe.buildFor({
        type: CommandButtonGroup,
        value: that.$getToggleCommandGroup(),
        dom: that.$getToggleElement(),
        properties: {
          onDisabled: "hide"
        },
        initializeParams: {
          toggleGroup: true
        }
      });
    });
  };

  /**
   * Load a `TriggerMode`.
   *
   * @param {baja.Simple} tm - any Type that inherits the `TriggerMode` Type.
   * @returns {Promise}
   */
  TriggerModeEditor.prototype.doLoad = function (tm) {
    return this.$loadValue(tm);
  };

  /**
   * Read a new `TriggerMode`.
   *
   * @returns {Promise}
   */
  TriggerModeEditor.prototype.doRead = function () {
    return this.$getValueEditor().read();
  };

  /**
   * Set all child editors readonly/writable.
   *
   * @param {Boolean} readonly
   * @returns {Promise}
   */
  TriggerModeEditor.prototype.doReadonly = function (readonly) {
    return this.getChildWidgets().setAllReadonly(readonly);
  };

  /**
   * Set all child editors enabled/disabled.
   *
   * @param {Boolean} enabled
   * @returns {Promise}
   */
  TriggerModeEditor.prototype.doEnabled = function (enabled) {
    return this.getChildWidgets().setAllEnabled(enabled);
  };

  /**
   * Destroy all child editors.
   *
   * @returns {Promise}
   */
  TriggerModeEditor.prototype.doDestroy = function () {
    return this.getChildWidgets().destroyAll();
  };
  TriggerModeEditor.prototype.valueToString = function (value) {
    var str = '';
    if (!baja.hasType(value, 'control:ManualTriggerMode')) {
      str += value.getDisplayName() + ': ';
    }
    return str + String(value);
  };

  /**
   * Load a `TriggerMode`.
   *
   * @private
   * @param {baja.Simple} tm - any Type that inherits the `TriggerMode` Type.
   * @returns {Promise}
   */
  TriggerModeEditor.prototype.$loadValue = function (tm) {
    var that = this,
      toggleIndex,
      valueEd = that.$getValueEditor(),
      originalValues = that.$originalValues || (that.$originalValues = {}),
      typeName = String(tm.getType()),
      originalTM = originalValues[typeName];
    toggleIndex = _.indexOf(that.$triggerTypes, typeName);
    function storeCurrentValue() {
      if (valueEd) {
        return that.read().then(function (result) {
          that.$originalValues[String(result.getType())] = result;
        })["catch"](function () {}).then(function () {
          return valueEd.destroy();
        });
      }
    }
    return Promise.resolve(storeCurrentValue()).then(function () {
      // get the properties, if properties are not set use the default values
      var props = that.properties();
      var hideTriggerModeList = (props.getValue("hideTriggerModeList") || "").split(";");
      var showDays = !!(props.getValue("showDay") || props.getValue("showDays"));
      var min = props.getValue('min');
      var max = props.getValue('max');
      if (originalTM) {
        tm = originalTM;
      }
      for (var entry = 0; entry < that.$toggleCommandGroup.size(); entry++) {
        var command = that.$getToggleCommand(entry);
        var commandType = that.$commandList[entry];
        if (hideTriggerModeList.includes(commandType) && toggleIndex !== entry) {
          command.setEnabled(false);
        } else {
          command.setEnabled(true);
        }
      }
      that.$getToggleCommand(toggleIndex).setSelected(true);
      return fe.getConstructors(String(tm.getType())).then(function (constructors) {
        return fe.buildFor({
          value: tm,
          // TODO: revise after completion of NCCB-23555
          type: _.find(constructors, function (c) {
            return c !== TriggerModeEditor;
          }),
          dom: that.$getValueElement(),
          formFactor: 'mini',
          properties: {
            showDays: showDays,
            min: min,
            max: max
          }
        });
      });
    });
  };
  TriggerModeEditor.prototype.$getToggleElement = function () {
    return this.jq().children('.key-toggle');
  };
  TriggerModeEditor.prototype.$getToggleEditor = function () {
    return this.$getToggleElement().data('widget');
  };
  TriggerModeEditor.prototype.$getToggleCommandGroup = function () {
    return this.$toggleCommandGroup;
  };
  TriggerModeEditor.prototype.$getToggleCommand = function (index) {
    return this.$getToggleCommandGroup().get(index);
  };
  TriggerModeEditor.prototype.$getValueElement = function () {
    return this.jq().children('.key-value');
  };
  TriggerModeEditor.prototype.$getValueEditor = function () {
    return this.$getValueElement().data('widget');
  };
  return TriggerModeEditor;
});
