/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Aaron Laniewicz
 */

/**
 * API Status: **Private**
 * @since Niagara 4.3
 * @module nmodule/control/rc/baja/DailyTriggerMode
 */
define(['baja!', 'lex!control'], function (baja, lexs) {
  'use strict';

  var objUtil = require('bajaScript/baja/obj/objUtil');
  var DaysOfWeekBits = baja.DaysOfWeekBits,
    RelTime = baja.RelTime,
    Simple = baja.Simple,
    Time = baja.Time,
    bajaDef = baja.def,
    callSuper = baja.callSuper,
    objectify = baja.objectify,
    subclass = baja.subclass,
    cacheDecode = objUtil.cacheDecode,
    cacheEncode = objUtil.cacheEncode,
    controlLex = lexs[0],
    dailyDisplayNameLex = controlLex.get({
      key: 'trigger.daily',
      def: 'Daily'
    });

  /**
   * Represents a `control:DailyTriggerMode` in BajaScript.
   *
   * When creating a `Simple`, always use the `make()` method instead of
   * creating a new object.
   *
   * @class
   * @extends baja.Simple
   * @alias module:nmodule/control/rc/baja/DailyTriggerMode
   */
  var DailyTriggerMode = function DailyTriggerMode() {
    callSuper(DailyTriggerMode, this, arguments);
  };
  subclass(DailyTriggerMode, Simple);

  /**
   * Make a new `DailyTriggerMode` instance.
   *
   * @param {Object} [obj] - Object Literal for the method's arguments.
   *
   * @param {baja.Time} [obj.timeOfDay] - time of day that the trigger will
   * fire.
   *
   * @param {baja.DaysOfWeekBits} [obj.daysOfWeek] - the days of the week that
   * the trigger will fire.
   *
   * @param {baja.RelTime} [obj.randomization] - the randomization interval.
   * When the next time is calculated, a random amount of time between zero
   * milliseconds and the randomization interval is added to the timeOfDay.
   *
   * @returns {module:nmodule/control/rc/baja/DailyTriggerMode}
   */
  DailyTriggerMode.make = function (obj) {
    var dtm = new DailyTriggerMode();
    obj = objectify(obj);
    dtm.$timeOfDay = bajaDef(obj.timeOfDay, Time.make({
      hour: 2
    }));
    dtm.$daysOfWeek = bajaDef(obj.daysOfWeek, DaysOfWeekBits.DEFAULT);
    dtm.$randomization = bajaDef(obj.randomization, RelTime.make());
    return dtm;
  };

  /**
   * Make a new `DailyTriggerMode` instance.
   *
   * @see module:nmodule/control/rc/baja/DailyTriggerMode.make
   */
  DailyTriggerMode.prototype.make = function () {
    return DailyTriggerMode.make.apply(DailyTriggerMode, arguments);
  };

  /**
   * Decode a `DailyTriggerMode` from a `String`.
   *
   * @param {String} str - an encoded `DailyTriggerMode`.
   *
   * @returns {module:nmodule/control/rc/baja/DailyTriggerMode}
   */
  DailyTriggerMode.prototype.decodeFromString = cacheDecode(function (str) {
    var tokens = str.split(';');
    return DailyTriggerMode.make({
      timeOfDay: Time.DEFAULT.decodeFromString(tokens[0]),
      daysOfWeek: DaysOfWeekBits.DEFAULT.decodeFromString(tokens[1]),
      randomization: RelTime.DEFAULT.decodeFromString(tokens[2])
    });
  });

  /**
   * Encode the `DailyTriggerMode` (itself) to a `String`.
   *
   * @returns {String}
   */
  DailyTriggerMode.prototype.encodeToString = cacheEncode(function () {
    var that = this;
    return that.$timeOfDay.encodeToString() + ';' + that.$daysOfWeek.encodeToString() + ';' + that.$randomization.encodeToString();
  });

  /**
   * Default `DailyTriggerMode` instance.
   *
   * @type {module:nmodule/control/rc/baja/DailyTriggerMode}
   */
  DailyTriggerMode.DEFAULT = DailyTriggerMode.make();

  /**
   * Return the `String` representation of the `DailyTriggerMode` (itself).
   *
   * @returns {String}
   */
  DailyTriggerMode.prototype.toString = function () {
    var that = this,
      str = String(that.$timeOfDay) + ' ' + String(that.$daysOfWeek),
      randomization = that.$randomization;

    // Append randomization if it is not 0
    if (randomization.getMillis()) {
      str += ' +~' + String(randomization);
    }
    return str;
  };

  /**
   * Get the display name for this trigger from lexicon.
   *
   * @returns {String}
   */
  DailyTriggerMode.prototype.getDisplayName = function () {
    return dailyDisplayNameLex;
  };

  /**
   * Get the time of day that the trigger will fire.
   *
   * @returns {baja.Time}
   */
  DailyTriggerMode.prototype.getTimeOfDay = function () {
    return this.$timeOfDay;
  };

  /**
   * Get the days of the week that the trigger will fire.
   *
   * @returns {baja.DaysOfWeekBits}
   */
  DailyTriggerMode.prototype.getDaysOfWeek = function () {
    return this.$daysOfWeek;
  };

  /**
   * Get the randomization interval.  When the next time is calculated, a random
   * amount of time between zero milliseconds and the randomization interval is
   * added to the timeOfDay.
   *
   * @returns {baja.RelTime}
   */
  DailyTriggerMode.prototype.getRandomization = function () {
    return this.$randomization;
  };
  return DailyTriggerMode;
});
