/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * @module bajaux/commands/ToggleCommand
 */
define(['bajaux/commands/Command', 'bajaux/events'], function (Command, events) {
  "use strict";

  /**
   * A ToggleCommand adds an additional state variable allowing it to be
   * turned on or off. If no function is specified in the Constructor, the default
   * behavior is to simply call 'toggle'.
   *
   * @class
   * @alias module:bajaux/commands/ToggleCommand
   * @extends module:bajaux/commands/Command
   * @param {object} [params] all parameters compatible with {@link module:bajaux/commands/Command}
   * @param {boolean} [params.selected] set to `true` to cause this command to be
   * pre-selected
   */
  var ToggleCommand = function ToggleCommand(params) {
    Command.apply(this, arguments);

    // Set selected flag (defaults to false)
    this.$selected = params && params.constructor === Object && typeof params.selected === "boolean" ? params.selected : false;
  };
  ToggleCommand.prototype = Object.create(Command.prototype);
  ToggleCommand.prototype.constructor = ToggleCommand;

  /**
   * Gets this command's selected status.
   * @returns {Boolean}
   */
  ToggleCommand.prototype.isSelected = function isSelected() {
    return this.$selected;
  };

  /**
   * Sets this command's selected status. Triggers a
   * `bajaux:changecommand`.
   * @param {Boolean} selected
   * @param {Object} [params] Some optional parameters to pass through 
   * to the events that are fired.
   */
  ToggleCommand.prototype.setSelected = function setSelected(selected, params) {
    var that = this;
    selected = !!selected;
    if (that.$selected !== selected) {
      that.$selected = selected;
      that.trigger(events.command.CHANGE_EVENT);
      that.trigger(events.command.SELECTION_EVENT, params);
    }
  };

  /**
   * Selects if deselected, or deselects if selected. Triggers a
   * `bajaux:changecommand`.
   *
   * @param {Object} [params] Some optional parameters to pass through to 
   * setting the selection.
   */
  ToggleCommand.prototype.toggle = function toggle(params) {
    this.setSelected(!this.isSelected(), params);
  };

  /**
   * Always returns true.
   */
  ToggleCommand.prototype.isToggleCommand = function isToggleCommand() {
    return true;
  };
  return ToggleCommand;
});
