/**
 * @copyright 2021 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/httpClient/rc/ClientSelectionEditor
 */
define(['baja!', 'log!nmodule.httpClient.rc.ClientSelectionEditor', 'jquery', 'Promise', 'underscore', 'bajaux/mixin/subscriberMixIn', 'nmodule/js/rc/switchboard/switchboard', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'hbs!nmodule/httpClient/rc/template/ClientSelectionEditor'], function (baja, log, $, Promise, _, subscriberMixIn, switchboard, BaseEditor, template) {
  'use strict';

  var logError = log.severe.bind(log),
    logFine = log.fine.bind(log);

  /**
   * A field editor for editing a list of selected http clients.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/httpClient/rc/ClientSelectionEditor
   */
  var ClientSelectionEditor = function ClientSelectionEditor(params) {
    BaseEditor.call(this, _.extend({
      keyName: 'ClientSelectionEditor',
      moduleName: 'httpClient'
    }, params));
    subscriberMixIn(this);
    switchboard(this, {
      '$updateClientSelection': {
        allow: 'oneAtATime',
        onRepeat: 'preempt'
      }
    });
  };
  ClientSelectionEditor.prototype = Object.create(BaseEditor.prototype);
  ClientSelectionEditor.prototype.constructor = ClientSelectionEditor;

  /**
   * Setup the button callbacks in the dom.
   *
   * @param {JQuery} dom
   */
  ClientSelectionEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.addClass('ClientSelectionEditor');
    dom.html(template());
    dom.on('click', '.add-client-button', function () {
      that.$addSelectedClient()["catch"](logError);
      return false;
    });
    dom.on('click', '.delete-client-button', function (e) {
      that.$deleteSelectedClient(e.target.getAttribute('data-client-to-delete'))["catch"](logError);
      return false;
    });
  };

  /**
   * Load the target client list into the dropdown and selected clients list.
   *
   * @param {baja.Component} value an instance of 'httpClient:TargetClientList' to populate from.
   * @returns {Promise}
   */
  ClientSelectionEditor.prototype.doLoad = function (value) {
    var that = this;
    that.$clientTargetList = value;
    return that.$loadClientMap(that.$clientTargetList).then(function () {
      return that.$updateClientSelection();
    })["catch"](logError);
  };

  /**
   * Read the current client target list.
   * @returns {baja.Component} the loaded 'httpClient:TargetClientList' instance value.
   */
  ClientSelectionEditor.prototype.doRead = function () {
    return this.$clientTargetList;
  };

  /**
   * Save the TargetClientList value.
   * @param {baja.Component} readValue - a 'httpClient:TargetClientList' instance value
   * @returns {Promise}
   */
  ClientSelectionEditor.prototype.doSave = function (readValue) {
    var that = this,
      batch = new baja.comm.Batch(),
      promises = [];
    readValue.getSlots().properties().dynamic().each(function (prop) {
      promises.push(readValue.remove({
        slot: prop,
        batch: batch
      }));
    });
    that.$getAlreadySelectedClients().forEach(function (selectedClientHandle) {
      if (selectedClientHandle) {
        promises.push(readValue.add({
          slot: "client?",
          value: baja.Ord.make(selectedClientHandle),
          batch: batch
        }));
      }
    });
    that.$alreadySelectedClients = null; // force refresh

    return batch.commit(Promise.all(promises)).then(function () {
      return BaseEditor.prototype.doSave.apply(that, arguments);
    });
  };

  /**
   * Enables or disables the select dropdown.
   *
   * @param {Boolean} enabled
   */
  ClientSelectionEditor.prototype.doEnabled = function (enabled) {
    var disabled = this.isReadonly() || !enabled;
    this.$getAddClientSelect().prop('disabled', disabled);
    return this.getChildEditors().setAllEnabled(enabled);
  };

  /**
   * Disables or enables the select dropdown.
   *
   * @param {Boolean} readonly
   */
  ClientSelectionEditor.prototype.doReadonly = function (readonly) {
    var disabled = !this.isEnabled() || readonly;
    this.$getAddClientSelect().prop('disabled', disabled);
    return this.getChildEditors().setAllReadonly(readonly);
  };

  /**
   * Clean up after this editor
   * @returns {Promise}
   */
  ClientSelectionEditor.prototype.doDestroy = function () {
    var that = this;
    that.jq().removeClass('ClientSelectionEditor');
    return Promise.resolve(that.$subscriber && that.$subscriber.unsubscribeAll()).then(function () {
      return that.getChildWidgets().destroyAll();
    });
  };

  /**
   * Get the handle ords of the clients that have already been selected.
   *
   * @private
   * @returns {Array.<String>}
   */
  ClientSelectionEditor.prototype.$getAlreadySelectedClients = function () {
    if (!this.$alreadySelectedClients) {
      var alreadySelected = [];
      this.$clientTargetList.getSlots().properties().dynamic().each(function (prop) {
        alreadySelected.push(prop.$val.toString()); // client list children will be BOrd's
      });
      this.$alreadySelectedClients = alreadySelected;
    }
    return this.$alreadySelectedClients;
  };

  /**
   * Load the list of all possible http clients.
   *
   * @private
   * @param {baja.Component} clientTargetList a 'httpClient:TargetClientList' instance value to load from.
   * @returns {Promise}
   */
  ClientSelectionEditor.prototype.$loadClientMap = function (clientTargetList) {
    var that = this;
    return that.$findAllClients(clientTargetList).then(function (clients) {
      logFine('Found clients: ' + clients);
      that.$clientMap = clients;
    })["catch"](logError);
  };

  /**
   * Fetch the list of all possible http clients from the station.
   *
   * @private
   * @param {baja.Component} clientTargetList a 'httpClient:TargetClientList' instance value to call upon.
   * @returns {Promise}
   */
  ClientSelectionEditor.prototype.$findAllClients = function (clientTargetList) {
    return baja.rpc({
      ord: clientTargetList.getNavOrd(),
      methodName: 'findAllClients'
    });
  };

  /**
   * Lookup the client handle associated with the client display name.
   *
   * @private
   * @param {String} selectedClient the client display name to get the handle for.
   * @returns the matching client handle as a string
   */
  ClientSelectionEditor.prototype.$getClientHandle = function (selectedClient) {
    var that = this;
    var matchingHandle = null;
    if (that.$clientMap) {
      Object.keys(that.$clientMap).forEach(function (handleOrd) {
        var clientDisplayName = that.$clientMap[handleOrd];
        if (clientDisplayName === _.unescape(selectedClient)) {
          matchingHandle = handleOrd;
        }
      });
    }
    return matchingHandle;
  };

  /**
   * Update the contents of the client dropdown and the list
   * of already selected clients.
   *
   * @private
   */
  ClientSelectionEditor.prototype.$updateClientSelection = function () {
    var that = this,
      addClientButtonElement = that.jq().find('.add-client-button'),
      enableAddButton = false,
      alreadySelectedClientsElement = that.jq().find('.selected-clients'),
      addClientDropdown = that.$getAddClientSelect();
    alreadySelectedClientsElement.empty();
    addClientDropdown.empty();
    addClientDropdown.append('<option class="ux-option" value=""></option>');
    if (that.$clientTargetList && that.$clientMap) {
      var alreadySelected = that.$getAlreadySelectedClients();
      logFine('Already selected clients: ' + alreadySelected.length);
      Object.keys(that.$clientMap).forEach(function (handleOrd) {
        var clientDisplayName = that.$clientMap[handleOrd];
        var safeDisplayName = _.escape(clientDisplayName);

        // add each client either to the dropdown of clients, or the div containing previously selected
        if (alreadySelected.indexOf(handleOrd) < 0) {
          addClientDropdown.append('<option class="ux-option" value="' + safeDisplayName + '">' + safeDisplayName + '</option>');
          enableAddButton = true;
        } else {
          alreadySelectedClientsElement.append('<div>' + '<span>' + safeDisplayName + '</span>' + '<span data-client-to-delete="' + safeDisplayName + '" class="delete-client-button icon-icons-x16-delete" ' + '</div>');
        }
      });
    }
    addClientButtonElement.toggle(enableAddButton);
  };

  /**
   * Add the currently selected client to the selection list, and update the editor.
   *
   * @private
   */
  ClientSelectionEditor.prototype.$addSelectedClient = function () {
    if (this.isReadonly() || !this.isEnabled()) {
      return Promise.resolve();
    }
    var addSlotDropdown = this.$getAddClientSelect(),
      clientToAdd = addSlotDropdown.val();
    if (!clientToAdd) {
      return Promise.resolve();
    }
    var clientHandle = this.$getClientHandle(clientToAdd);
    if (clientHandle) {
      this.$getAlreadySelectedClients().push(clientHandle);
      this.setModified(true);
    }
    return this.$updateClientSelection();
  };

  /**
   * Delete the specified client from the selection list, and update the editor
   *
   * @private
   * @param {String} clientToDelete the display name of the client to delete from the selection list.
   */
  ClientSelectionEditor.prototype.$deleteSelectedClient = function (clientToDelete) {
    if (this.isReadonly() || !this.isEnabled()) {
      return Promise.resolve();
    }
    var handleToDelete = this.$getClientHandle(clientToDelete);
    var alreadySelectedClients = this.$getAlreadySelectedClients(),
      indexToRemove = alreadySelectedClients.indexOf(handleToDelete);
    if (indexToRemove < 0) {
      return Promise.resolve();
    }
    alreadySelectedClients.splice(indexToRemove, 1);
    this.setModified(true);
    return this.$updateClientSelection();
  };

  /**
   * Get the 'addClient' select element.
   *
   * @private
   * @returns {JQuery}
   */
  ClientSelectionEditor.prototype.$getAddClientSelect = function () {
    return this.jq().find('select.client-selection');
  };
  return ClientSelectionEditor;
});
