/*
 * Copyright 2023 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.workbench.commands;

import javax.baja.gx.BImage;
import javax.baja.gx.BSize;
import javax.baja.sys.BIcon;
import javax.baja.sys.BString;
import javax.baja.sys.Context;
import javax.baja.ui.BDialog;
import javax.baja.ui.BWidget;
import javax.baja.ui.Command;
import javax.baja.ui.CommandArtifact;
import javax.baja.ui.pane.BConstrainedPane;
import javax.baja.ui.util.UiLexicon;
import javax.baja.util.Lexicon;

import com.tridium.util.HtmlUtils;
import com.tridium.workbench.web.browser.BWebBrowser;

/**
 * Command which renders the given text in a browser, allowing limited HTML formatting.
 * These are the elements supported as of Niagara 4.14:
 * - &lt;b&gt;
 * - &lt;code&gt;
 * - &lt;i&gt;
 * - &lt;li&gt;
 * - &lt;p&gt;
 * - &lt;ul&gt;
 *
 * @author Nick Dodd
 * @since Niagara 4.14
 */
public class HtmlHelpCommand
  extends Command
{
  public HtmlHelpCommand(BWidget parent, String title, String helpText, BSize size, Context context)
  {
    this(parent, null, title, helpText, size, context);
  }

  public HtmlHelpCommand(BWidget parent, String label, String title, String helpText,
    BSize size, Context context)
  {
    super(parent, label);
    icon = ICON;
    this.parent = parent;
    this.title = title;
    this.helpText = HtmlUtils.getLimitedHtml(helpText);
    this.size = size;
    this.context = context;
  }

  /**
   * Display the default "Help" label next to the icon.
   */
  public void withDefaultLabel()
  {
    label = HELP_LABEL;
  }

  @Override
  public CommandArtifact doInvoke()
    throws Exception
  {
    BWebBrowser browser = new BWebBrowser();
    browser.load(BString.make(formatHtmlDescription(helpText)), context);

    BConstrainedPane cons = new BConstrainedPane(browser);
    cons.setMinSize(size.width(), size.height());
    cons.setMaxSize(size.width(), size.height());

    BDialog.open(
      parent,
      title,
      cons,
      BDialog.OK
    );
    return null;
  }

  /*
   * Get the BFormat class description as an HTML string with the appropriate styling.
   */
  private static String formatHtmlDescription(String description)
  {
    return String.format(
      "<html>" +
        "<head>" +
        "<link rel='stylesheet' href='module://bajaui/doc/style.css' type='text/css'/>" +
        "</head>" +
        "<body>%s</body>" +
        "</html>",
      description);
  }

  private final BWidget parent;
  private final String title;
  private final String helpText;
  private final BSize size;
  private final Context context;

  private static final Lexicon LEX = UiLexicon.bajaui();
  private static final BImage ICON = BImage.make(BIcon.std("questionMark.png"));
  private static final String HELP_LABEL = LEX.getText("menu.help.label");
}
