/*
 * Copyright 2013 Tridium, Inc. All Rights Reserved.
 */

package com.tridium.testng;

import static org.testng.Assert.assertTrue;

import static com.tridium.testng.TestUtil.waitFor;

import java.lang.reflect.Field;

import javax.baja.naming.BOrd;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.role.BRoleService;
import javax.baja.security.BPermissions;
import javax.baja.security.BPermissionsMap;
import javax.baja.sync.BProxyComponentSpace;
import javax.baja.sys.BComponent;
import javax.baja.sys.BStation;
import javax.baja.sys.BString;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.test.BTestNgStation;
import javax.baja.test.TestException;
import javax.baja.user.BUser;
import javax.baja.user.BUserService;

import org.eclipse.jetty.server.Server;
import org.testng.Assert;

import com.tridium.authn.BAuthenticationService;
import com.tridium.crypto.core.cert.CertUtils;
import com.tridium.fox.sys.BFoxService;
import com.tridium.fox.sys.BFoxSession;
import com.tridium.jetty.BJettyWebServer;
import com.tridium.nd.BNiagaraNetwork;
import com.tridium.ui.NullUiEnv;
import com.tridium.ui.UiEnv;

/**
 * Test support base class that allows configuring a
 * station for web and fox testing.
 *
 * @author Dan Heine on 2013-06-14
 * @since Niagara 4.0
 */
@NiagaraType
public abstract class BStationTestBase
  extends BTestNgStation
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $com.tridium.testng.BStationTestBase(2979906276)1.0$ @*/
/* Generated Wed Jan 05 17:05:31 EST 2022 by Slot-o-Matic (c) Tridium, Inc. 2012-2022 */

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BStationTestBase.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

////////////////////////////////////////////////////////////////
// Utilities and Helpers
////////////////////////////////////////////////////////////////

  /**
   * Connect to the local (in-process) test station via Fox using the specified account.
   *
   * @return Fox session
   * @throws Exception
   */
  protected BFoxSession connect(String userName, String password)
    throws Exception
  {
    return (BFoxSession)super.connect(userName, password, MAX_FOX_CONNECTION_ATTEMPTS);
  }

  protected void configureTestStation(BStation station, String stationName, int webPort, int foxPort)
    throws Exception
  {
    // Replace default authentication and fox services with the ones available here.
    // The methods that create them here are not private, and there are test classes
    // that call and override them.
    makeDefaultAuthFoxServices = false;
    BComponent services = station.getServices();

    services.add(AUTH_SERVICE, makeAuthService());
    services.add(FOX_SERVICE, makeFoxService(foxPort));

    super.configureTestStation(station, stationName, webPort, foxPort);
  }

  /**
   * Create an authentication service with a Digest, LegacyDigest,
   * and Basic authentication scheme.
   *
   * @return An authentication service
   */
  protected BAuthenticationService makeAuthService()
  {
    BAuthenticationService authService = new BAuthenticationService();

    authService.get("authenticationSchemes");

    return authService;
  }

  /**
   * Create a Fox service. This will have the fox authentication policy
   * set to "BASIC" to work in tandem with webservice "cookie digest"
   * authentication scheme.
   *
   * @param foxPort The port number to use for the FOX service (not FOXS)
   * @return The component for a Fox service set to the specified port and BASIC authentication
   * @throws Exception
   */
  public BComponent makeFoxService(int foxPort)
    throws Exception
  {
    BFoxService foxSvc = new BFoxService();

    foxSvc.set("foxsCert", BString.make(CertUtils.FACTORY_CERT_ALIAS));
    foxSvc.getCertAliasAndPassword().resetAliasAndPassword();
    foxSvc.getFoxPort().setPublicServerPort(foxPort);

    return foxSvc;
  }

  /**
   * Create a RoleService with roles for  test users.
   *
   * @return BRoleService with 'superUser', 'TestAdmin, 'TestOperator',
   * TestVisitor, TestDelimitedUser.
   * @throws Exception
   */
  public BComponent makeRoleService()
    throws Exception
  {
    BRoleService roleService = new BRoleService();

    addRole(roleService, TEST_SUPER_USER, BPermissionsMap.SUPER_USER);
    addRole(roleService, TEST_ADMIN_USER, map(BPermissions.make("rwiRWI")));
    addRole(roleService, TEST_OPERATOR_USER, map(BPermissions.make("rwi")));
    addRole(roleService, TEST_VISITOR_USER, map(BPermissions.make("r")));
    addRole(roleService, TEST_DELIMITED_USER, BPermissionsMap.SUPER_USER);

    return roleService;
  }

  /**
   * Create a user service with test users of various privileges.
   *
   * @return BUserService with 'superUser', 'TestAdmin, 'TestOperator'.
   * All passwords are set to 'Test@1234_5678'. Except the TEST_DELIMITED_USER which
   * has a password containing a colon (see: TEST_DELIMITED_PASS).
   * @throws Exception
   */
  public BComponent makeUserService()
    throws Exception
  {
    // Create service
    BUserService userService = new BUserService();

    addUser(userService, TEST_SUPER_USER);
    addUser(userService, TEST_ADMIN_USER);
    addUser(userService, TEST_OPERATOR_USER);
    addUser(userService, TEST_VISITOR_USER);
    addUser(userService, TEST_DELIMITED_USER, TEST_DELIMITED_PASS);

    return userService;
  }

  /**
   * Connect to the local (in-process) test station via Fox using the specified account
   * with the specified number of attempts.
   *
   * @return Fox session
   * @throws Exception
   */
  protected BFoxSession connect(String userName, String password, int maxAttempts)
    throws Exception
  {
    return (BFoxSession)super.connect(userName, password, maxAttempts);
  }

  protected BOrd localOrd(String query)
  {
    return BOrd.make(String.format("local:|%s", query));
  }

  protected BOrd remoteOrd(String query)
  {
    return BOrd.make(String.format("local:|fox:%d|%s", foxPort, query));
  }

  protected BOrd remoteSlotPath(String slotPath)
  {
    return BOrd.make(String.format("local:|fox:%d|station:|slot:%s", foxPort, slotPath));
  }

  protected BOrd localSlotPath(String slotPath)
  {
    return BOrd.make(String.format("local:|station:|slot:%s", slotPath));
  }

  protected void sync()
    throws Exception
  {
    BOrd.make(String.format("local:|fox:%d|station:", foxPort)).get().as(BProxyComponentSpace.class).sync();
  }

  protected void pause(int ms)
  {
    try
    {
      System.out.println("      Pausing " + ms + "ms...");
      Thread.sleep(ms);
    }
    catch (InterruptedException e)
    {
      throw new TestException(e.toString());
    }
  }

////////////////////////////////////////////////////////////////
// Utils
////////////////////////////////////////////////////////////////

  protected void waitForWebServerStart()
  {
    //Wait for the webserver to complete its startup sequence
    try
    {
      Thread.sleep(1000);
    }
    catch (Exception ignored)
    {
    }

    assertTrue(
      waitFor(() -> getWebService().getWebServer() != null, 10000, 100),
      "Failed to start Jetty webserver in a timely manner"
    );
    BJettyWebServer jettyWebServer = (BJettyWebServer)getWebService().getWebServer();
    assertTrue(
      waitFor(() -> "started".equalsIgnoreCase(jettyWebServer.getServerState()), 10000, 100),
      "Failed to start Jetty webserver in a timely manner"
    );

    // can't use PA here so have to do this the hard way
    Server jettyServer;
    try
    {
      Field jettyField = BJettyWebServer.class.getDeclaredField("jetty");
      jettyField.setAccessible(true);
      jettyServer = (Server)jettyField.get(jettyWebServer);
    }
    catch (Exception e)
    {
      throw new IllegalArgumentException("can't get value of jetty", e);
    }
    assertTrue(waitFor(jettyServer::isRunning, 10000, 100), "Failed to start Jetty webserver in a timely manner");
  }

  protected void startWebServer()
  {
    //Enabled and start the server
    getWebService().setEnabled(true);

    //NOTE: Toggling the enable state of the server back on will automatically schedule a "start"
    //jettyWebServer.doStartWebServer();
    //Wait for the scheduled action to fire
    try
    {
      Thread.sleep(2500);
    }
    catch (Exception ignored)
    {
    }

    assertTrue(
      waitFor(() -> getWebService().getWebServer() != null, 10000, 100),
      "Failed to start Jetty webserver in a timely manner"
    );
    BJettyWebServer jettyWebServer = (BJettyWebServer)getWebService().getWebServer();
    assertTrue(
      waitFor(() -> "started".equalsIgnoreCase(jettyWebServer.getServerState()), 10000, 100),
      "Failed to start Jetty webserver in a timely manner"
    );

    // can't use PA here so have to do this the hard way
    Server jettyServer;
    try
    {
      Field jettyField = BJettyWebServer.class.getDeclaredField("jetty");
      jettyField.setAccessible(true);
      jettyServer = (Server)jettyField.get(jettyWebServer);
    }
    catch (Exception e)
    {
      throw new IllegalArgumentException("can't get value of jetty", e);
    }
    assertTrue(waitFor(jettyServer::isRunning, 10000, 100), "Failed to start Jetty webserver in a timely manner");
  }

  protected void stopWebServer()
  {
    //Disable and stop the server

    //Retain a reference to the jetty server before it shuts down so we can get status, avoid NPE
    BJettyWebServer jettyWebServer = (BJettyWebServer)getWebService().getWebServer();

    // can't use PA here so have to do this the hard way
    Server jettyServer;
    try
    {
      Field jettyField = BJettyWebServer.class.getDeclaredField("jetty");
      jettyField.setAccessible(true);
      jettyServer = (Server)jettyField.get(jettyWebServer);
    }
    catch (Exception e)
    {
      throw new IllegalArgumentException("can't get value of jetty", e);
    }

    getWebService().setEnabled(false);
    //NOTE: Toggling the enable state of the server back on will automatically schedule a "stop",
    // babysit the shutdown so we don't wait forever
    //PA.invokeMethod(jettyWebServer, "stopWebServer()");
    //Wait for the scheduled action to fire
    try
    {
      Thread.sleep(2500);
    }
    catch (Exception ignored)
    {
    }

    //Make sure the internal Jetty server has stopped
    if (jettyServer != null)
    {
      if (!waitFor(() -> !jettyServer.isRunning(), 10000, 100))
      {
        jettyServer.destroy();
        Assert.fail("Failed to stop the Jetty WebServer in a timely fashion");
      }
    }

    //Make sure the BObject version of the server has shutdown
    if (jettyWebServer != null)
    {
      if (!waitFor(() -> "stopped".equalsIgnoreCase(jettyWebServer.getServerState()), 10000, 100))
      {
        Assert.fail("Failed to stop the Jetty WebServer in a timely fashion");
      }
    }
  }

  /**
   * Get the Fox service from the test station.
   *
   * @return The Fox service
   */
  protected BFoxService getFoxService()
  {
    return (BFoxService)getServices().get(FOX_SERVICE);
  }

  /**
   * Get the authentication service from the test station.
   *
   * @return The authentication service
   */
  protected BAuthenticationService getAuthService()
  {
    return (BAuthenticationService)getServices().get(AUTH_SERVICE);
  }

  /**
   * Get the Niagara network from the test station.
   *
   * @return The Niagara network
   */
  protected BNiagaraNetwork getNiagaraNetwork()
  {
    return (BNiagaraNetwork)getDrivers().get(NIAGARA_NETWORK);
  }

  /**
   * Get the test visitor user
   *
   * @return The test visitor user
   */
  protected BUser getTestVisitorUser()
  {
    BUserService userService = BUserService.getService();
    return userService.getUser(TEST_VISITOR_USER);
  }

  /**
   * Determine if a valid UI environment exists.
   *
   * @return true if a valid UI environment exists
   */
  public static boolean hasUi()
  {
    UiEnv uiEnv = UiEnv.get();
    return !(uiEnv == null) && !(uiEnv instanceof NullUiEnv);
  }

  /**
   * Get the base URI string for the test station.
   *
   * @return The base URI string
   */
  protected String getBaseURI()
  {
    return "http://localhost:" + webPort + '/';
  }

  protected static final String TEST_VISITOR_USER = "TestVisitor";
  public static final String TEST_DELIMITED_USER = "username";
  public static final String TEST_DELIMITED_PASS = "user:password1";
}
