/*
 * Copyright 2023 Tridium, Inc. All Rights Reserved.
 */

package javax.baja.nre.util;

import javax.security.auth.x500.X500Principal;

import org.bouncycastle.asn1.x500.X500Name;

/**
 * Utility class for handling certificates.
 *
 * @author Melanie Coggan on 2023-11-08
 * @since Niagara 4.14
 */
public final class CertificateUtils
{
  /**
   * Checks if the two provided X500Principals are equivalent. This method will ignore the
   * ordering and case of the DN parts (e.g. CN, OU), and will ignore any whitespace around the
   * DN parts (e.g. CN = testCN vs. CN=testCN). It will also ignore case in the DN part values
   * (e.g CN=TESTCN, CN=testcn).
   *
   * The following DNs are considered equal:
   * <ul>
   *   <li>CN=testCN,O=testO,OU=testOU,L=testL,C=US</li>
   *   <li>CN=testcn,O=testo,OU=testou,L=testl,C=us</li>
   *   <li>CN = testCN, O = testO, OU = testOU, L = testL, C = US</li>
   *   <li>cn=testCN,o=testO,ou=testOU,l=testL,c=US</li>
   *   <li>C=US,L=testL,OU=testOU,O=testO,CN=testCN</li>
   *   <li>O=testO,C=US,OU=testOU,L=testL,CN=testCN</li>
   * </ul>
   *
   * The following DNs are also considered equals
   * <ul>
   *   <li>CN=testCN,O=testO,OU=,C=US</li>
   *   <li>CN=testCN,OU=,O=testO,C=US</li>
   * </ul>
   *
   * The following DNs are <b>not considered equal</b>:
   * <ul>
   *   <li>CN=testCN,O=testO,OU=testOU,C=US</li>
   *   <li>CN=testCN,O=testO,OU=testOU,L=,C=US</li>
   *   <li>CN=testCN,O=testO,OU=anotherOU,C=US</li>
   * </ul>
   *
   * @param dn1
   * @param dn2
   * @return
   */
  public static boolean checkDnEquality(X500Principal dn1, X500Principal dn2)
  {
    if (dn1 == null || dn2 == null)
    {
      return dn1 == dn2;
    }

    return checkDnEquality(dn1.getName(), dn2.getName());
  }

  /**
   * Checks if the two provided Strings are equivalent DNs. This method will ignore the
   * ordering and case of the DN parts (e.g. CN, OU), and will ignore any whitespace around the
   * DN parts (e.g. CN = testCN vs. CN=testCN). It will also ignore case in the DN part values
   * (e.g CN=TESTCN, CN=testcn).
   *
   * The following DNs are considered equal:
   * <ul>
   *   <li>CN=testCN,O=testO,OU=testOU,L=testL,C=US</li>
   *   <li>CN=testcn,O=testo,OU=testou,L=testl,C=us</li>
   *   <li>CN = testCN, O = testO, OU = testOU, L = testL, C = US</li>
   *   <li>cn=testCN,o=testO,ou=testOU,l=testL,c=US</li>
   *   <li>C=US,L=testL,OU=testOU,O=testO,CN=testCN</li>
   *   <li>O=testO,C=US,OU=testOU,L=testL,CN=testCN</li>
   * </ul>
   *
   * The following DNs are also considered equals
   * <ul>
   *   <li>CN=testCN,O=testO,OU=,C=US</li>
   *   <li>CN=testCN,OU=,O=testO,C=US</li>
   * </ul>
   *
   * The following DNs are <b>not considered equal</b>:
   * <ul>
   *   <li>CN=testCN,O=testO,OU=testOU,C=US</li>
   *   <li>CN=testCN,O=testO,OU=testOU,L=,C=US</li>
   *   <li>CN=testCN,O=testO,OU=anotherOU,C=US</li>
   * </ul>
   *
   * @param dn1
   * @param dn2
   * @return
   */
  public static boolean checkDnEquality(String dn1, String dn2)
  {
    if (dn1 == null || dn2 == null)
    {
      return dn1 == dn2;
    }

    return new X500Name(dn1).equals(new X500Name(dn2));
  }
}
