/*
 * Copyright 2000 Tridium, Inc. All Rights Reserved.
 */
package com.tridium.kitControl;

import javax.baja.alarm.BAlarmRecord;
import javax.baja.alarm.ext.offnormal.BTwoStateAlgorithm;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraProperty;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.status.BStatusValue;
import javax.baja.sys.BComponent;
import javax.baja.sys.BDouble;
import javax.baja.sys.BFacets;
import javax.baja.sys.BString;
import javax.baja.sys.Property;
import javax.baja.sys.Slot;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.units.BUnit;
import javax.baja.util.BFormat;

import com.tridium.sys.schema.Fw;

/**
 * BLoopAlarmAlgorithm implements a standard out-of-range
 * alarming algorithm
 *
 * @author    Dan Giorgis
 * @creation  25 May 01
 * @version   $Revision: 22$ $Date: 3/30/2004 3:31:03 PM$
 * @since     Baja 1.0
 */
@NiagaraType
/*
 This is the error required to generate an alarm. Starting in Niagara 4.14, when lowDiffLimitEnabled
 is true, this property sets the upper limit of the error's normal range only. Otherwise, this
 property sets both the upper and lower limits.
 */
@NiagaraProperty(
  name = "errorLimit",
  type = "double",
  defaultValue = "0"
)
/*
 Differential value applied to high and low limits before return-to-normal.
 Deadband is subtracted from highLimit and added to lowLimit.
 */
@NiagaraProperty(
  name = "deadband",
  type = "double",
  defaultValue = "0"
)
/*
 If true, the lower limit of the error's normal range is -1 * lowDiffLimit. Otherwise, it is
 -1 * errorLimit. The upper limit of the error's normal range is always errorLimit.
 @since Niagara 4.14
 */
@NiagaraProperty(
  name = "lowDiffLimitEnabled",
  type = "boolean",
  defaultValue = "false"
)
/*
 When lowDiffLimitEnabled is true, this property sets the lower limit of the error's normal range.
 @since Niagara 4.14
 */
@NiagaraProperty(
  name = "lowDiffLimit",
  type = "double",
  defaultValue = "0"
)
public class BLoopAlarmAlgorithm
  extends BTwoStateAlgorithm
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $com.tridium.kitControl.BLoopAlarmAlgorithm(3755948323)1.0$ @*/
/* Generated Wed Jul 05 13:15:16 CDT 2023 by Slot-o-Matic (c) Tridium, Inc. 2012-2023 */

  //region Property "errorLimit"

  /**
   * Slot for the {@code errorLimit} property.
   * This is the error required to generate an alarm. Starting in Niagara 4.14, when lowDiffLimitEnabled
   * is true, this property sets the upper limit of the error's normal range only. Otherwise, this
   * property sets both the upper and lower limits.
   * @see #getErrorLimit
   * @see #setErrorLimit
   */
  @Generated
  public static final Property errorLimit = newProperty(0, 0, null);

  /**
   * Get the {@code errorLimit} property.
   * This is the error required to generate an alarm. Starting in Niagara 4.14, when lowDiffLimitEnabled
   * is true, this property sets the upper limit of the error's normal range only. Otherwise, this
   * property sets both the upper and lower limits.
   * @see #errorLimit
   */
  @Generated
  public double getErrorLimit() { return getDouble(errorLimit); }

  /**
   * Set the {@code errorLimit} property.
   * This is the error required to generate an alarm. Starting in Niagara 4.14, when lowDiffLimitEnabled
   * is true, this property sets the upper limit of the error's normal range only. Otherwise, this
   * property sets both the upper and lower limits.
   * @see #errorLimit
   */
  @Generated
  public void setErrorLimit(double v) { setDouble(errorLimit, v, null); }

  //endregion Property "errorLimit"

  //region Property "deadband"

  /**
   * Slot for the {@code deadband} property.
   * Differential value applied to high and low limits before return-to-normal.
   * Deadband is subtracted from highLimit and added to lowLimit.
   * @see #getDeadband
   * @see #setDeadband
   */
  @Generated
  public static final Property deadband = newProperty(0, 0, null);

  /**
   * Get the {@code deadband} property.
   * Differential value applied to high and low limits before return-to-normal.
   * Deadband is subtracted from highLimit and added to lowLimit.
   * @see #deadband
   */
  @Generated
  public double getDeadband() { return getDouble(deadband); }

  /**
   * Set the {@code deadband} property.
   * Differential value applied to high and low limits before return-to-normal.
   * Deadband is subtracted from highLimit and added to lowLimit.
   * @see #deadband
   */
  @Generated
  public void setDeadband(double v) { setDouble(deadband, v, null); }

  //endregion Property "deadband"

  //region Property "lowDiffLimitEnabled"

  /**
   * Slot for the {@code lowDiffLimitEnabled} property.
   * If true, the lower limit of the error's normal range is -1 * lowDiffLimit. Otherwise, it is
   * -1 * errorLimit. The upper limit of the error's normal range is always errorLimit.
   * @since Niagara 4.14
   * @see #getLowDiffLimitEnabled
   * @see #setLowDiffLimitEnabled
   */
  @Generated
  public static final Property lowDiffLimitEnabled = newProperty(0, false, null);

  /**
   * Get the {@code lowDiffLimitEnabled} property.
   * If true, the lower limit of the error's normal range is -1 * lowDiffLimit. Otherwise, it is
   * -1 * errorLimit. The upper limit of the error's normal range is always errorLimit.
   * @since Niagara 4.14
   * @see #lowDiffLimitEnabled
   */
  @Generated
  public boolean getLowDiffLimitEnabled() { return getBoolean(lowDiffLimitEnabled); }

  /**
   * Set the {@code lowDiffLimitEnabled} property.
   * If true, the lower limit of the error's normal range is -1 * lowDiffLimit. Otherwise, it is
   * -1 * errorLimit. The upper limit of the error's normal range is always errorLimit.
   * @since Niagara 4.14
   * @see #lowDiffLimitEnabled
   */
  @Generated
  public void setLowDiffLimitEnabled(boolean v) { setBoolean(lowDiffLimitEnabled, v, null); }

  //endregion Property "lowDiffLimitEnabled"

  //region Property "lowDiffLimit"

  /**
   * Slot for the {@code lowDiffLimit} property.
   * When lowDiffLimitEnabled is true, this property sets the lower limit of the error's normal range.
   * @since Niagara 4.14
   * @see #getLowDiffLimit
   * @see #setLowDiffLimit
   */
  @Generated
  public static final Property lowDiffLimit = newProperty(0, 0, null);

  /**
   * Get the {@code lowDiffLimit} property.
   * When lowDiffLimitEnabled is true, this property sets the lower limit of the error's normal range.
   * @since Niagara 4.14
   * @see #lowDiffLimit
   */
  @Generated
  public double getLowDiffLimit() { return getDouble(lowDiffLimit); }

  /**
   * Set the {@code lowDiffLimit} property.
   * When lowDiffLimitEnabled is true, this property sets the lower limit of the error's normal range.
   * @since Niagara 4.14
   * @see #lowDiffLimit
   */
  @Generated
  public void setLowDiffLimit(double v) { setDouble(lowDiffLimit, v, null); }

  //endregion Property "lowDiffLimit"

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BLoopAlarmAlgorithm.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

////////////////////////////////////////////////////////////////
//  Callbacks
////////////////////////////////////////////////////////////////

  @Override
  public final Object fw(int x, Object a, Object b, Object c, Object d)
  {
    switch(x)
    {
      case Fw.STARTED: fwStarted(); break;
      case Fw.CHANGED: fwChanged((Property)a); break;
    }
    return super.fw(x, a, b, c, d);
  }

  /**
   * Call back when device is started.
   */
  private void fwStarted()
  {
    setErrorLimitDisplayName();
  }

  /**
   * To handle the properties that have changed
   */
  private void fwChanged(Property property)
  {
    if (isRunning())
    {
      if (property.equals(lowDiffLimitEnabled))
      {
        setErrorLimitDisplayName();
      }
    }
  }

  private void setErrorLimitDisplayName()
  {
    BFormat format = getDisplayNameFormat(errorLimit);
    if (format == null ||
        format.equals(HIGH_DIFF_LIMIT_FORMAT) ||
        format.equals(LOW_AND_HIGH_DIFF_LIMIT_FORMAT))
    {
      format = getLowDiffLimitEnabled() ? HIGH_DIFF_LIMIT_FORMAT : LOW_AND_HIGH_DIFF_LIMIT_FORMAT;
      setDisplayName(errorLimit, format, null);
    }
  }

////////////////////////////////////////////////////////////////
//  Parent checking
////////////////////////////////////////////////////////////////

  /**
   * A BLoopAlarmAlgorithm's grandparent must be a
   * BLoopPoint
   */
  public boolean isGrandparentLegal(BComponent grandparent)
  {
    return (grandparent instanceof BLoopPoint);
  }

////////////////////////////////////////////////////////////////
//  property facet checking
////////////////////////////////////////////////////////////////

  @Override
  public BFacets getSlotFacets(Slot slot)
  {
    if (slot.equals(deadband) || slot.equals(errorLimit) || slot.equals(lowDiffLimit))
    {
      BLoopPoint point = (BLoopPoint)getParentPoint();
      if (point == null)
      {
        return BFacets.DEFAULT;
      }

      BFacets facets = point.getInputFacets();
      BUnit unit = (BUnit) facets.getFacet(BFacets.UNITS);
      if (unit != null)
      {
        facets = BFacets.make(facets, BFacets.UNITS, unit.getDifferentialUnit());
      }
      return facets;
    }
    return super.getSlotFacets(slot);
  }

////////////////////////////////////////////////////////////////
//  Algorithm implementation
////////////////////////////////////////////////////////////////

  /**
   * Return true if the present value is normal
   */
  protected boolean isNormal(BStatusValue out)
  {
    BLoopPoint loopPt = (BLoopPoint)getParentPoint();

    double error = loopPt.getSetpoint().getValue() - loopPt.getControlledVariable().getValue();

    double highDiffLimit = getErrorLimit();
    double lowDiffLimit = getLowDiffLimitEnabled() ? -getLowDiffLimit() : -getErrorLimit();

    //  If in alarm / offnormal state, must return
    //  to within (error - deadband)
    if (isCurrentOffnormal())
    {
      return error < (highDiffLimit - getDeadband()) &&
             error > (lowDiffLimit  + getDeadband());
    }
    else
    {
      return error < highDiffLimit && error > lowDiffLimit;
    }
  }

  /**
   *  Write the key-value pairs defining alarm data for the
   *  alarm algorithm and state to the given Facets.
   * <p>
   *  The alarm data for an Out of Range alarm is given by
   *  BACnet table 13-3, Standard Object Property Values
   *  returned in notifications.
   *
   * @param out The relevant control point.
   * @param map The map.
   */
  @SuppressWarnings({"rawtypes","unchecked"})
  @Override
  public void writeAlarmData(BStatusValue out, java.util.Map map)
  {
    BLoopPoint loopPt = (BLoopPoint)getParentPoint();

    map.put(BAlarmRecord.CONTROLLED_VALUE, BString.make(loopPt.getControlledVariable().valueToString(loopPt.getInputFacets())));
    map.put(BAlarmRecord.STATUS, BString.make(out.getStatus().toString(null)));
    map.put(BAlarmRecord.SETPT_VALUE, BString.make(loopPt.getSetpoint().valueToString(loopPt.getInputFacets())));
    map.put(BAlarmRecord.DEADBAND, BString.make(BDouble.toString(getDeadband(), loopPt.getInputFacets())));
    // When lowDiffLimitEnabled is false, the errorLimit is both the highDiffLimit and the
    // lowDiffLimit. Keep errorLimit in case it's being used by clients.
    BString errorLimit = BString.make(BDouble.toString(getErrorLimit(), loopPt.getInputFacets()));
    map.put(BAlarmRecord.ERROR_LIMIT, errorLimit);
    map.put(BAlarmRecord.HIGH_DIFF_LIMIT, errorLimit);
    map.put(BAlarmRecord.LOW_DIFF_LIMIT, BString.make(BDouble.toString(getLowDiffLimitEnabled() ? getLowDiffLimit() : getErrorLimit(), loopPt.getInputFacets())));
  }

  private static final BFormat HIGH_DIFF_LIMIT_FORMAT = BFormat.make(BFormat.getLexiconPattern("kitControl", "loopAlarmAlgorithm.highDiffLimit", null));
  private static final BFormat LOW_AND_HIGH_DIFF_LIMIT_FORMAT = BFormat.make(BFormat.getLexiconPattern("kitControl", "loopAlarmAlgorithm.lowAndHighDiffLimit", null));
}
