/*
 * Copyright 2011 - Tridium Inc, All Rights Reserved.
 */

package javax.baja.security.crypto;

import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.logging.Logger;

import javax.baja.nre.security.IX509CertificateEntry;

import com.tridium.crypto.core.cert.NX509CertificateEntry;

/**
 * ITrustStore is the type of object returned from
 * a call to getSystemTrustStore or getUserTrustStore from an
 * ICryptoManager.
 *
 * @author    $Author$
 * @creation  29 June 2011
 * @version   $Revision$ $Date$
 * @since     Niagara 3.7
 */

public interface ITrustStore
{
  /**
   * Get an Enumeration of aliases stored in this trust store
   *
   * @return Enumeration of aliases in the trust store
   * @throws Exception if an error occurs retrieving the aliases
   * @throws java.security.AccessControlException if the calling module does
   *         not have the necessary KeyStorePermission.
   */
  Enumeration<String> aliases() throws Exception;

  /**
   * Return true if the specified alias is stored in this trust store
   *
   * @param alias the alias to check
   * @return true if the alias is present, false otherwise
   * @throws Exception if an error occurs checking for the alias
   * @throws java.security.AccessControlException if the calling module does
   *         not have the necessary KeyStorePermission.
   */
  boolean containsAlias(String alias) throws Exception;

  /**
   * Delete the specified entry from the trust store
   *
   * @param alias the alias to delete
   * @throws Exception if an error occurs deleting the entry
   * @throws java.security.AccessControlException if the calling module does
   *         not have the necessary KeyStorePermission.
   */
  void deleteEntry(String alias) throws Exception;

  /**
   * Get the certificate stored for the specified alias
   *
   * @param alias the alias to retrieve
   * @return The certificate stored for the alias, or null if the alias does
   *         not exist
   * @throws Exception if an error occurs retrieving the certificate
   * @throws java.security.AccessControlException if the calling module does
   *         not have the necessary KeyStorePermission.
   */
  X509Certificate getCertificate(String alias) throws Exception;

  /**
   * Get the alias that the specified certificate is stored as
   *
   * @param cert the certificate to get the alias for
   * @return The the alias that the certificate is stored as, or null if the
   *         certificate is not present
   * @throws Exception if an error occurs retrieving the alias
   * @throws java.security.AccessControlException if the calling module does
   *         not have the necessary KeyStorePermission.
   */
  String getCertificateAlias(X509Certificate cert) throws Exception;

  /**
   * Get the certificate chain stored for the specified alias
   *
   * @param alias the alias to retrieve
   * @return The certificate chain stored for the alias, or null if the alias
   *         does not exist
   * @throws Exception if an error occurs retrieving the certificate chain
   * @throws java.security.AccessControlException if the calling module does
   *         not have the necessary KeyStorePermission.
   */
  X509Certificate[] getCertificateChain(String alias) throws Exception;

  /**
   * Get the creation date of the specified alias
   *
   * @param alias the alias to check
   * @return The creation date or the alias, or null if the alias does not exist
   * @throws Exception if an error occurs retrieving the creation date
   * @throws java.security.AccessControlException if the calling module does
   *         not have the necessary KeyStorePermission.
   */
  Date getCreationDate(String alias) throws Exception;

  /**
   * Return true if the alias exists and contains a certificate
   *
   * @param alias the alias to check
   * @return true if the alias exists and contains a certificate, false otherwise
   * @throws Exception if an error occurs checking the alias
   * @throws java.security.AccessControlException if the calling module does
   *         not have the necessary KeyStorePermission.
   */
  boolean isCertificateEntry(String alias) throws Exception;

  /**
   * Return true if the alias exists and contains a key
   *
   * @param alias the alias to check
   * @return true if the alias exists and contains a key, false otherwise
   * @throws Exception if an error occurs checking the alias
   * @throws java.security.AccessControlException if the calling module does
   *         not have the necessary KeyStorePermission.
   */
  boolean isKeyEntry(String alias) throws Exception;

  /**
   * Store a certificate with the specified alias
   *
   * @param alias the alias for the entry
   * @param cert the certificate to store
   * @throws Exception if an error occurs setting the certificate
   * @throws java.security.AccessControlException if the calling module does
   *         not have the necessary KeyStorePermission.
   */
  void setCertificateEntry(String alias, X509Certificate cert) throws Exception;

  /**
   * Return the number of entries in the trust store
   *
   * @return the number of entries in the trust store
   * @throws Exception if an error occurs checking the entries
   * @throws java.security.AccessControlException if the calling module does
   *         not have the necessary KeyStorePermission.
   */
  int size() throws Exception;

  /**
   * Save the trust store to file
   *
   * @throws Exception if an error occurs saving the trust store
   * @throws java.security.AccessControlException if the calling module does
   *         not have the necessary KeyStorePermission.
   */
  void save() throws Exception;

  /**
   * Return an Iterable of all certificates in the trust store
   *
   * @return Iterable of certificates in the trust store
   * @throws Exception if an error occurs retrieving the certificates
   * @throws java.security.AccessControlException if the calling module does
   *         not have the necessary KeyStorePermission.
   */
  Iterable<IX509CertificateEntry> getCertificateEntries() throws Exception;

  /**
   * Return an Enumeration of all certificates in the trust store
   *
   * @return Enumeration of certificates in the trust store
   * @throws Exception if an error occurs retrieving the certificates
   * @throws java.security.AccessControlException if the calling module does
   *         not have the necessary KeyStorePermission.
   *
   * @deprecated since Niagara 4.13 - will be removed in Niagara 5.0
   */
  @Deprecated
  default Enumeration<NX509CertificateEntry> getCertificates()
    throws Exception
  {
    List<NX509CertificateEntry> nx509CertificateEntries = new ArrayList<>();
    Iterable<IX509CertificateEntry> ix509certificateEntries = getCertificateEntries();
    for (IX509CertificateEntry ix509CertificateEntry : ix509certificateEntries)
    {
      if (ix509CertificateEntry instanceof NX509CertificateEntry)
      {
        nx509CertificateEntries.add((NX509CertificateEntry) ix509CertificateEntry);
      }
      else
      {
        Logger.getLogger("crypto.cert").fine("Skipping non-NX509CertificateEntry: " + ix509CertificateEntry.getAlias());
      }
    }
    return Collections.enumeration(nx509CertificateEntries);
  }

  /**
   * Return the alias of the certificate in the trust store that matches the
   * specified certificate's public key
   *
   * @param match the certificate to check for
   * @return the alias of the certificate with a matching public key, or null
   *         if no match is found
   * @throws Exception if an error occurs checking for the certificate
   * @throws java.security.AccessControlException if the calling module does
   *         not have the necessary KeyStorePermission.
   */
  String findCertificate(X509Certificate match) throws Exception;

  /**
   * Delete the specified aliases from the trust store
   *
   * @param aliases the aliases to delete
   * @throws Exception if an error occurs deleting the entries
   * @throws java.security.AccessControlException if the calling module does
   *         not have the necessary KeyStorePermission.
   */
  void deleteEntries(String[] aliases) throws Exception;
}
