/*
 * Copyright 2002 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.security;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;

import javax.baja.nre.annotations.NiagaraType;
import javax.baja.nre.annotations.NoSlotomatic;
import javax.baja.nre.util.IntHashMap;
import javax.baja.sys.BObject;
import javax.baja.sys.BSimple;
import javax.baja.sys.Context;
import javax.baja.sys.IllegalNameException;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

/**
 * BPermissions encapsulates a set of permissions for a given
 * security domain.  String encoding for BPermissions is:
 * <pre>
 *  permissions := [oRead][oWrite][oInvoke][aRead][aWrite][aInvoke]
 *  oRead       := r
 *  oWrite      := w
 *  oInvoke     := i
 *  aRead       := R
 *  aWrite      := W
 *  aInvoke     := I
 * </pre>
 *
 * @author    Brian Frank
 * @creation  25 Mar 02
 * @version   $Revision: 10$ $Date: 2/16/09 9:23:59 AM EST$
 * @since     Baja 1.0
 */
@NiagaraType
@NoSlotomatic //BSimple needs manual DEFAULT, Type implementation
public final class BPermissions
  extends BSimple
{
////////////////////////////////////////////////////////////////
// Factory Methods
////////////////////////////////////////////////////////////////

  /**
   * Make a BPermissions for the specified bit mask.
   */
  public static BPermissions make(int mask)
  {
    // normalize
    if ((mask & OPERATOR_WRITE) != 0) mask |= OPERATOR_READ;
    if ((mask & ADMIN_READ) != 0)     mask |= OPERATOR_READ;
    if ((mask & ADMIN_WRITE) != 0)    mask |= (OPERATOR_READ | OPERATOR_WRITE | ADMIN_READ);
    if ((mask & ADMIN_INVOKE) != 0)   mask |= OPERATOR_INVOKE;

    // map each unique mask to one cached instance
    synchronized(cache)
    {
      BPermissions permissions = (BPermissions)cache.get(mask);
      if (permissions == null)
      {
        permissions = new BPermissions(mask);
        cache.put(mask, permissions);
      }
      return permissions;
    }
  }

  /**
   * Create a new BPermissions from the bitwise OR of this BPermissions
   * instance and the specified BPermissions.
   *
   * @since Niagara 3.5
   */
  public BPermissions or(BPermissions other)
  {
    return BPermissions.make(this.mask | other.mask);
  }

  /**
   * Create a new BPermissions from the bitwise AND of this BPermissions
   * instance and the specified BPermissions.
   *
   * @since Niagara 3.5
   */
  public BPermissions and(BPermissions other)
  {
    return BPermissions.make(this.mask & other.mask);
  }

  static IntHashMap cache = new IntHashMap();

  /**
   * Make a BPermissions for the specified string encoding.
   */
  public static BPermissions make(String s)
    throws IOException
  {
    return (BPermissions)DEFAULT.decodeFromString(s);
  }

////////////////////////////////////////////////////////////////
// Constructor
////////////////////////////////////////////////////////////////

  /**
   * Private constructor.
   */
  private BPermissions(int mask)
  {
    this.mask = mask;
  }

////////////////////////////////////////////////////////////////
// Access
////////////////////////////////////////////////////////////////

  /**
   * Get the permission mask.
   */
  public int getMask()
  {
    return mask;
  }

  /**
   * Return true if all the permissions specified
   * in in the given bitmask are enabled.
   */
  public boolean has(int required)
  {
    return (mask & required) == required;
  }

  /**
   * Return true if this instance has all the permissions
   * enabled which are enabled in the specified instance.
   */
  public boolean has(BPermissions permissions)
  {
    return (mask & permissions.mask) == permissions.mask;
  }

  /**
   * Is the operator read permission enabled.
   */
  public boolean hasOperatorRead()
  {
    return (mask & OPERATOR_READ) != 0;
  }

  /**
   * Is the operator write permission enabled.
   */
  public boolean hasOperatorWrite()
  {
    return (mask & OPERATOR_WRITE) != 0;
  }

  /**
   * Is the operator invoke permission enabled.
   */
  public boolean hasOperatorInvoke()
  {
    return (mask & OPERATOR_INVOKE) != 0;
  }

  /**
   * Is the admin read permission enabled.
   */
  public boolean hasAdminRead()
  {
    return (mask & ADMIN_READ) != 0;
  }

  /**
   * Is the admin write permission enabled.
   */
  public boolean hasAdminWrite()
  {
    return (mask & ADMIN_WRITE) != 0;
  }

  /**
   * Is the admin invoke permission enabled.
   */
  public boolean hasAdminInvoke()
  {
    return (mask & ADMIN_INVOKE) != 0;
  }

  /**
   * Array of all individual permissions, useful for iterating
   */
  public static BPermissions[] getAllPermissions()
  {
    return allPermissions;
  }

////////////////////////////////////////////////////////////////
// BSimple
////////////////////////////////////////////////////////////////

  /**
   * Hash is based on <code>System.identityHashCode()</code>.
   * Added override for this method in Niagara 3.4.
   */
  public int hashCode()
  {
    // System's identityHashCode is fine since these are
    // cached as singletons
    return System.identityHashCode(this);
  }

  /**
   * BPermissions equality is based on identical bitmask.
   */
  public boolean equals(Object obj)
  {
    // one instance cached for each unique mask
    return obj == this;
  }

  /**
   * To string method.
   */
  public String toString(Context context)
  {
    return encodeToString();
  }

  /**
   * BPermissions is encoded as using writeUTF(encodeToString()).
   */
  public void encode(DataOutput out)
    throws IOException
  {
    out.writeUTF(encodeToString());
  }

  /**
   * BPermissions is decoded using decodeFromString(readUTF()).
   */
  public BObject decode(DataInput in)
    throws IOException
  {
    return decodeFromString(in.readUTF());
  }

  /**
   * Write the simple in text format.
   */
  public String encodeToString()
  {
    if (string == null)
    {
      StringBuilder s = new StringBuilder();
      if (hasOperatorRead()) s.append('r');
      if (hasOperatorWrite()) s.append('w');
      if (hasOperatorInvoke()) s.append('i');
      if (hasAdminRead()) s.append('R');
      if (hasAdminWrite()) s.append('W');
      if (hasAdminInvoke()) s.append('I');
      string = s.toString();
    }
    return string;
  }

  /**
   * Read the simple from text format.
   */
  public BObject decodeFromString(String s)
    throws IOException
  {
    try
    {
      int mask = 0;
      for(int i=0; i<s.length(); ++i)
      switch(s.charAt(i))
      {
        case 'i': mask |= OPERATOR_INVOKE; break;
        case 'r': mask |= OPERATOR_READ; break;
        case 'w': mask |= OPERATOR_WRITE; break;
        case 'I': mask |= ADMIN_INVOKE; break;
        case 'R': mask |= ADMIN_READ; break;
        case 'W': mask |= ADMIN_WRITE; break;
      }
      return make(mask);
    }
    catch(IllegalNameException e)
    {
      throw e;
    }
    catch(Exception e)
    {
      throw new IOException("Invalid BPermissions: " + s);
    }
  }

////////////////////////////////////////////////////////////////
// Constants
////////////////////////////////////////////////////////////////

  /** Operator read privilege. */
  public static final int OPERATOR_READ   = 0x0001;
  /** Operator write privilege (implies OPERATOR_READ). */
  public static final int OPERATOR_WRITE  = 0x0002;
  /** Operator invoke privilege. */
  public static final int OPERATOR_INVOKE = 0x0004;
  /** Admin read privilege (implies OPERATOR_READ). */
  public static final int ADMIN_READ   = 0x0010;
  /** Admin write privilege (implies OPERATOR_READ, OPERATOR_WRITE, ADMIN_READ). */
  public static final int ADMIN_WRITE  = 0x0020;
  /** Admin invoke privilege (implies OPERATOR_INVOKE). */
  public static final int ADMIN_INVOKE = 0x0040;

  /** Operator read privilege. */
  public static final BPermissions operatorRead = make(OPERATOR_READ);
  /** Operator write privilege (implies OPERATOR_READ). */
  public static final BPermissions operatorWrite = make(OPERATOR_WRITE);
  /** Operator invoke privilege. */
  public static final BPermissions operatorInvoke = make(OPERATOR_INVOKE);
  /** Admin read privilege (implies OPERATOR_READ). */
  public static final BPermissions adminRead = make(ADMIN_READ);
  /** Admin write privilege (implies OPERATOR_READ, OPERATOR_WRITE, ADMIN_READ). */
  public static final BPermissions adminWrite = make(ADMIN_WRITE);
  /** Admin invoke privilege (implies OPERATOR_INVOKE). */
  public static final BPermissions adminInvoke = make(ADMIN_INVOKE);
  /** Array of all permissions that can be granted. */
  private static final BPermissions[] allPermissions = new BPermissions[]{operatorRead,
    operatorWrite, operatorInvoke, adminRead, adminWrite, adminInvoke
  };

  /** All permissions enabled. */
  public static final BPermissions all = make(
    OPERATOR_READ | OPERATOR_WRITE | OPERATOR_INVOKE |
    ADMIN_READ | ADMIN_WRITE | ADMIN_INVOKE);

////////////////////////////////////////////////////////////////
// DEFAULT Constants
////////////////////////////////////////////////////////////////

  /** No permissions enabled. */
  public static final BPermissions none = make(0);

  /**
   * The default is none.
   */
  public static final BPermissions DEFAULT = none;

//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
  /*@ $javax.baja.security.BPermissions(2979906276)1.0$ @*/
  /* Generated Tue Jun 06 15:45:33 EDT 2023 by Slot-o-Matic (c) Tridium, Inc. 2012-2023 */

  //region Type

  @Override
  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BPermissions.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////

  private int mask;
  private String string;
}
