/*
 * Copyright 2000 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.security;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.AccessControlException;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.security.SecureRandom;
import java.util.Base64;
import java.util.Objects;
import java.util.Optional;

import javax.baja.nre.annotations.NiagaraType;
import javax.baja.nre.annotations.NoSlotomatic;
import javax.baja.nre.util.ByteArrayUtil;
import javax.baja.nre.util.SecurityUtil;
import javax.baja.sys.BBoolean;
import javax.baja.sys.BIComparable;
import javax.baja.sys.BIUnlinkable;
import javax.baja.sys.BObject;
import javax.baja.sys.BSimple;
import javax.baja.sys.Context;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

import com.tridium.crypto.core.bundle.CryptographicAlgorithmBundle;
import com.tridium.nre.security.AesAlgorithmBundle;
import com.tridium.nre.security.EncryptionAlgorithmBundle;
import com.tridium.nre.security.EncryptionKeySource;
import com.tridium.nre.security.ISecretBytesSupplier;
import com.tridium.nre.security.PBEValidator;
import com.tridium.nre.security.SecretBytes;
import com.tridium.nre.security.SecretChars;
import com.tridium.util.PasswordUtil;

/**
 * The BPassword is a string containing secret data.
 *
 * Secret keys can be provided to make() in form &lt;plaintextValue&gt; and they will be encoded
 * as appropriate, otherwise the make() methods expect the form [&lt;encodingType&gt;]=&lt;encodedValue&gt;
 *
 * @author Brian Frank on 1 Feb 00
 * @since     Baja 1.0
 */
@NiagaraType
@NoSlotomatic //BSimple needs manual DEFAULT, Type implementation
public final class BPassword
  extends BSimple
  implements BIComparable, BIPasswordValidator, BIUnlinkable
{
  /**
   * Create a BPassword with the given value.
   * <p>
   * NOTE: the change in the source field from the name password
   * to key is to recognize that not all supplied data is a password.
   *
   * @throws NullPointerException     if value is {@code null}.
   * @throws IllegalArgumentException if key can't be parsed
   */
  public static BPassword make(String key)
  {
    return make(key, (Context)null);
  }

  /**
   * Create a BPassword with the given value.
   * <p>
   * NOTE: the change in the source field from the name password
   * to key is to recognize that not all supplied data is a password.
   *
   * @throws NullPointerException     if value is {@code null}.
   * @throws IllegalArgumentException if key can't be parsed
   * @since Niagara 4.0
   */
  public static BPassword make(String key, Context context)
  {
    Objects.requireNonNull(key);
    return new BPassword(key, context);
  }

  /**
   * Create a BPassword with the given value.
   * <p>
   * NOTE: the change in the source field from the name password
   * to key is to recognize that not all supplied data is a password.
   *
   * @throws NullPointerException     if value is {@code null}.
   * @throws IllegalArgumentException if key can't be parsed
   */
  public static BPassword make(char[] key)
  {
    Objects.requireNonNull(key);
    String skey = new String(key);
    return make(skey, (Context)null);
  }

  /**
   * Create a BPassword with the given value.
   * <p>
   * NOTE: the change in the source field from the name password
   * to key is to recognize that not all supplied data is a password.
   *
   * @throws NullPointerException if values are {@code null}.
   */
  public static BPassword make(String password, String encodingType)
  {
    Objects.requireNonNull(password);
    return new BPassword(password, encodingType);
  }

  /**
   * Create a BPassword with the given value.
   * <p>
   * NOTE: the change in the source field from the name password
   * to key is to recognize that not all supplied data is a password.
   *
   * @throws NullPointerException if values are {@code null}.
   */
  public static BPassword make(char[] password, String encodingType)
  {
    Objects.requireNonNull(password);
    Objects.requireNonNull(encodingType);
    String spassword = new String(password);
    return make(spassword, encodingType);
  }

  /**
   * Create a BPassword from a PBEValidator
   *
   * @since Niagara 4.0
   */
  public static BPassword make(PBEValidator pbeValidator)
  {
    Objects.requireNonNull(pbeValidator);
    return make(pbeValidator.getEncodedValidator(), (Context)null);
  }

  /**
   * Creates a BPassword with the given encoder
   * <p>
   * The encoder should have all its field correctly initialized
   * for encoding.
   */
  public static BPassword make(BAbstractPasswordEncoder encoder)
  {
    return new BPassword(encoder);
  }

  /**
   * Private constructor.
   * @since Niagara 4.0
   */
  private BPassword(String key, Context context)
  {
    Optional<PasswordEncodingContext> pContext = PasswordEncodingContext.find(context);
    if (pContext.isPresent())
    {
      constructEncoder(key, pContext.get());
    }
    else
    {
      // Close the temporary password encoding context once the encoder is constructed. We do not
      // want to close the PasswordEncodingContext found within the context passed in.
      try (PasswordEncodingContext newPContext = new PasswordEncodingContext(context))
      {
        constructEncoder(key, newPContext);
      }
    }
  }

  private void constructEncoder(String key, PasswordEncodingContext pContext)
  {
    // Check value for an encoding type
    boolean encoded = true;
    String encodingType;
    try
    {
      encodingType = CryptographicAlgorithmBundle.extractName(key);
    }
    catch (IllegalArgumentException iae)
    {
      encodingType = null;
    }

    if (encodingType == null)
    {
      // If no encoding type is found, use default reversible encoding
      if (pContext.getDecryptionKeySource() == EncryptionKeySource.none 
        && pContext.getEncryptionKeySource() == EncryptionKeySource.none)
      {
        // we don't have enough information to encode/decode
        handleConstructorMissingDecodingKey(pContext);
        return;
      }
      else
      {
        encoded = false;
        encodingType = BReversiblePasswordEncoder.getDefaultEncodingType();
      }
    }

    try
    {
      encoder = BAbstractPasswordEncoder.make(encodingType);
    }
    catch (Exception e)
    {
      throw new IllegalArgumentException("unrecognized encoding type: " + encodingType);
    }

    if (encoder instanceof BReversiblePasswordEncoder)
    {
      BReversiblePasswordEncoder reversiblePasswordEncoder = (BReversiblePasswordEncoder)encoder;
      // Check our encryption source--none is invalid, undefined means we just copy
      // the decryption context
      if (pContext.getEncryptionKeySource() == EncryptionKeySource.none)
      {
        handleConstructorMissingDecodingKey(pContext);
      }
      else if (pContext.getEncryptionKeySource() == EncryptionKeySource.undefined)
      {
        // Encode and hope for the best
        try
        {
          AccessController.doPrivileged((PrivilegedExceptionAction<Void>)() ->
          {
            if (pContext.getDecryptionKeySource() == EncryptionKeySource.keyring)
            {
              reversiblePasswordEncoder.setUsesExternalEncryptionKey(false);
            }
            else
            {
              reversiblePasswordEncoder.setUsesExternalEncryptionKey(true);
              if (pContext.getDecryptionKey().isPresent())
              {
                reversiblePasswordEncoder.setExternalEncryptionKey(pContext.getDecryptionKey().get().get());
              }
            }
            encoder.parse(key);
            // TODO: attempt decrypt?
            return null;
          });
        }
        catch (Exception e)
        {
          throw new SecurityException();
        }
      }
      // otherwise we want to transcode from something to something else. Set up
      // our encoder from the encryption key source and load it
      else
      {
        // set up our encoder
        if (pContext.getEncryptionKeySource() == EncryptionKeySource.keyring)
        {
          reversiblePasswordEncoder.setUsesExternalEncryptionKey(false);
        }
        else if (pContext.getEncryptionKeySource() == EncryptionKeySource.external
          || pContext.getEncryptionKeySource() == EncryptionKeySource.shared)
        {
          reversiblePasswordEncoder.setUsesExternalEncryptionKey(true);
          if (pContext.getEncryptionKey().isPresent())
          {
            try
            {
              AccessController.doPrivileged((PrivilegedExceptionAction<Void>)() ->
              {
                reversiblePasswordEncoder.setExternalEncryptionKey(pContext.getEncryptionKey().get().get());
                return null;
              });
            }
            catch (Exception e)
            {
              throw new SecurityException();
            }
          }
        }
        if (pContext.getDecryptionKeySource() == EncryptionKeySource.shared 
          && pContext.getEncryptionKeySource() == EncryptionKeySource.shared)
        {
          if (reversiblePasswordEncoder.validateExternalEncryptionKey(pContext.getDecryptionKey()))
          {
            throw new IllegalArgumentException("Cannot re-use shared key");
          }
        }
        
        // If we don't need to decrypt our input--or if it's not an encoded input--we just
        // use our encoder as-is
        if (pContext.getDecryptionKeySource() == EncryptionKeySource.none || !encoded) // also if not encoded
        {
          try
          {
            if (encoded)
            {
              reversiblePasswordEncoder.parse(key);
              // TODO: attempt decrypt?
            }
            else
            {
              reversiblePasswordEncoder.encode(key);
            }
          }
          catch (MissingEncodingKeyException rethrow)
          {
            handleConstructorMissingEncodingKey(pContext);
          }
          catch (Exception e)
          {
            throw new SecurityException();
          }
        }
        // Otherwise we need to transcode between storage formats, if we can.
        else
        {
          try
          {
            AccessController.doPrivileged((PrivilegedExceptionAction<Void>)() ->
            {
              reversiblePasswordEncoder.transcode(key, pContext.getDecryptionKey());
              return null;
            });
          }
          catch (MissingEncodingKeyException rethrow)
          {
            handleConstructorMissingEncodingKey(pContext);
          }
          catch (Exception e)
          {
            throw new SecurityException();
          }
        }
      }
    }
    else
    {
      try
      {
        if (encoded)
        {
          encoder.parse(key);
        }
        else
        {
          encoder.encode(key);
        }
      }
      catch (MissingEncodingKeyException meke)
      {
        handleConstructorMissingEncodingKey(pContext);
      }
      catch (Exception e)
      {
        throw new SecurityException();
      }
    }
  }

  private void handleConstructorMissingDecodingKey(PasswordEncodingContext pContext)
  {
    BPassword fallback = pContext.getDecodeErrorPasswordValue();
    encoder = (BAbstractPasswordEncoder)fallback.getPasswordEncoder().getType().getInstance();
    try
    {
      encoder.parse(fallback.getPasswordEncoder().getEncodedValue());
    }
    catch (SecurityException rethrow)
    {
      throw rethrow;
    }
    catch (Exception e)
    {
      throw new SecurityException();
    }
  }

  private void handleConstructorMissingEncodingKey(PasswordEncodingContext pContext)
  {
    BPassword fallback = make(pContext.getErrorEncodedValue());
    encoder = (BAbstractPasswordEncoder)fallback.getPasswordEncoder().getType().getInstance();
    try
    {
      encoder.parse(fallback.getPasswordEncoder().getEncodedValue());
    }
    catch (SecurityException rethrow)
    {
      throw rethrow;
    }
    catch (Exception e)
    {
      throw new SecurityException();
    }
  }

  /**
   * Private constructor.
   */
  private BPassword(String password, String encodingType)
  {
    try
    {
      encoder = BAbstractPasswordEncoder.make(encodingType);
    }
    catch (Exception e)
    {
      throw new IllegalArgumentException("unrecognized encoding type: " + encodingType);
    }

    try
    {
      encoder.encode(password);
    }
    catch (Exception e)
    {
      throw new SecurityException();
    }
  }
  
  /**
   * Private constructor.
   */
  private BPassword(BAbstractPasswordEncoder encoder)
  {
    this.encoder = encoder;
  }
  

  /**
   * If the password is in a reversible format, getValue() retrieves the
   * plain text String value of the password.
   * If the password is not reversible, returns a Base64 encoded random byte array.
   *
   * Since Niagara 4.6, reversible passwords are encrypted by module specific keys and
   * protected by a permission check. To avoid permissions issues, it is recommended to
   * wrap all calls to getValue() in a doPrivileged block like this:
   *
   * {@code AccessController.doPrivileged((PrivilegedAction<String>)password::getValue)}
   * 
   * @return the String value of the password, or a random byte array
   * @throws SecurityException if the password is reversible but cannot be recovered
   * (e.g. missing security key), or if the password is not reversible and a random
   * number cannot be generated.
   */
  public String getValue() throws SecurityException
  {
    try
    {
      if (encoder.isReversible() || encoder instanceof BNullPasswordEncoder)
      // we need BNullPasswordEncoder to not be reversible so it does not get converted to
      // PBKDF2, but we also need it to return "" as its value so it fails password strength
      {
        return encoder.getValue();
      }
      else 
      {
        byte[] val = new byte[16];
        new SecureRandom().nextBytes(val);
        return Base64.getEncoder().encodeToString(val);
      }
    }
    catch(MissingEncodingKeyException | AccessControlException rethrow)
    {
      throw rethrow;
    }
    catch(Exception e)
    {
      throw new SecurityException();
    }
  }

  /**
   * If the password is in a reversible format, getSecretChars() returns the unencrypted value,
   * otherwise it returns random characters.
   *
   * Callers should take care to ensure that the result is closed when the secret data
   * is no longer needed.  Putting it in a try-with-resources is a good way to do that.
   *
   * Since Niagara 4.6, reversible passwords are encrypted by module specific keys and
   * protected by a permission check. To avoid permissions issues, it is recommended to
   * wrap all calls to getSecretChars() in a doPrivileged block like this:
   *
   * {@code AccessController.doPrivileged((PrivilegedAction<SecretChars>)password::getSecretChars)}
   *
   * @since Niagara 4.0
   */
  public SecretChars getSecretChars() throws SecurityException
  {
    try
    {
      if (encoder.isReversible() || encoder instanceof BNullPasswordEncoder)
      {
        return encoder.getSecretChars();
      }
      else
      {
        byte[] val = new byte[16];
        new SecureRandom().nextBytes(val);
        return SecretChars.fromString(ByteArrayUtil.toHexString(val));
      }
    }
    catch(Exception e)
    {
      throw new SecurityException();
    }
  }

  /**
   * BPassword uses its String value's hash code. 
   */
  @Override
  public int hashCode()
  {
    return encoder.getEncodedValue().hashCode();
  }
  
  /**
   * BPassword equality is based on String value equality.
   */
  @Override
  public boolean equals(Object obj)
  {
    return obj instanceof BPassword &&
      SecurityUtil.equals(encoder.getEncodedValue(), ((BPassword)obj).encoder.getEncodedValue());
  }

  /**
   * Comparison of BPassword to other passwords serves no purpose so changed to 
   * just return 0
   */
  @Override
  public int compareTo(Object obj)
  {
    return 0;
  }


  /**
   * Return true if the given password matches this object's.
   * @since Niagara 4.0
   */
  public boolean validate(BPassword toValidate)
  {
    if (toValidate != null && toValidate.getPasswordEncoder().isReversible())
    {
      return AccessController.doPrivileged((PrivilegedAction<Boolean>) () -> {
        try (SecretChars secretChars = toValidate.getSecretChars())
        {
          return validate(secretChars);
        }
      });
    }
    return false;
  }
  /**
   * Return true if the given password matches this object's.
   * @since Niagara 4.0
   */
  @Override
  public boolean validate(SecretChars password)
  {
    try
    {
      return encoder.validate(password);
    }
    catch (Exception e)
    {
      throw new SecurityException();
    }
  }

  /**
   * Return true if the given data matches this object's value.
   */
  public boolean validate(SecretBytes secretData)
  {
    try
    {
      if (encoder instanceof BReversiblePasswordEncoder)
      {
        return ((BReversiblePasswordEncoder)encoder).validate(secretData);
      }
      else
      {
        try(SecretChars chars = SecretChars.fromSecretBytes(secretData, StandardCharsets.UTF_8, false))
        {
          return encoder.validate(chars);
        }
      }
    }
    catch (Exception e)
    {
      throw new SecurityException();
    }
  }

  /**
   * To string method.
   */
  @Override
  public String toString(Context context)
  {
    return "--password--";
  }

  /**
   * BPassword is encoded as using {@link java.io.DataOutput#writeUTF(java.lang.String)}.
   */
  @Override
  public void encode(DataOutput out)
    throws IOException
  {
    encode(out, null);
  }
  
  /**
   * BPassword is encoded as using {@link java.io.DataOutput#writeUTF(java.lang.String)}.
   * @since Niagara 4.0
   */
  @Override
  public void encode(DataOutput out, Context context)
    throws IOException
  {
    out.writeUTF(encodeToString(context));
  }
  
  /**
   * BPassword is decoded using {@link java.io.DataInput#readUTF()}.
   */
  @Override
  public BObject decode(DataInput in)
    throws IOException
  {
    return decode(in, null);
  }
  
  /**
   * BPassword is decoded using {@link java.io.DataInput#readUTF()}.
   */
  @Override
  public BObject decode(DataInput in, Context context)
    throws IOException
  {
    String s = in.readUTF();
    if (s.equals("")) return DEFAULT;
    return decodeFromString(s, context);
  }

  /**
   * Write the simple in text format which is encrypted.
   */
  @Override
  public String encodeToString()
    throws IOException
  {
    return encodeToString(null);
  }
  
  /**
   * Write the simple in text format which is encrypted.
   * @since Niagara 4.0
   */
  @Override
  public String encodeToString(Context context)
    throws IOException
  {
    if (context != null && this != PasswordUtil.SUBSTITUTE_PASSWORD)
    {
      BObject skipFacet = context.getFacet(PasswordUtil.SKIP_ENCODING_SENSITIVE_KEY);
      if (BBoolean.TRUE == skipFacet && getPasswordEncoder().isReversible())
      { // Only do for reversible passwords, since user sync needs to encode non-reversible ones
        return PasswordUtil.SUBSTITUTE_PASSWORD.encodeToString(context);
      }
    }

    Optional<PasswordEncodingContext> pContext = PasswordEncodingContext.find(context);
    if (pContext.isPresent())
    {
      return encodeToString(context, pContext.get());
    }
    else
    {
      // Close the temporary password encoding context once the password is encoded. We do not
      // want to close the PasswordEncodingContext found within the context passed in.
      try (PasswordEncodingContext newPContext = new PasswordEncodingContext(context))
      {
        return encodeToString(context, newPContext);
      }
    }
  }

  private String encodeToString(Context context, PasswordEncodingContext pContext) throws IOException
  {
    if (encoder instanceof BReversiblePasswordEncoder)
    {
      BReversiblePasswordEncoder reversibleEncoder = (BReversiblePasswordEncoder)encoder;

      if (EncryptionKeySource.keyring.equals(pContext.getEncryptionKeySource()))
      {
        // I'm expected to encode with key ring, not keys
        if (reversibleEncoder.usesExternalEncryptionKey())
        {
          // NCCB-71645: This previously set the transcoder to BAbstractPasswordEncoder.makeDefaultInstance(false),
          // which returns a non-reversible password encoder and was almost certainly wrong. This was
          // already fixed in 4.15 for NCCB-59299 (encrypted hashed passwords). It started causing some
          // problems in some unit tests in 4.14u2/4.10u11 after some tests were showing the wrong encoding type
          // being used.
          try
          {
            BAbstractPasswordEncoder transcoder = makeTranscoder(pContext);
            try(SecretChars chars = encoder.getSecretChars())
            {
              transcoder.encode(chars);
            }
            return transcoder.getEncodedValue();
          }
          catch (MissingEncodingKeyException meke)
          {
            return pContext.getErrorEncodedValue();
          }
          catch (IOException ioe)
          {
            throw ioe;
          }
          catch (Exception e)
          {
            throw new IOException(e);
          }

        }
        // else we're already encoding with keyring, so getEncodedValue() is either fine
      }
      else if (EncryptionKeySource.none.equals(pContext.getEncryptionKeySource()))
      {
        // If the password is being encoded to a string while it's parent component is being saved to a bog
        // file, the PasswordEncodingContext encryptionKeySource is set to EncryptionKeySource.none and this
        // code path is executed.  If the component has been copied up from the station, and the password is
        // reversible, the password value would have been changed to PasswordUtil.SUBSTITUTE_PASSWORD and
        // BPassword.DEFAULT.encodeToString(context) will be returned.
        if (PasswordUtil.SUBSTITUTE_PASSWORD.validate(this))
        {
          return BPassword.DEFAULT.encodeToString(context);
        }
        return pContext.getErrorEncodedValue();
      }
      else
      {
        // I must encode with a key, not the key ring
        if (reversibleEncoder.usesExternalEncryptionKey())
        {
          // this password was decoded using a pass phrase
          if (pContext.hasEncryptionKey())
          {
            // NCCB-71465: Previously, we only performed transcoding if the encoding key was different
            // from what is used in our current encoding. This resulted in a new encoder with the
            // same initialization vector, which is a security concern. Now, we transcode whether the
            // encoding key is the same or not.
            try
            {
              return AccessController.doPrivileged((PrivilegedExceptionAction<String>)()->
                {
                  BReversiblePasswordEncoder transcoder = makeTranscoder(pContext);
                  transcoder.setExternalEncryptionKey(pContext.getEncryptionKey().get().get());
                  Optional<byte[]> encryptionKey = ((BReversiblePasswordEncoder)encoder).getEncryptionKey();
                  if (!encryptionKey.isPresent())
                  {
                    throw new MissingEncodingKeyException();
                  }
                  try (SecretBytes bytes = new SecretBytes(((BReversiblePasswordEncoder)encoder).getEncryptionKey().get(), false))
                  {
                    ISecretBytesSupplier key = ISecretBytesSupplier.wrap(bytes);
                    transcoder.transcode(encoder.getEncodedValue(), Optional.of(key));
                    return transcoder.getEncodedValue();
                  }
                });
            }
            catch (PrivilegedActionException pae)
            {
              if (pae.getCause() instanceof IOException)
              {
                throw (IOException)pae.getCause();
              }
              else
              {
                throw new IOException(pae.getCause());
              }
            }
          }
          // else I wasn't given an encoding key - but I can assume that the encoder used the right one
        }
        else
        {
          // This password was decoded using the keyring, but needs a pass phrase
          if (!pContext.hasEncryptionKey())
          {
            // don't have one, can't encode
            return pContext.getErrorEncodedValue();
          }
          try
          {
            return AccessController.doPrivileged((PrivilegedExceptionAction<String>)() ->
              {
                BReversiblePasswordEncoder transcoder = makeTranscoder(pContext);
                transcoder.setExternalEncryptionKey(pContext.getEncryptionKey().get().get());
                transcoder.transcode(encoder.getEncodedValue(), Optional.empty());
                return transcoder.getEncodedValue();
              });
          }
          catch (PrivilegedActionException pae)
          {
            if (pae.getCause() instanceof IOException)
            {
              throw (IOException)pae.getCause();
            }
            else
            {
              throw new IOException(pae.getCause());
            }
          }
        }
      }
    }
    else if (encoder instanceof BPlainPasswordEncoder)
    {
      // TODO: what if this value should use PBE?  how would it know?
      if (EncryptionKeySource.keyring.equals(pContext.getEncryptionKeySource()))
      {
        try
        {
          // todo: This is turning a plain password, which was probably intended to be reversible,
          // into an irriversible password. I think this will only happen if a password is explicitly
          // created with BPlainPasswordEncoder, which should happen rarely if ever, which is probably
          // why it hasn't caused issues. In case I'm wrong and this is the correct thing to do, I am
          // leaving it be to avoid breaking things.
          BAbstractPasswordEncoder transcoder = BAbstractPasswordEncoder.makeDefaultInstance(false);
          try (SecretChars chars = encoder.getSecretChars())
          {
            transcoder.encode(chars);
          }
          return transcoder.getEncodedValue();
        }
        catch (MissingEncodingKeyException meke)
        {
          return pContext.getErrorEncodedValue();
        }
        catch (IOException ioe)
        {
          throw ioe;
        }
        catch (Exception e)
        {
          throw new IOException(e);
        }
      }
      else if (EncryptionKeySource.external.equals(pContext.getEncryptionKeySource()))
      {
        if (!pContext.hasEncryptionKey())
        {
          // don't have one, can't encode
          return pContext.getErrorEncodedValue();
        }
        try
        {
          return AccessController.doPrivileged((PrivilegedExceptionAction<String>)() ->
          {
            // todo: This creates a BAes256PasswordEncoder as the transcoder, and attempts to transcode
            // a plain password with it. I think BAes256PasswordEncoder can only transcode other passwords
            // encoded with a BAes256PasswordEncoder, so I don't think this will work. This probably
            // hasn't caused issues because we rarely, if ever, use BPlainPasswordEncoder. In case I'm
            // wrong and this is the correct thing to do, I am leaving it be to avoid breaking things.
            BReversiblePasswordEncoder transcoder = BReversiblePasswordEncoder.makeDefaultInstance();
            transcoder.setExternalEncryptionKey(pContext.getEncryptionKey().get().get());
            transcoder.transcode(encoder.getEncodedValue(), Optional.empty());
            return transcoder.getEncodedValue();
          });
        }
        catch (PrivilegedActionException pae)
        {
          if (pae.getCause() instanceof IOException)
          {
            throw (IOException)pae.getCause();
          }
          else
          {
            throw new IOException(pae.getCause());
          }
        }
      }
      else if (EncryptionKeySource.none.equals(pContext.getEncryptionKeySource()))
      {
        return pContext.getErrorEncodedValue();
      }
    }

    // else it's not reversible, there's no extra checking necessary
    return encoder.getEncodedValue();
  }

  /**
   * Make the BReversiblePasswordEncoder to use for transcoding for the given PasswordEncodingContext
   *
   * @param context the PasswordEncodingContext
   * @return the encoder to use for transcoding
   * @throws Exception if an error occurs initializing the encoder
   * @since Niagara 4.9
   */
  private BReversiblePasswordEncoder makeTranscoder(PasswordEncodingContext context) throws Exception
  {
    Optional<EncryptionAlgorithmBundle> bundle = context.getEncryptionAlgorithmBundle();
    if (bundle.isPresent() && bundle.get() instanceof AesAlgorithmBundle)
    {
      if (encoder instanceof BAbstractAes256PasswordEncoder)
      {
        // Use the algorithm type of the encoder and the version of the algorithm bundle
        String encodingType = ((BAbstractAes256PasswordEncoder)encoder).getAlgorithmBundle().getAlgorithmType() + '.' + bundle.get().getAlgorithmVersion();
        return (BReversiblePasswordEncoder) BAbstractPasswordEncoder.make(encodingType);
      }
    }
    return (BReversiblePasswordEncoder) BAbstractPasswordEncoder.make(encoder.getEncodingType());
  }
  
  /**
   * Read the simple from text format which is encrypted.
   */
  @Override
  public BObject decodeFromString(String s, Context context)
    throws IOException
  {
    if (s.equals(""))
    {
      return DEFAULT;
    }
    return make(s, context);
  }

  @Override
  public BObject decodeFromString(String s)
    throws IOException
  {
    return decodeFromString(s, null);
  }

  public String getEncodingType()
  {
    return encoder.getEncodingType();
  } 
  
  public BAbstractPasswordEncoder getPasswordEncoder()
  {
    return encoder;
  }

  /**
   * Returns true if this object is, or is equivalent to {@link #DEFAULT}
   *
   * @since Niagara 4.0
   */
  public boolean isDefault() { return getPasswordEncoder() instanceof BNullPasswordEncoder; }

////////////////////////////////////////////////////////////////
// DEFAULT Constants
////////////////////////////////////////////////////////////////

  /**
   * The default instance is encoded with an empty string and is used for "no password", specifically:
   * <ul>
   *   <li>its {@link #validate(String)}, {@link #validate(SecretChars)} and {@link #validate(SecretBytes)} always return false</li>
   *   <li>its {@link #getValue()} and {@link #getSecretChars()} return random characters</li>
   * </ul>
   */
  public static final BPassword DEFAULT = new BPassword(BNullPasswordEncoder.getEncodedDefaultPassword(), (Context)null);

//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
  /*@ $javax.baja.security.BPassword(2979906276)1.0$ @*/
  /* Generated Tue Jun 06 08:28:20 EDT 2023 by Slot-o-Matic (c) Tridium, Inc. 2012-2023 */

  //region Type

  @Override
  public Type getType() { return TYPE; }
  public static final Type TYPE = Sys.loadType(BPassword.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  private BAbstractPasswordEncoder encoder;
  public static final String PLACEHOLDER_TEXT = "placeholderText";
}
