/*
 * Copyright 2023 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.security;

import java.security.cert.X509Certificate;

import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.sys.BComplex;
import javax.baja.sys.BInterface;
import javax.baja.sys.BStation;
import javax.baja.sys.Context;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

import com.tridium.crypto.core.cert.CertUtils;

/**
 * BICertificateAliasAndPasswordContainer is an interface for
 * Niagara {@link javax.baja.sys.BComplex} types that contain a
 * property of type {@link BCertificateAliasAndPassword}. It is
 * often used to support components that wish to request (and
 * renew on a periodic basis) a properly signed certificate from
 * a station running Niagara's SigningService.
 *
 * @author Scott Hoye on 4/6/2023
 * @since Niagara 4.14
 */
@NiagaraType
public interface BICertificateAliasAndPasswordContainer
  extends BInterface
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.security.BICertificateAliasAndPasswordContainer(2979906276)1.0$ @*/
/* Generated Thu Apr 06 10:13:53 EDT 2023 by Slot-o-Matic (c) Tridium, Inc. 2012-2023 */

  //region Type

  @Generated
  Type TYPE = Sys.loadType(BICertificateAliasAndPasswordContainer.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  /**
   * Get a reference to the {@link Property} in this container whose value
   * is a {@link BCertificateAliasAndPassword}.
   *
   * @param context The Context associated with this request (can be null).
   *                Future use cases may specify a special Context facet to
   *                indicate which BCertificateAliasAndPassword property to
   *                retrieve if a single container has multiple properties
   *                of the same type (not common).
   */
  Property getCertificateAliasAndPasswordProperty(Context context);

  /**
   * Gets the password to use for protecting the signed certificate specified by the
   * given BCertificateAliasAndPassword-valued property. Used to support containers that wish
   * to request (and renew on a periodic basis) a properly signed certificate from a station
   * running Niagara's SigningService. Be sure to wrap the returned module owned BPassword in
   * a doPrivileged block to ensure the password can be read by the caller.
   * <p>
   * To protect access to your password you must also make a check with the security manager, to
   * ensure any module calling this method has the required permission.
   * <p>
   * <pre>{@code
   * SecurityManager sm = System.getSecurityManager();
   * if (sm != null)
   * {
   *   sm.checkPermission(new SigningPasswordPermission(TYPE.getTypeSpec().getModuleName()));
   * }
   * return BPassword.make(AccessController.doPrivileged(
   *       (PrivilegedAction<String>)() -> get(certAliasAndPasswordProperty).as(BCertificateAliasAndPassword.class).getPassword().getValue()));
   * }</pre>
   * <p>
   * Along with the following in your module-permissions.xml entry for the station:
   * <pre>{@code
   *     <req-permission>
   *       <name>SIGNING</name>
   *       <purposeKey>Protect my signed certificate passwords</purposeKey>
   *       <parameters>
   *         <parameter name="moduleName" value="nameOfModule"/>
   *       </parameters>
   *     </req-permission>
   * }</pre>
   * <p>
   * Now only the framework itself, and modules declaring a matching moduleName parameter value can invoke
   * this method and gain access to the certificate password value.
   *
   * @param certAliasAndPasswordProperty the {@link Property} in this container whose value
   *                                     is a {@link BCertificateAliasAndPassword} from which to
   *                                     obtain the password.
   * @param context The Context associated with the call to this method.
   * @return password to use for the certificate, if null, a default or empty password is returned, it will
   * not be used.
   */
  BPassword retrieveCertificatePassword(Property certAliasAndPasswordProperty, Context context);

  /**
   * Callback to the container to inform that the certificate has been signed for
   * the BCertificateAliasAndPassword specified by the given property. Used to support
   * containers that wish to request (and renew on a periodic basis) a properly signed
   * certificate from a station running Niagara's SigningService. This callback gives
   * the container a notification that the certificate has been signed and stored in
   * the core keystore in case it needs to do any further updates in response. The
   * default implementation is a no-op, but can be optionally overridden by implementing
   * classes.
   *
   * @param certAliasAndPasswordProperty the {@link Property} in this container whose value
   *                                     is a {@link BCertificateAliasAndPassword} for which
   *                                     the certificate it specifies has just been signed
   *                                     (or re-signed).
   * @param certificateChain the signed certificate chain.
   * @param context The Context associated with the call to this method.
   */
  default void certificateSigned(Property certAliasAndPasswordProperty, X509Certificate[] certificateChain, Context context)
  {
    // Default implementation is a no-op
  }

  /**
   * Get a recommended certificate alias for this container. The default implementation of this
   * method is implemented as follows:
   * <ul>
   * <li>If the current value of the property specified by {@link #getCertificateAliasAndPasswordProperty(Context)}
   * is using a non-empty and non-default {@link CertUtils#FACTORY_CERT_ALIAS} alias, then that value will be
   * returned as the recommended certificate alias for this container.</li>
   * <li>Otherwise, the value returned by {@link #supplyRecommendedCertificateAlias(String, Context)} is used.</li>
   * <li>If {@link #supplyRecommendedCertificateAlias(String, Context)} throws an error, just the container's display name
   * will be returned (if available) or null will be returned as a last resort if no recommended certificate alias can be determined.</li>
   * </ul>
   *
   * @param context The Context associated with this request (can be null).
   *                For the default implementation, this Context is passed into
   *                {@link #getCertificateAliasAndPasswordProperty(Context)}, so
   *                refer to the Context description there.
   */
  default String getRecommendedCertificateAlias(Context context)
  {
    String recommendedAlias = null;
    try
    {
      BComplex complex = as(BComplex.class);
      recommendedAlias = complex.getDisplayName(null);
      BCertificateAliasAndPassword certAliasAndPassword =
        (BCertificateAliasAndPassword) complex.get(getCertificateAliasAndPasswordProperty(context));
      String existingAlias = certAliasAndPassword.getAlias();
      if (!existingAlias.trim().isEmpty() && !CertUtils.FACTORY_CERT_ALIAS.equals(existingAlias))
      {
        recommendedAlias = existingAlias;
      }
      else
      {
        recommendedAlias = supplyRecommendedCertificateAlias(recommendedAlias, context);
      }
    }
    catch (Throwable ignore)
    {
      // Since the recommended alias is built up from least-favored to most-favored match, any
      // problems during computation can be ignored and the result returned immediately. Even a null
      // recommended alias is fine if a better one can't be computed.
    }

    return recommendedAlias;
  }

  /**
   * Supplies the recommended certificate alias for this container, only used if the current value of the
   * property specified by {@link #getCertificateAliasAndPasswordProperty(Context)} is using a non-empty
   * and non-default {@link CertUtils#FACTORY_CERT_ALIAS} alias.
   *
   * @param containerDisplayName The display name of this container.
   * @param context The Context associated with this request (can be null).
   * @return A concatenation of the current station name with the display name of this container
   * (separated by a dash). If called outside the context of a running station, containerDisplayName
   * is returned.
   */
  default String supplyRecommendedCertificateAlias(String containerDisplayName, Context context)
  {
    BStation station = Sys.getStation();
    if (containerDisplayName != null && station != null)
    {
      return station.getStationName() + '-' + containerDisplayName;
    }
    return containerDisplayName;
  }
}
