/*
 * Copyright 2015 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.security;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;

import javax.baja.agent.AgentFilter;
import javax.baja.agent.AgentList;
import javax.baja.agent.BIAgent;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.registry.TypeInfo;
import javax.baja.sys.BObject;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.util.Lexicon;

import com.tridium.nre.security.SecretChars;

/**
 * 
 * A BAbstractPasswordEncoder provides a way for passwords
 * to be encoded using different encoding methods, such as
 * hashing or encryption.
 * 
 * This allows us store passwords differently based on what
 * is required for the situation.
 * 
 * All subclasses must include an ENCODING_TYPE field, which
 * will be used to create a map between encoding names and
 * actual classes.
 * 
 * Subclasses must also be agents on BPassword, or they will
 * not be added to the map.
 *
 * @author Melanie Coggan
 * @creation Mar 18, 2013
 *
 */
@NiagaraType
public abstract class BAbstractPasswordEncoder
    extends BObject
    implements BIAgent
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.security.BAbstractPasswordEncoder(2979906276)1.0$ @*/
/* Generated Wed Dec 29 19:27:38 CST 2021 by Slot-o-Matic (c) Tridium, Inc. 2012-2021 */

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BAbstractPasswordEncoder.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  /**
   * Takes a password and encodes it. The encoding will depend on the
   * particular subclass of BAbstractPasswordEncoder.
   */
  public final void encode(String password) throws Exception
  {
    try(SecretChars chars = SecretChars.fromString(password))
    {
      encode(chars);
    }
  }

  /**
   * Takes a password and encodes it. The encoding will depend on the
   * particular subclass of BAbstractPasswordEncoder.
   */
  public abstract void encode(SecretChars password) throws Exception;

  /**
   * Takes an encoded string and parses it to instantiate the encoder.
   * The string should be in the format returned by getEncodedValue().
   */
  public abstract void parse(String key) throws Exception;

  /**
   * Returns the plaintext value of the password, if the password is 
   * reversible.
   *
   * Since Niagara 4.6, reversible passwords are encrypted by module specific keys and
   * protected by a permission check. To avoid permissions issues, it is recommended to
   * wrap all calls to getValue() on a reversible encoder in a doPrivileged block like this:
   *
   * {@code AccessController.doPrivileged((PrivilegedExceptionAction<String>)encoder::getValue)}
   */
  public abstract String getValue() throws Exception;

  /**
   * Returns the unencrypted value of the password, if it is reversible.
   *
   * Callers should take care to ensure that the result is closed when the secret data
   * is no longer needed.  Putting it in a try-with-resources is a good way to do that.
   *
   * Since Niagara 4.6, reversible passwords are encrypted by module specific keys and
   * protected by a permission check. To avoid permissions issues, it is recommended to
   * wrap all calls to getSecretChars() in a doPrivileged block like this:
   *
   * {@code AccessController.doPrivileged((PrivilegedExceptionAction<SecretChars>)encoder::getSecretChars)}
   *
   * @since Niagara 4.0
   */
  public SecretChars getSecretChars() throws Exception
  {
    // Subclasses should implement this method in a way that doesn't hold the result in a String
    // object
    return SecretChars.fromString(getValue());
  }
  
  /**
   * Returns a String describing the encoding type. 
   */
  public abstract String getEncodingType();

  /**
   * Returns true is the original password can be recovered from the
   * encoded password.
   * Returns false if the password encoding is one-way.
   */
  public abstract boolean isReversible();
  
  /**
   * Validates that the provided String "password" corresponds to the
   * encoded password.
   */
  public final boolean validate(String password) throws Exception
  {
    try(SecretChars chars = SecretChars.fromString(password))
    {
      return validate(chars);
    }
  }

  public abstract boolean validate(SecretChars password) throws Exception;
  
  /**
   * Returns an encoded string describing the password. This should include
   * the encrypted/hashed password and any public data associated with it
   * (e.g. the salt, or initialization vector).
   * 
   */
  public abstract String getEncodedValue();

  /**
   * Return a system default encoding type
   *
   * @param isReversible if true, returns the default encoding type for reversible encodings, otherwise returns
   *                     the default encoding type for one-way hashing
   * @since Niagara 4.0
   */
  public static String getDefaultEncodingType(boolean isReversible)
  {
    return isReversible ? BAes256PasswordEncoder.ENCODING_TYPE : BPbkdf2HmacSha256PasswordEncoder.ENCODING_TYPE;
  }

  /**
   * Return the an instance of a system default encoder
   *
   * @param isReversible if true, returns the default encoder for reversible encodings, otherwise returns
   *                     the default encoder for one-way hashing
   * @since Niagara 4.0
   */
  public static BAbstractPasswordEncoder makeDefaultInstance(boolean isReversible)
    throws Exception
  {
    return make(getDefaultEncodingType(isReversible));
  }

  /**
   * Creates a new encoder of the specified type. The encoding type must correspond
   * go an encoding type of a subclass of BAbstractPasswordEncoder as obtained
   * using getEncodingType().
   */
  public static BAbstractPasswordEncoder make(String encodingType)
    throws Exception
  {
    encodingType = stripBrackets(encodingType);
    Type resolvedType = EncodingTypesHolder.encodingTypes.get(encodingType);
    Object o;
    
    if (resolvedType != null)
    {
      o = resolvedType.getInstance();
    }
    else
    {
      throw new IllegalArgumentException("unrecognized encoding type: " + encodingType);
    }
    
    return (BAbstractPasswordEncoder) o;
  }
  
////////////////////////////////////////////////////////////////
// Helpers
////////////////////////////////////////////////////////////////

  private static String stripBrackets(String encodingType)
  {
    if (encodingType.startsWith("["))
    {
      return encodingType.substring(1, encodingType.length() - 1);
    }
    else
    {
      return encodingType;
    }
  }
  
////////////////////////////////////////////////////////////////
// Initialization
////////////////////////////////////////////////////////////////

  /**
   * Used to lazily load the encoding types known by the Niagara registry
   */
  private static final class EncodingTypesHolder
  {
    private static final Map<String, Type> encodingTypes = createEncodingMap();

    /**
     * Creates a HashMap of all the different password encoders (encoding type/Niagara type).
     * Password encoders must be subclasses of BAbstractPasswordEncoder and
     * be agents on BPassword.
     */
    private static Map<String,Type> createEncodingMap()
    {
      HashMap<String,Type> encodingMap = new HashMap<>();
      Lexicon lex = Lexicon.make(BAbstractPasswordEncoder.class);
      TypeInfo agentType = null;
      AgentList agents = Sys.getRegistry().getAgents(Sys.getRegistry().getType("baja:Password"));
      agents = agents.filter(AgentFilter.is(TYPE));

      for (int i = 0; i < agents.size(); i++)
      {
        try
        {
          agentType = agents.get(i).getAgentType();
          Type resolvedType = agentType.getTypeSpec().getResolvedType();
          Class<?> c = resolvedType.getTypeClass();
          String encodingType = (String)c.getField("ENCODING_TYPE").get(null);
          encodingMap.put(encodingType, resolvedType);
        }
        // Ensure that the subclasses has the ENCODING_TYPE field
        catch (NoSuchFieldException e)
        {
          LOG.severe(lex.getText("password.encoder.noTypeField", agentType.getTypeName()));
        }
        // Any other exception just print out a generic error
        catch (Exception e)
        {
          LOG.severe(lex.getText("password.encoder.genericFailure", agentType == null ? "unknown agent type" : agentType.getTypeName(), e));
        }
      }

      return Collections.unmodifiableMap(encodingMap);
    }
  }

////////////////////////////////////////////////////////////////
// Attributes
////////////////////////////////////////////////////////////////
  private static final Logger LOG = Logger.getLogger("baja");
}
