/*
 * Copyright 2025 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.bacnet.export;

import static com.tridium.bacnet.schedule.ScheduleSupport0.BACNET_IDX;
import static com.tridium.bacnet.schedule.ScheduleSupport0.PRIORITY_16;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.baja.schedule.BCompositeSchedule;
import javax.baja.schedule.BDailySchedule;
import javax.baja.schedule.BWeeklySchedule;
import javax.baja.sys.BComponentEvent;
import javax.baja.sys.BComponentEventMask;
import javax.baja.sys.BInteger;
import javax.baja.sys.Flags;
import javax.baja.sys.Subscriber;

import com.tridium.util.CompUtil;

/**
 * BacnetSpecialEventsSubscriber handles resetting BACnet indexes and priorities when daily
 * schedules on the special events of a weekly schedule are reordered or removed.
 *
 * @author Bishal Debbarma on 21st May, 2025
 * @since Niagara 4.14U3
 * @since Niagara 4.15U2
 */
class BacnetSpecialEventsSubscriber
  extends Subscriber
{
  public BacnetSpecialEventsSubscriber()
  {
    setMask(SPECIAL_EVENTS_MASK);
  }

  public void subscribe(BBacnetScheduleDescriptor descriptor, BCompositeSchedule specialEvents)
  {
    descriptorMap.put(specialEvents, descriptor);
    super.subscribe(specialEvents);
  }

  public void unsubscribe(BCompositeSchedule specialEvents)
  {
    if (specialEvents == null)
    {
      return;
    }
    descriptorMap.remove(specialEvents);
    super.unsubscribe(specialEvents);
  }

  @Override
  public void event(BComponentEvent event)
  {
    BCompositeSchedule specialEvents = (BCompositeSchedule) event.getSourceComponent();
    BWeeklySchedule schedule = getParentWeeklySchedule(specialEvents);
    BBacnetScheduleDescriptor descriptor = descriptorMap.get(specialEvents);
    if (descriptor == null)
    {
      logger.info("Error handling event from schedule's special events: no schedule descriptor found" +
        "; schedule: " + schedule.getSlotPath());
      unsubscribe(specialEvents);
      return;
    }

    try
    {
      switch (event.getId())
      {
        // Listen for removed and reorder events on schedules so the BACnet indexes and priority
        // properties can be reset. Added events are provided with a BACnet index and priority so
        // they do not need to be handled here.
        case BComponentEvent.PROPERTY_REMOVED:
        case BComponentEvent.PROPERTIES_REORDERED:
          // When special event priority is changed via BACnet write property request and special
          // events are reordered, do not reset priorities here.
          if (descriptor.isRemoteChange.get())
          {
            break;
          }

          BDailySchedule[] events = specialEvents.getChildren(BDailySchedule.class);
          for (int i = 0; i < events.length; i++)
          {
            // Reset existing priorities to 16
            CompUtil.setOrAdd(events[i], "priority", PRIORITY_16, Flags.USER_DEFINED_1, null, null);
            // Set the BACnet index to match the Niagara order
            CompUtil.setOrAdd(events[i], BACNET_IDX, BInteger.make(i + 1), Flags.USER_DEFINED_1, null, null);
          }
          break;
      }
    }
    catch (Exception e)
    {
      logger.log(
        Level.WARNING,
        "Error handling event from schedule's special events" +
          "; descriptor: " + descriptor +
          ", schedule: " + schedule.getSlotPath() +
          ", exception: " + e,
        logger.isLoggable(Level.FINE) ? e : null);
    }
  }

  private BWeeklySchedule getParentWeeklySchedule(BCompositeSchedule specialEvents)
  {
    // Special events parent is a composite schedule that contains it and the weekly schedule.
    BCompositeSchedule parent = (BCompositeSchedule) specialEvents.getParent();
    // The parent of the composite schedule is the BWeeklySchedule.
    return (BWeeklySchedule) parent.getParent();
  }

  private static final BComponentEventMask SPECIAL_EVENTS_MASK = BComponentEventMask.make(
    new int[] {
      BComponentEvent.PROPERTY_REMOVED,
      BComponentEvent.PROPERTIES_REORDERED
    });

  private static final Logger logger = Logger.getLogger("bacnet.server");
  private final Map<BCompositeSchedule, BBacnetScheduleDescriptor> descriptorMap = new HashMap<>();
}
