/*
 * Copyright 2025 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.bacnet.datatypes;

import java.util.ArrayList;
import java.util.List;

import javax.baja.bacnet.io.AsnException;
import javax.baja.bacnet.io.AsnInput;
import javax.baja.bacnet.io.AsnOutput;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraAction;
import javax.baja.nre.annotations.NiagaraProperty;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.sys.Action;
import javax.baja.sys.BBoolean;
import javax.baja.sys.BComponent;
import javax.baja.sys.BDouble;
import javax.baja.sys.BEnum;
import javax.baja.sys.BFloat;
import javax.baja.sys.BInteger;
import javax.baja.sys.BString;
import javax.baja.sys.BValue;
import javax.baja.sys.Context;
import javax.baja.sys.Flags;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;
import javax.baja.util.BTypeSpec;

import com.tridium.bacnet.asn.AsnUtil;

/**
 * BBacnetSequenceOf represents a non-indexed sequence of a particular BACnet data type that allows
 * duplicates. It is based on BBacnetListOf but without extra handling for AddListElement and
 * RemoveListElement services.
 *
 * @author Uday Rapuru on 06-Jun-2025
 * @since Niagara 4.14u3, 4.15u2
 */
@NiagaraType
@NiagaraProperty(
  name = "sequenceTypeSpec",
  type = "BTypeSpec",
  defaultValue = "BTypeSpec.DEFAULT",
  flags = Flags.READONLY | Flags.HIDDEN
)
@NiagaraAction(
  name = "add",
  parameterType = "BValue",
  defaultValue = "BBoolean.FALSE"
)
@NiagaraAction(
  name = "remove",
  parameterType = "BInteger",
  defaultValue = "BInteger.make(1)",
  returnType = "BValue"
)
public class BBacnetSequenceOf
  extends BComponent
  implements BIBacnetDataType
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.bacnet.datatypes.BBacnetSequenceOf(425322871)1.0$ @*/
/* Generated Wed Jul 23 17:02:25 CDT 2025 by Slot-o-Matic (c) Tridium, Inc. 2012-2025 */

  //region Property "sequenceTypeSpec"

  /**
   * Slot for the {@code sequenceTypeSpec} property.
   * @see #getSequenceTypeSpec
   * @see #setSequenceTypeSpec
   */
  @Generated
  public static final Property sequenceTypeSpec = newProperty(Flags.READONLY | Flags.HIDDEN, BTypeSpec.DEFAULT, null);

  /**
   * Get the {@code sequenceTypeSpec} property.
   * @see #sequenceTypeSpec
   */
  @Generated
  public BTypeSpec getSequenceTypeSpec() { return (BTypeSpec)get(sequenceTypeSpec); }

  /**
   * Set the {@code sequenceTypeSpec} property.
   * @see #sequenceTypeSpec
   */
  @Generated
  public void setSequenceTypeSpec(BTypeSpec v) { set(sequenceTypeSpec, v, null); }

  //endregion Property "sequenceTypeSpec"

  //region Action "add"

  /**
   * Slot for the {@code add} action.
   * @see #add(BValue parameter)
   */
  @Generated
  public static final Action add = newAction(0, BBoolean.FALSE, null);

  /**
   * Invoke the {@code add} action.
   * @see #add
   */
  @Generated
  public void add(BValue parameter) { invoke(add, parameter, null); }

  //endregion Action "add"

  //region Action "remove"

  /**
   * Slot for the {@code remove} action.
   * @see #remove(BInteger parameter)
   */
  @Generated
  public static final Action remove = newAction(0, BInteger.make(1), null);

  /**
   * Invoke the {@code remove} action.
   * @see #remove
   */
  @Generated
  public BValue remove(BInteger parameter) { return invoke(remove, parameter, null); }

  //endregion Action "remove"

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BBacnetSequenceOf.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  @SuppressWarnings("unused")
  public BBacnetSequenceOf()
  {
  }

  public BBacnetSequenceOf(Type sequenceType)
  {
    setSequenceTypeSpec(BTypeSpec.make(sequenceType));
  }

  @Override
  public final void changed(Property property, Context context)
  {
    super.changed(property, context);

    notifyParent(context);
  }

  @Override
  public final void added(Property property, Context context)
  {
    super.added(property, context);

    notifyParent(context);
  }

  @Override
  public final void removed(Property property, BValue oldValue, Context context)
  {
    super.removed(property, oldValue, context);

    notifyParent(context);
  }

  @Override
  public void reordered(Context context)
  {
    super.reordered(context);

    notifyParent(context);
  }

  private void notifyParent(Context context)
  {
    if (!isRunning() || context == noWrite)
    {
      return;
    }

    // Notify the parent in case a write property request should be sent.
    getParent().asComponent().changed(getPropertyInParent(), context);
  }

  public void doAdd(BValue newElement, Context context)
  {
    if (!newElement.getType().equals(getSequenceTypeSpec().getResolvedType()))
    {
      throw new IllegalArgumentException("Invalid type " + newElement.getType() + " for " + this);
    }

    add("element1?", newElement, context);
  }

  public BValue doRemove(BInteger index, Context context)
  {
    // Gather all the properties that match the sequence type.
    Type sequenceType = getSequenceTypeSpec().getResolvedType();
    Property[] allProperties = getDynamicPropertiesArray();
    List<Property> elementProperties = new ArrayList<>(allProperties.length);
    for (Property property : allProperties)
    {
      if (get(property).getType().equals(sequenceType))
      {
        elementProperties.add(property);
      }
    }

    int indexInt = index.getInt();
    if (indexInt < 1 || indexInt > elementProperties.size())
    {
      throw new IllegalArgumentException("Invalid index: " + indexInt);
    }

    Property removedProperty = elementProperties.get(indexInt - 1);
    BValue removedValue = get(removedProperty);
    remove(removedProperty, context);
    return removedValue;
  }

  @Override
  public BValue getActionParameterDefault(Action action)
  {
    if (action.equals(add))
    {
      BTypeSpec typeSpec = getSequenceTypeSpec();
      if (typeSpec.isNull())
      {
        return null;
      }

      return (BValue) typeSpec.getInstance();
    }

    return super.getActionParameterDefault(action);
  }

  @Override
  public void writeAsn(AsnOutput out)
  {
    Type sequenceType = getSequenceTypeSpec().getResolvedType();
    int asnType = AsnUtil.getAsnType(sequenceType);
    for (Object element : getChildren(sequenceType.getTypeClass()))
    {
      switch (asnType)
      {
        case ASN_NULL:
          out.writeNull();
          break;
        case ASN_BOOLEAN:
          out.writeBoolean((BBoolean)element);
          break;
        case ASN_UNSIGNED:
          out.writeUnsigned((BBacnetUnsigned)element);
          break;
        case ASN_INTEGER:
          out.writeSignedInteger((BInteger)element);
          break;
        case ASN_REAL:
          out.writeReal((BFloat)element);
          break;
        case ASN_DOUBLE:
          out.writeDouble((BDouble)element);
          break;
        case ASN_OCTET_STRING:
          out.writeOctetString((BBacnetOctetString)element);
          break;
        case ASN_CHARACTER_STRING:
          out.writeCharacterString((BString)element);
          break;
        case ASN_BIT_STRING:
          out.writeBitString((BBacnetBitString)element);
          break;
        case ASN_ENUMERATED:
          out.writeEnumerated((BEnum)element);
          break;
        case ASN_DATE:
          out.writeDate((BBacnetDate)element);
          break;
        case ASN_TIME:
          out.writeTime((BBacnetTime)element);
          break;
        case ASN_OBJECT_IDENTIFIER:
          out.writeObjectIdentifier((BBacnetObjectIdentifier)element);
          break;
        default:
          ((BIBacnetDataType)element).writeAsn(out);
          break;
      }
    }
  }

  public static List<BValue> readAsn(AsnInput in, int closingTag, Type sequenceType)
    throws AsnException
  {
    int asnType = AsnUtil.getAsnType(sequenceType);

    // Decode the new values.
    List<BValue> newValues = new ArrayList<>();
    int tag = in.peekTag();
    while (tag != AsnInput.END_OF_DATA && !in.isClosingTag(closingTag))
    {
      BValue value;
      switch (asnType)
      {
        case ASN_NULL:
          value = in.readNull();
          break;
        case ASN_BOOLEAN:
          value = BBoolean.make(in.readBoolean());
          break;
        case ASN_UNSIGNED:
          value = in.readUnsigned();
          break;
        case ASN_INTEGER:
          value = BInteger.make(in.readSignedInteger());
          break;
        case ASN_REAL:
          value = BFloat.make(in.readReal());
          break;
        case ASN_DOUBLE:
          value = BDouble.make(in.readDouble());
          break;
        case ASN_OCTET_STRING:
          value = BBacnetOctetString.make(in.readOctetString());
          break;
        case ASN_CHARACTER_STRING:
          value = BString.make(in.readCharacterString());
          break;
        case ASN_BIT_STRING:
          value = in.readBitString();
          break;
        case ASN_ENUMERATED:
          BEnum d = (BEnum)sequenceType.getInstance();
          value = d.getRange().get(in.readEnumerated());
          break;
        case ASN_DATE:
          value = in.readDate();
          break;
        case ASN_TIME:
          value = in.readTime();
          break;
        case ASN_OBJECT_IDENTIFIER:
          value = in.readObjectIdentifier();
          break;
        default:
          value = (BValue)sequenceType.getInstance();
          ((BIBacnetDataType)value).readAsn(in);
          break;
      }

      newValues.add(value);
      tag = in.peekTag();
    }

    return newValues;
  }

  @Override
  public void readAsn(AsnInput in)
    throws AsnException
  {
    BTypeSpec typeSpec = getSequenceTypeSpec();
    if (typeSpec.isNull())
    {
      throw new AsnException("Invalid sequence type: " + typeSpec);
    }

    Type sequenceType = typeSpec.getResolvedType();
    List<BValue> newValues = readAsn(in, AsnInput.END_OF_DATA, sequenceType);
    updateElements(newValues, sequenceType, noWrite);
  }

  /**
   * Avoid remove/replace all by updating in place.  Call set on elements that already exist, remove
   * elements beyond the newValues, and add any additional newValues.
   */
  public void updateElements(List<BValue> newValues, Type sequenceType, Context context)
  {
    int newIndex = 0;
    Property[] oldProps = getDynamicPropertiesArray();
    for (Property oldProp : oldProps)
    {
      BValue oldValue = get(oldProp);
      if (!oldValue.getType().equals(sequenceType))
      {
        continue;
      }

      if (newIndex < newValues.size())
      {
        BValue newValue = newValues.get(newIndex);
        // Replace the existing value if not equivalent. Set will skip BSimples that are equal but
        // does not check for equivalence on BComplexes.
        if (!oldValue.equivalent(newValue))
        {
          set(oldProp, newValue, context);
        }
        ++newIndex;
      }
      else
      {
        // There are more old values than new values.
        remove(oldProp, context);
      }
    }

    // Add any remaining new values.
    for (int i = newIndex; i < newValues.size(); ++i)
    {
      add("element1?", newValues.get(i), context);
    }
  }

  @Override
  public String toString(Context context)
  {
    return "Sequence of " + getSequenceTypeSpec().getTypeName();
  }
}
