/*
 * Copyright 2023 Tridium, Inc. All Rights Reserved.
 */
package javax.baja.alarm.ext.offnormal;

import java.util.Map;

import javax.baja.alarm.BAlarmRecord;
import javax.baja.control.BNumericPoint;
import javax.baja.nre.annotations.Facet;
import javax.baja.nre.annotations.Generated;
import javax.baja.nre.annotations.NiagaraProperty;
import javax.baja.nre.annotations.NiagaraType;
import javax.baja.status.BStatusNumeric;
import javax.baja.status.BStatusValue;
import javax.baja.sys.BComponent;
import javax.baja.sys.BEnumRange;
import javax.baja.sys.BFacets;
import javax.baja.sys.BInteger;
import javax.baja.sys.BString;
import javax.baja.sys.Property;
import javax.baja.sys.Sys;
import javax.baja.sys.Type;

/**
 * BNumericChangeOfStateAlgorithm implements a change of state alarm detection algorithm for numeric
 * objects as described in BACnet Clause 13.3.2. Each algorithm instance defines a set of integer
 * values that should be considered "offnormal" conditions and therefore generate an alarm. This
 * algorithm is similar to {@link BEnumChangeOfStateAlgorithm} but for numeric points because those
 * points may be exported as discrete BACnet types such as Integer and Positive Integer.
 * <p>
 * The numeric value is converted to an int by rounding toward zero. Integer.MAX_VALUE is used for
 * values greater than or equal to that and Integer.MIN_VALUE is used for values less than or equal
 * to that. Therefore, setting an alarm value to those limits will cause an alarm to be raised for
 * any values beyond those limits.
 *
 * @author Uday Rapuru on 11 Apr 2023
 * @since Niagara 4.14
 */
@NiagaraType
@NiagaraProperty(
  name = "alarmValues",
  type = "BEnumRange",
  defaultValue = "BEnumRange.DEFAULT",
  facets = {
    @Facet(name = "BFacets.FIELD_EDITOR", value = "\"alarm:EnumAlarmRangeFE\""),
    @Facet(name = "BFacets.UX_FIELD_EDITOR", value = "\"alarm:EnumAlarmRangeEditor\"")
  }
)
public class BNumericChangeOfStateAlgorithm
  extends BTwoStateAlgorithm
{
//region /*+ ------------ BEGIN BAJA AUTO GENERATED CODE ------------ +*/
//@formatter:off
/*@ $javax.baja.alarm.ext.offnormal.BNumericChangeOfStateAlgorithm(2329592457)1.0$ @*/
/* Generated Wed Jul 05 12:57:14 CDT 2023 by Slot-o-Matic (c) Tridium, Inc. 2012-2023 */

  //region Property "alarmValues"

  /**
   * Slot for the {@code alarmValues} property.
   * @see #getAlarmValues
   * @see #setAlarmValues
   */
  @Generated
  public static final Property alarmValues = newProperty(0, BEnumRange.DEFAULT, BFacets.make(BFacets.make(BFacets.FIELD_EDITOR, "alarm:EnumAlarmRangeFE"), BFacets.make(BFacets.UX_FIELD_EDITOR, "alarm:EnumAlarmRangeEditor")));

  /**
   * Get the {@code alarmValues} property.
   * @see #alarmValues
   */
  @Generated
  public BEnumRange getAlarmValues() { return (BEnumRange)get(alarmValues); }

  /**
   * Set the {@code alarmValues} property.
   * @see #alarmValues
   */
  @Generated
  public void setAlarmValues(BEnumRange v) { set(alarmValues, v, null); }

  //endregion Property "alarmValues"

  //region Type

  @Override
  @Generated
  public Type getType() { return TYPE; }
  @Generated
  public static final Type TYPE = Sys.loadType(BNumericChangeOfStateAlgorithm.class);

  //endregion Type

//@formatter:on
//endregion /*+ ------------ END BAJA AUTO GENERATED CODE -------------- +*/

  /**
   * Grandparent must be BNumericPoint.
   */
  @Override
  public boolean isGrandparentLegal(BComponent grandparent)
  {
    return grandparent instanceof BNumericPoint;
  }

  /**
   * Return false if the out value, after rounding toward zero, is equal to any of the alarmValues.
   * This method will return false if an alarm value is set to Integer.MAX_VALUE and the out value
   * is greater than or equal to that value or an alarm value is set to Integer.MIN_VALUE and the
   * out value is less than or equal to that value.
   */
  @Override
  protected boolean isNormal(BStatusValue out)
  {
    int outValue = (int)((BStatusNumeric)out).getValue();
    for (int alarmOrdinal : getAlarmValues().getOrdinals())
    {
      if (outValue == alarmOrdinal)
      {
        return false;
      }
    }
    return true;
  }

  /**
   * Write the key-value pairs defining alarm data for the alarm algorithm and state to the given
   * Facets.
   *
   * @param out The relevant control point status value
   * @param map The map.
   */
  @Override
  @SuppressWarnings({"rawtypes","unchecked"})
  public void writeAlarmData(BStatusValue out, Map map)
  {
    map.put(BAlarmRecord.STATUS, BString.make(out.getStatus().toString(null)));
    map.put(BAlarmRecord.NUMERIC_VALUE, BInteger.make((int)((BStatusNumeric)out).getValue()));
  }
}
